;******************************************************************************
;*  NAME:        r_polar_ceppad_ips_mask
;*
;*  DESCRIPTION: Routine to read polar_ceppad_ips_mask files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_polar_ceppad_ips_mask, VERBOSE=VERBOSE

common polar_ceppad_ips_mask_data, input_header, input_data
common mjdt, mjdt_start, mjdt_end
common get_error, get_err_no, get_err_msg
   
get_err_no=0 & get_err_msg='polar_ceppad_ips_mask read succesfully'
   
; check for env. variable pointing to data directories
if not (papco_check_data_env('PO_CEP_MASK_DIR', PATH = path)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
while mjdt_end.mjd ge time.mjd do begin
     ; type of name: ips_mask_19970803.idl
     ; assemble the data file name from the time range requested
     convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second,$
                          month, cmonth, day_om, /mjdt   
     datestr=string(year,month,day_om,format='(i4.4,i2.2,i2.2)')
     fln='ips_mask_'+datestr
     
     ; read one day of data
     r_polar_ceppad_ips_mask_onefile, path+fln, /VERBOSE
   
     ; check for "OPENR: Error opening file" in error string - and then get 
     ; the file remotely!
     ; comment this section out if you do not want this feature.
     res=strpos(!err_string,'OPENR: Error opening file')
     if res ne -1 then begin
       c_url=papco_getenv('POLAR_CEPPAD_IPS_MASK_FTP_ARCHIVE')
       c_file=fln+'.idl'
       c_download_dir=path
       papco_get_remote_data, c_url, c_file, c_download_dir
       ;check if file "arrived" and then read it
       if get_err_no eq 0 then $
         r_polar_ceppad_ips_mask_onefile, path+fln, /VERBOSE $
       else begin
         get_err_msg='No data locally, remote get failed!C  ' $
           + get_err_msg
         return
       endelse  
     endif   
     
     ;if a file was succesfully read, concatonate!
     if get_err_no eq 0 then begin 
       if no_files_read eq 0 then begin   ;start the array buffer
         input_data.time=input_data.time+86400.0*day_count
         input_data_buf=temporary(input_data)
       endif else begin                   ;otherwise concatonate arrays
         input_data.time=input_data.time+86400*day_count
         input_data_buf=[temporary(input_data_buf),temporary(input_data)]
       endelse        
       input_data=0                     ;clear array for reuse
       no_files_read=no_files_read+1
     endif
     day_count=day_count+1
     time.mjd=time.mjd+1                  ; next day
    ; check for any time left on last day - skip if none
     if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then goto, done     
endwhile         
done:
   
; check if any files were read   
   if (no_files_read ne 0) and (get_err_no ne 1) then begin
     get_err_no=0
     get_err_msg=string(no_files_read,format="(i2)")+ $
       ' files read succesfully'
     ;clear arrays and load buffer into output array
     input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
   endif else begin
     get_err_no=1
     get_err_msg='no files found for timerange'
     return
   endelse      
   
; restrict data to actual time range requested
   ut1=mjdt_start.t
   ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
   if count ne 0 then begin
      input_data=input_data(index)
   endif   

end

;******************************************************************************
;*  NAME:        r_polar_ceppad_ips_mask_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO r_polar_ceppad_ips_mask_onefile, fln, VERBOSE=VERBOSE
  
COMMON polar_ceppad_ips_mask_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
   
get_err_no=0
get_err_msg='polar_ceppad_ips_mask onefile read succesfully'
   
;clear data arrays for reuse
input_header=0  &  input_data=0   
   
on_ioerror, make_original
restore,fln+'.idl',/verbose
dealloc_lun,/quiet    ;frees logical units used by save/restore
message, 'savest read: '+fln+'.idl', /cont
   
GOTO, have_mask

make_original:
papco_sepdir, fln+'.idl', path, file, ext
date = strmid(file, 09, 8)
cep_mask, date, mask, error

on_ioerror, done
restore,fln+'.idl',/verbose
dealloc_lun,/quiet    ;frees logical units used by save/restore
message, 'savest read: '+fln+'.idl', /cont

have_mask:

input_header={type:'POLAR CEPPAD IPS SUN/EARTH MASK'}
   
dummy={tai:0.0d, time:0.0, data:bytarr(9,32)}
ndat=n_elements(mask.ut)
input_data=replicate(dummy,ndat)
   
input_data.time=mask.ut*3600
input_data.tai = mask.tai
for i=0,ndat-1 do input_data(i).data=mask.blank(*,*,i)
   
return
   
done:
get_err_no = 3 & get_err_msg = 'cannot make mask'
message, get_err_msg, /cont

END 
