;******************************************************************************
;*  NAME:        r_polar_ceppad_pitch
;*
;*  DESCRIPTION: Routine to read intermediate CEPPAD PITCH files. Adheres to
;*               papco conventions: Error common block, changes time format
;*               to seconds since start time.
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    INSTRUMENT   a string indicating IPS, IES or HIST
;*               MODE         a string indicating the RAL IF format
;*               LOCATION     set to LANL, MPAE, IOWA, GODDARD, AEROSPACE
;*                            to use that institution's storage conventions.
;*                            e.g. at LANL, files are kept zipped, etc
;*                            default is MPAE
;*
;*  DATE:        April 1997
;*  AUTOR:       R. Friedel
;******************************************88**********************************
pro r_polar_ceppad_pitch, INSTRUMENT=INSTRUMENT, MODE=MODE, OUTPUT=OUTPUT

  common cep_pitch_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
  common mjdt, mjdt_start, mjdt_end              ;common time limits in mjdt
      
; if output is set and equal to 3, only read the data, do not
; concatonate across files. Used for "making" data...
  IF keyword_set(OUTPUT) THEN IF output EQ 3 THEN $
    readonly = OUTPUT ELSE readonly = 0 ELSE readonly = 0

; clear data arrays for reuse
  input_header=0  &  input_data=0

; loop from start to end time concatonating data files into one array.
  time=mjdt_start  &  no_files_read=0

  while mjdt_end.mjd ge time.mjd do begin

; assemble the data file from the time range requested.
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
      month, cmonth, day_om, /mjdt 
    datestr=string(year,day_oy,format='(i4.4,i3.3)')
    datestr=string(year,month,day_om,format='(i4.4,i2.2,i2.2)')      
    bu_datestr=string(year-1900,month,day_om,format='(i2.2,i2.2,i2.2)')

    r_polar_ceppad_pitch_oneday, datestr, bu_datestr, $
      INSTRUMENT=INSTRUMENT, MODE=MODE, OUTPUT=OUTPUT

    IF readonly THEN GOTO, next_day

    if get_err_no eq 0 then begin          ;a file was read succesfully
      if no_files_read eq 0 then begin    ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin		     ;otherwise concatonate arrays
        existing_number=n_elements(input_data_buf)
        added_number=n_elements(input_data)
        new_number=existing_number+added_number
        input_header.NPOINTS=new_number
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse 
      no_files_read=no_files_read+1
    ENDIF
    next_day:
    time.mjd=time.mjd+1                   ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then goto,done
  endwhile  
  done:

  if (no_files_read ne 0) then begin
    get_err_no=0  &  get_err_msg='cep_pitch read succesfully'
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
  endif else BEGIN
    IF readonly THEN BEGIN
      get_err_no=1  &  get_err_msg='no data returned, data read only'
    ENDIF ELSE BEGIN
      get_err_no=1  &  get_err_msg='no files found for timerange'
    ENDELSE
    return
  endelse 
  
end

;******************************************************************************
;*  NAME:        r_polar_ceppad_pitch_oneday
;*
;*  DESCRIPTION: Routine to read RAL intermediate CEPPAD files. Adheres to
;*               papco conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       datestr - string of correctly formatted day
;*
;*  KEYWORDS     INSTRUMENT - the name of the CEPPAD instrument to be
;*                            read - IES, IPS, HISTe, HISTp
;*               MODE         a string indicating the RAL IF format
;*               LOCATION     set to LANL, MPAE, IOWA, GODDARD, AEROSPACE
;*                            to use that institution's storage conventions.
;*                            e.g. at LANL, files are kept zipped, etc
;*                            default is MPAE
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        April 1997
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_polar_ceppad_pitch_oneday, datestr, bu_datestr, $
                                 INSTRUMENT=INSTRUMENT, MODE=MODE, OUTPUT=OUTPUT

  common cep_pitch_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
  common mjdt, mjdt_start, mjdt_end              ;common time limits in mjdt
  common polar_ceppad

  get_err_no=0 &  get_err_msg='cep_pitch read succesfully'
   
;clear arrays
  input_header=0  &  input_data=0
   
; if output is set and equal to 3, only check for data file, do not
; actually read exisiting files! Used for "making" data...
  IF keyword_set(OUTPUT) THEN IF output EQ 3 THEN $
    readonly = OUTPUT ELSE readonly = 0 ELSE readonly = 0

; check env. variable pointing to data directory
  if not (papco_check_data_env('PO_CEP_PITCH_DIR', PATH=path)) then return

  fln= datestr+'_'+combNames(MODE)+'_'+INSTRUMENT+'_pitch.idl'

  result = findfile(path+fln,  count = c)  
  IF (c EQ 1) AND (readonly NE 0) THEN BEGIN
    message, fln + ' exisits', /cont
    return
  ENDIF 

  on_ioerror, bad_restore
  message,'reading ' + path+fln, /cont
  restore,path+fln 

; the if files use time in TAI, International Atomic time. Since the original
; files are already idl save sets, making a new file with mjdt time is
; just wasting space without getting any gain in read speed. Also, converions
; routines from TAI exisit, which use as their internal standard the same as
; mjdt time! So here we leave the files as are!
  close,/file
  return
  
  bad_restore:
  
  ;try to make the missing pitch file
  sec = fix(strmid(combNames(mode), 0, 2))
  spn = fix(strmid(combNames(mode), 3, 2)) 

  m_polar_ceppad_pitch_loop, datestr, INSTRUMENT=INSTRUMENT, SPIN=SPN, $
                                      SECTOR=SEC
  IF (readonly NE 0) THEN return

  ;reread data from saveset
  on_ioerror, bad_make_original
  message,'reading ' + path+fln, /cont
  restore,path+fln   
  return 
 
  bad_make_original:  
  get_err_no=2
  get_err_msg='Error reading: '+ path+fln
  message, get_err_msg, /cont

end
