;******************************************************************************
;* FUNCTION:     
;*      FUNCTION polar_eph_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typi%cally this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from plotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*	newPlotData=polar_eph_panel(plotinfo, ACTION=, GROUP=)
;*
;* MODIFICATION HISTORY:       
;*      written july 2002, Reiner Friedel
;******************************************************************************
FUNCTION polar_eph_panel, PlotInfo, GROUP=group, ACTION=ACTION
   
COMMON polar_eph_panel_data, widgetData
COMMON polar_ephemeris

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(polar_eph_control)
control = *plotinfo.USR_PTR1
;control now contains the user defiend control structure for this module.

;--- create the base-window ---------------------------------------------------
title = PlotInfo.panelKind + ' - ' + papco_module_version('polar_eph')
panel_base=papco_widget_base(TITLE = title, COLUMN = 1, GROUP_LEADER = group)
panel_middle = widget_base(panel_base, ROW = 1)
panel_left   = widget_base(panel_middle, COLUMN=1)
panel_right  = widget_base(panel_middle,COLUMN=1)
panel_bottom = widget_base(panel_base,ROW=1)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
widgets=papco_panel_AddHeightWidgets(panel_left, plotinfo)
additionalWData=create_struct(widgets, additionalWData)
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

;-- select orginal or model coords - ------------------------------------------
names = ['Original Ephemeris', 'Mag Model Coords']
orig_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
orig_base = CW_BGROUP(orig_outer, names , ROW=1, /EXCLUSIVE, $
                   LABEL_LEFT='Coord Source:', $
                   UVALUE='spb_orig', $
                   SET_VALUE=control.orig)

;--- select orig coordinates -------------------------------------------------
eph_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
eph_names = orig_names(*, 0)
eph_base=CW_BGROUP(eph_outer, eph_names, ROW=1, /EXCLUSIVE, $
                   LABEL_TOP='Original Ephemeris Coordinates', $
                   UVALUE='spb_eph', $
                   SET_VALUE=control.product)

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   

;--- add Mag Field model choices ----------------------------------------------
widgets=papco_panel_Add_MagMod_Widgets(panel_left, control)
additionalWData=create_struct(widgets, additionalWData)

;--- add the widgets for choosing the plot-style ------------------------------
widgets=papco_panel_add_PlotStyleButtons(panel_right, plotinfo, 0, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
;--- add the widgets for choosing the y-axis-ranges --------------------------
widgets=papco_PANEL_AddYScale_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
;--- add the Actionbuttons (help, cancel, done) -------------------------------
widgets=papco_panel_AddActionButtons(panel_bottom, Action, plotinfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'polar_eph_panel.help', $
                 eph_outer:eph_outer}

; add control struct, papco_panel structs
widgetData=papco_panel_CreateWidgetData(additionalWData, thisWidgetsData)
widgetData=papco_panel_CreateWidgetData(widgetData, control)

;--- realize the window
papco_Panel_Realize, panel_base, widgetData
polar_eph_refresh, /ALL
papco_xmanager, 'polar_eph_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified }

additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_eph_refresh
;* 
;* DESCRIPTION:  
;*      refresh rotuine for polar_eph_panel, called by event handler
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO polar_eph_refresh, ALL = ALL, COORDS = COORDS

COMMON polar_eph_panel_data, widgetData

IF keyword_set(ALL) THEN BEGIN
    COORDS = 1
ENDIF

IF keyword_set(COORDS) THEN BEGIN
    IF widgetData.orig EQ 0 THEN BEGIN 
        widget_control, widgetData.eph_outer,  SENSITIVE = 1
        widget_control, widgetData.mag_model_outer_base,  SENSITIVE = 0
    ENDIF ELSE BEGIN
        widget_control, widgetData.eph_outer,  SENSITIVE = 0
        widget_control, widgetData.mag_model_outer_base,  SENSITIVE = 1
    ENDELSE 
END

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_panel_template_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;******************************************************************************
PRO polar_eph_panel_event, event
   
COMMON polar_eph_panel_data, widgetData
   
WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF

    'spb_orig': BEGIN
        widgetData.orig = event.value
        polar_eph_refresh, /COORDS
    END 
    'spb_eph' : widgetData.product = event.value
    ELSE: BEGIN
        widgetData=papco_panel_Event(event, widgetData)
    END

ENDCASE

END















