;***************************************************************************
;* PROCEDURE:     
;*      PRO draw_ephemeris_polar_eph, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for polar_eph
;*      calling a seperate window for the plot.
;*      slice_type in draw_polar_eph must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      READDATA    indicates the data need to be reread, new panel
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      december 1995, written by R.Friedel 
;*      april 1996, R. Friedel
;***************************************************************************
PRO draw_ephemeris_polar_eph, panelNr, seconds, yValue, READDATA=READDATA

  COMMON DRAW_POLAR_EPHEMRIS_SLICE, sliceWidgetData
  COMMON PLOT_COMPOSER, widgetData

  common polar_neweph, input_header, input_data 
  common redraw, pnr
  COMMON mjdt, mjdt_start, mjdt_end

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; t90 is the time array in T90 format.
; papco uses the the getdatacall you set in papco_add_polar_eph to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 
  result=size(pnr) & if result(1) eq 0 then pnr=-1
  WIDGET_CONTROL,/HOURGLASS
  thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
  thisPlot=widgetData.plotsDrawn(panelNr)
  type=thisPlot.typeVector

; set the panel window for the slice to full window
  !p.position=[0.1,0.1,0.9,0.9]

; select the index for the time chosen out of the time array
  difft=abs(input_data.time-seconds)
  result=min(difft,minindex)

; these two info strings are user-definable output strings to label the slice 
; window.
  time_str=papco_cursor_time(seconds,panelNr)
  info1=time_str

  gsm=input_data(minindex).GSM_XYZ / 6371.0
  radius=sqrt(gsm(0)^2+gsm(1)^2+gsm(2)^2)

  info2=string(input_data(minindex).EDMLT_TIME, $
               input_data(minindex).MAG_LATITUDE, $
               input_data(minindex).L_shell, radius, $
    format="('MLT: ',f4.1,'   Mlat: ',f5.1,'   L: ',f6.1,'   Rad: ',f7.1)")

; Draw the coordinate system and orbit. Only once, first time it's called!
   if sliceWidgetData.lastPanelDrawn ne pnr then $
     p_eph_3d,30,30,1,[0,0,0],input_data.GSM_XYZ / 6371.0
   
; Draw a large dot of position of spacecraft
   trace=input_data(minindex).trace_3_2
   p_pos_3d,30,30,1,[0,0,0],gsm, trace=trace

; user defined routine for plotting the slice included here. An example is in 
; $papco_MODULES/panels&slices/crres_slices.pro
; You typically need to include the commons from the read routine, a way
; of finding the index of the data you want to plot using the time "seconds"
; (in t90 format of course), plus a simple plot statement to plot the data.
; Two strings, info1 and info2, can be used to annotate the slice window
; by providing additional information, ie polar_eph data, etc.

; these calls put your info strings in the provided labels of the slice widget

  WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
  WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2

; set flag to indicate the panel Nr. for which slice is drawn
  pnr=sliceWidgetData.lastPanelDrawn

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO user_view_ephemeris_polar_eph_event, Event
;* 
;* DESCRIPTION:  
;*	The slice-window displays a slice through the data, 
;*	when the user clicks in the drawing-widget.
;*	This procedure handles events for that window.
;* INPUTS:       
;*	Event	 	an XMANAGER-event
;*	
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written august and september 1995, Andreas Keese
;******************************************************************************
PRO user_view_ephemeris_polar_eph_event, Event

  COMMON DRAW_POLAR_EPHEMRIS_SLICE, sliceWidgetData

  WIDGET_CONTROL,Event.Id,GET_UVALUE=Ev

  CASE Ev OF
     'pb_done' : BEGIN
        WIDGET_CONTROL, sliceWidgetData.base, /DESTROY
     END
     
     'pb_help': $
       xdisplayfile, papco_helpfile('papco_polar_eph_slice.help'), $
       title='papco Online-Help', group=event.top

     ELSE : BEGIN
     END
  ENDCASE

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO user_view_ephemeris_polar_eph, panelNr, seconds,yValue,CLOSE=CLOSE
;* 
;* DESCRIPTION:  
;*	The slice-window displays polar_eph info, when the user clicks
;*	in the drawing-widget.
;*	This procedure opens that window and calls other routines, that draw
;*	the ephemris info.
;*
;* INPUTS:       
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		a float
;*	yvalue		a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*
;* KEYWORDS:
;* 	CLOSE		when specified, the window is destroyed
;* 
;* CALLING SEQUENCE:
;*	papco_DrawSlice, aPanel, convert_tzero_to_t90(3600, 1, 1990), 42.42
;*	papco_DrawSlice, /CLOSE
;*
;* MODIFICATION HISTORY:       
;*     written october 1996, Reiner Friedel
;******************************************************************************
PRO user_view_ephemeris_polar_eph, panelNr, seconds, yValue, $
                                   CLOSE=CLOSE, READDATA=READDATA

  COMMON DRAW_POLAR_EPHEMRIS_SLICE, sliceWidgetData
  COMMON PLOT_COMPOSER, widgetData
  COMMON ADDINS, plot_types
  COMMON ISREAD, IsReadArr

;-- close the window ? --------------------------------------------------------
  IF KEYWORD_SET(CLOSE) THEN BEGIN
    IF XREGISTERED('USER_VIEW_EPHEMERIS_POLAR_EPH') THEN $
      IF N_ELEMENTS(sliceWidgetData) GT 0 THEN BEGIN
	 IF sliceWidgetData.base NE 0 THEN $
  	    WIDGET_CONTROL, sliceWidgetData.base, /DESTROY
         sliceWidgetData.base=0
      ENDIF
    RETURN
  ENDIF

;-- if the window exists, bring it in front. Else, create it ------------------
  IF NOT XREGISTERED('USER_VIEW_EPHEMERIS_POLAR_EPH') THEN BEGIN
      Slice_Base=WIDGET_BASE(COLUMN=1, $
		GROUP=widgetData.main_base, $
		title='VIEW POLAR EPHEMERIS')
      dw_Slice  =WIDGET_DRAW(Slice_Base, XSIZE=600, YSIZE=600)
      lbl_slice1=WIDGET_LABEL(Slice_Base, VALUE=strcenter('',80))
      lbl_slice2=WIDGET_LABEL(Slice_Base, VALUE=strcenter('',80))
  
      Butt_Base=WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
      pb_help=WIDGET_BUTTON(Butt_Base, $
                              Value=strcenter('Help', 12), $
			      UVALUE='pb_help')
      pb_done=WIDGET_BUTTON(Butt_Base, $
                              Value=strcenter('Close', 12), $
			      UVALUE='pb_done')
      sliceWidgetData={USER_VIEW_polar_eph_polar_eph, $
		base:slice_base, $
	        dw_slice:dw_slice, $
		lbl_slice1:lbl_slice1, $
		lbl_slice2:lbl_slice2, $
	        panelNr:panelNr, $
		lastPanelDrawn:-1}
      WIDGET_CONTROL, Slice_Base, /REALIZE
      XMANAGER, 'USER_VIEW_EPHEMERIS_POLAR_EPH', Slice_Base
  ENDIF

  thisPlot=widgetData.plotsDrawn(panelNr)

;--- now, we need to set the get data flag-------------------------------------
;    but we will set the flag, if the panel the user clicked is not the
;    panel that he last clicked into (in this case, the data is already read).

  IF sliceWidgetData.lastPanelDrawn NE panelNr THEN READDATA=1 $
                                                ELSE READDATA=0
  sliceWidgetData.lastPanelDrawn = panelNr
 
;--- prepare the drawing widget for plotting ----------------------------------
  SET_PLOT, 'X'  
  WIDGET_CONTROL, sliceWidgetData.dw_slice,  GET_VALUE=drawIndex
  WSET, drawIndex

;--- store the current coordinate system --------------------------------------
  old_x = !x  &  old_y = !y  & old_p = !p

; ---  Call add-in routine for slice  -----------------------------------------
  draw_ephemeris_polar_eph, panelNr, seconds, yValue, READDATA=READDATA
  
  jumpslice:   

; --- restore the coordinate-system -------------------------------------------
   !x = old_x  &  !y = old_y  &  !p = old_p
   
END
