;***************************************************************************
;* file polar_eph_slice.pro
;*
;* this file contains the routines needed to enable the extra mouse features
;* provided for in papco. Mouse zooms is part of main papco. Currently possible:
;*
;*    mouse views slice
;*    mouse views slice (polar)
;*    mouse accesses level zero
;* 
;* Here the procedure names must again adhere to a strict naming convention so 
;* that they can be automatically called when needed.
;*
;* The slice types supported by these routines must be ENABLED by the user, so 
;* that papco can call them. This is done by setting the value of 
;* slice_type in draw_polar_eph.pro (contained in papcoadd_polar_eph.pro).
;* slice_type is defined in the following way:
;*
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 swittches. Currently
;*                      Three are supported (this may be extended later)
;*                       1 = mouse zooms (default, always possible)
;*                       2 = mouse slices
;*                       4 = mouse calls level zero
;*                       8, 16, 32, 64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* The "mouse views slice" function uses the time returned by papco and the
;* get_data_call specified for the polar_eph to read the data and to draw a 
;* slice at the clicked position. This is commonly used in spectra color plots 
;* to draw a spectrum at that time, or a pitch angle distribution etc, or to
;* call up any other plot at that time as specified by the user.
;*
;* The "mouse accesses levelzero" uses the time passed and calls up another
;* widget application to read and work with level zero data. This application
;* must be intirely independant of papco and is only called from papco via
;* a mouse click in the panel required for the time required.
;*
;* If your new data contains several sub-data types, one slice and/or levelzero
;* routine has to be provided for each type, since in general they would do
;* different things. See $papco_MODULES\panel&slices\crres_slices.pro for 
;* such an example.
;*
;*      PRO DrawSlice_polar_eph, panelNr, seconds, yValue
;*        -> Draw a slice for polar_eph
;*
;*      PRO levelzero_polar_eph, seconds, yvalue
;*        -> This procedure calls the widget for levelzero data access 
;*           for this plottype.
;*    
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;*      april 1996, R. Friedel
;***************************************************************************

;***************************************************************************
;* PROCEDURE:     
;*      PRO DrawSlice_polar_eph, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for polar_eph
;*      calling a seperate window for the plot.
;*      slice_type in draw_polar_eph must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      december 1995, written by R.Friedel at Goddard
;*      april 1996, R. Friedel
;***************************************************************************
PRO DrawSlice_polar_eph, panelNr, seconds, yValue

   COMMON papco_DRAWSLICE, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData

   common polar_ephem, polar_ephem
   common redraw, pnr

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; t90 is the time array in T90 format.
; papco uses the the getdatacall you set in papco_add_polar_eph to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 

   result=size(pnr) & if result(1) eq 0 then pnr=-1

   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector

; set the panel window for the slice to full window

   !p.position=[0.1,0.1,0.9,0.9]

; select the index for the time chosen out of the time array

   difft=abs(polar_ephem.t90-seconds)
   result=min(difft,minindex)

; these two info strings are user-definable output strings to label the slice 
; window.

   convert_t90_to_Date,seconds, year, day_oy, hour, min, sec, $
			      month, cmonth, day_om 
   info1=string(day_om,cmonth,year,format="('Date: ',i2.2,' ',a3,' ',i2.2,' ')")
   info1=info1+string(hour,min,sec,format="('  Time: ',i2.2,':',i2.2,':',i2.2)")

   gsm=polar_ephem(minindex).GSM_POS
   radius=sqrt(gsm(0)^2+gsm(1)^2+gsm(2)^2)

   info2=string(polar_ephem(minindex).EDMLT_TIME, $
                polar_ephem(minindex).MAG_LATITUDE, $
                polar_ephem(minindex).L_shell, $
                radius, $
 format="('MLT: ',f4.1,'    ilat: ',f5.1,  '    L-value: ',f6.1,'    radius: ',f7.1)")

; Replace these labels and include your plot! 

   gsm=polar_ephem.GSM_POS/6372.0

; Draw the coordinate system and orbit. Only once, first time it's called!
   if sliceWidgetData.lastPanelDrawn ne pnr then begin
     p_eph_3d,30,30,1,[0,0,0],gsm
   endif

; Draw a large dot of position of spacecraft
   current_pos=gsm(*,minindex)
   p_pos_3d,30,30,1,[0,0,0],current_pos

; user defined routine for plotting the slice included here. An example is in 
; $papco_MODULES/panels&slices/crres_slices.pro
; You typically need to include the commons from the read routine, a way
; of finding the index of the data you want to plot using the time "seconds"
; (in t90 format of course), plus a simple plot statement to plot the data.
; Two strings, info1 and info2, can be used to annotate the slice window
; by providing additional information, ie polar_eph data, etc.

; these calls put your info strings in the provided labels of the slice widget

   WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
   WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2

; set flag to indicate the panel Nr. for which slice is drawn
  pnr=sliceWidgetData.lastPanelDrawn

END

;***************************************************************************
;* PROCEDURE:     
;*      PRO levelzero_polar_eph, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*      This procedure calls the widget for level zero data acces 
;*      for this plottype. 
;*      slice_type in draw_polar_eph must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*      seconds           T90 time at which mouse is clicked
;*      yvalue            y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      levelzero_polar_eph, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel at Goddard
;*     april 1996, R. Friedel
;***************************************************************************
pro levelzero_polar_eph, seconds, yvalue

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $papco_MODULES/addpapco_swe_plots.pro.
; Provided as paramters to be passed to the new application, are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be swittched to (e.g. to get a new
; time from the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('papco_main') 
; to swittch back to the papco main calling widget. 

; call the levelzero widget.  
;   polar_eph_levelzero, seconds, yvalue

; the following line is printed on your xterm as a check to see if papco
; correctly called this routine. If there is a problem make sure your naming 
; convention is O.K. : This routine MUST be called levelzero_polar_eph, 
; where polar_eph is the name of the panel you have set in papcoadd_polar_eph.

    print, 'got to the levelzero interface  ',seconds, yvalue

END
