;******************************************************************************
;* PROCEDURE:     
;*      p_polar_eph, panel, plotinfo, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*          SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type DATA_TYPE for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie polar_eph) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale     
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_polar_eph, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;******************************************************************************
pro p_polar_eph, panel, plotinfo, OUTPUT=OUTPUT, $
                 OVERPLOT=OVERPLOT, $
                 PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end   	;common time limits in mjdt
COMMON shift_label, down                ;common for x-axis label shifting
COMMON yscale, yscl			;man/auto yscaling
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

COMMON polar_eph, input_header, input_data ;POLAR polar_eph data
COMMON polar_ephemeris
COMMON polar_eph_tick1, time, yray, tick_plotinfo, tick_label
COMMON papco_color_names

IF keyword_set(OUTPUT) THEN output = OUTPUT ELSE output = 0

;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1

control = *plotinfo.USR_PTR1
utitle = 'polar eph'

;select data to plot
Re=6371.0
;set bad data flag
nodata = -99 & tick_label = ''
idx = where(data.rad eq 0, cc) 
IF cc EQ n_elements(data) THEN GOTO, noplot

CASE control.orig OF 
    0: BEGIN                    ;original epehemeris data
        CASE control.product OF 
            0:BEGIN 
                yray=data.L_SHELL
                yst=1 &  yen=100
            END 
            1:BEGIN 
                yray=data.EDMLT_TIME
                yst=-1 & yen=25
            END 
            2:BEGIN 
                yray=data.MAG_LATITUDE
                yst=-90 & yen=90
            END 
            3:BEGIN 
                yray=data.RAD/Re
                yst=1 & yen=10
            END 
            4:BEGIN 
                yray=data.GSM_XYZ(0,*)/Re
                yst=-10 & yen=10
            END  
            5:BEGIN 
                yray=data.GSM_XYZ(1,*)/Re
                yst=-10 & yen=10
            END  
            6:BEGIN 
                yray=data.GSM_XYZ(2,*)/Re
                yst=-10 & yen=10
            END 
            7:BEGIN 
                yray=data.GSE_XYZ(0,*)/Re
                yst=-10 & yen=10
            END  
            8:BEGIN 
                yray=data.GSE_XYZ(1,*)/Re
                yst=-10 & yen=10
            END  
            9:BEGIN 
                yray=data.GSE_XYZ(2,*)/Re
                yst=-10 & yen=10
            END 
        ENDCASE 
        utitle = utitle+'!C'+orig_names(control.product, 0)
        uytitle = orig_names(control.product, 1)
        tick_label = orig_names(control.product, 0) +'('+uytitle+')'
    END         
    1: BEGIN                    ;Model ephemeris / mag field data
        nodata = -1e31
        model_num = varprt(control.internal_model)+'_'+$
                             varprt(control.external_model)
        IF control.MAG_DYN THEN dyn_str = '_DYNA' ELSE dyn_str = '_STAT'
        mag_model = 'MODEL_'+model_num+dyn_str
        tags = TAG_NAMES(input_data(0))
        mag_idx = where(tags EQ mag_model, c)
        IF c EQ 0 THEN BEGIN 
            message, 'Magnetic model data for '+mag_model+' not found', /cont
            GOTO, noplot
        ENDIF 

        mag_coord_names = papco_make_choice_names(control.MAG_COORDS_INFO)
        mag_coord_units = papco_make_choice_names(control.MAG_COORDS_UNITS)

        CASE 1 OF
            (control.mag_coords EQ 0): BEGIN ;L coords are indep.of PA
                yray = data.(mag_idx(0)).L
                yray = transpose(yray)
            END  
            (control.mag_coords EQ 1): BEGIN ;L* coords are per 9 PA's
                pitch_names = papco_make_choice_names(control.pitch_proc_info)
                pa_idx = control.pitch_plot
                mag_model = mag_model+'_'+varprt(pitch_names(pa_idx))
                yray = data.(mag_idx(0)).LSTAR(pa_idx)
                yray = transpose(yray)
            END  
            (control.mag_coords GE 2) AND (control.mag_coords LE 6): BEGIN
                yray = data.(mag_idx(0)).(control.mag_coords)
            END 
            (control.mag_coords EQ 7): BEGIN ;Btot, derived
                yray = sqrt(data.(mag_idx(0)).b1^2 + $
                            data.(mag_idx(0)).b2^2 + $
                            data.(mag_idx(0)).b3^2)
            END 
            (control.mag_coords EQ 8): BEGIN ;Btot/Bmin
                btot = sqrt(data.(mag_idx(0)).b1^2 + $
                            data.(mag_idx(0)).b2^2 + $
                            data.(mag_idx(0)).b3^2)
                bmin =  data.(mag_idx(0)).bmin
                idx = where(bmin NE 0, c)
                yray = fltarr(n_elements(data)) & yray(*) = nodata
                IF c NE 0 THEN yray(idx) = btot(idx)/bmin(idx)
            END 
        ENDCASE 
        uytitle = mag_coord_units(control.mag_coords)
        utitle=utitle+'!C' + $
               mag_model+' '+mag_coord_names(control.mag_coords)
        idx = where(yray EQ nodata, cc)
        IF cc NE 0 THEN yray(idx) = nodata
        tick_label = mag_coord_names(control.mag_coords) + $
                     '('+uytitle+':'+model_num+')'
    END 
ENDCASE

IF output EQ 2 THEN return

panelset,panel	         		;sets the panel position viewport

if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else begin
    papco_ylim,time,yray,yst,yen, LOG=yscl(panel(0),3), BAD_DATA=nodata
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
endelse

;use automatic "nice" y-lables
papco_y_label, yst, yen, log=yscl(panel(0),3)

; set up extra plot keywords, common ones first
extra_plotPar={yrange:[yst, yen], xrange:[xut1, xut2], $
               ylog:yscl(panel(0),3), xtickformat:'noticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this routine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar   

; now do the actual data plot, data only, no axis
papco_gap_plot, time, yray, nodata, $
                _extra = extra_plotPar, xstyle=5, ystyle=5
   
; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel,uytitle,/rot90  

; store the coordinate information into common block
plotted_x = !x  &  plotted_y = !y

; plot info right of the panel. Multiple panel labels are handled.
papco_rs_lbl, panel, utitle, /LINE, _extra=extra_Par
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='polar_ephemeris data'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
ENDIF

return

;jump to here if no data to plot 
noplot: 
message,'No data to plot', /cont
plotted_x = !x  &  plotted_y = !y
 
END

;******************************************************************************
pro w_polar_eph_event, time, yray
    
  COMMON mjdt, mjdt_start, mjdt_end   	;common time limits in mjdt
   
;setup path for event data
  data_dir='PAPCO_EVENT_DATA'
  path=papco_getenv(data_dir)
  if strlen(path) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif  
  
; check for existence of data directory
  cc=papco_finddir(path)
  if cc eq 0 then begin
    get_err_no=2
    get_err_msg='Data directory '+path+' not found'
    print,get_err_msg
    return
  endif
  
  fln='polar_eph.event'
  print,'% w_polar_eph_event: writing to ',path+fln
  openw,unit,path+fln,/get_lun
  
  ev_val=6.6                    ; L-values for the event times!
  diff=abs(yray-ev_val)
  
  for i=1,n_elements(diff)-2 do begin
    if (diff(i-1) gt diff(i)) and (diff(i) lt diff(i+1)) then begin
      tmjdt=mjdt_start
      tmjdt.mjd=tmjdt.mjd+long(time(i))/86400
      tmjdt.t= time(i) mod 86400
      time_str=convert_secondsToTime(tmjdt, /MJDT, /T90)
      print,'Time for value =',ev_val,'  -> ',time_str
      printf,unit,time_str
    endif  
  endfor
  
  close,unit & free_lun,unit
  
end  
