;******************************************************************************
;* PROCEDURE:     
;*      p_polar_eph_ticks, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*          SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type DATA_TYPE for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie polar_eph_ticks) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale     
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_polar_eph_ticks, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;******************************************************************************
pro p_polar_eph_ticks, panel, plotinfo, OUTPUT=OUTPUT, $
                 OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end   	;common time limits in mjdt
COMMON shift_label, down                ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

COMMON polar_eph, input_header, input_data
COMMON polar_ephemeris ;see polar_eph_commons.pro for def. of this common block
COMMON polar_eph_tick1, time, yray, tick_plotinfo, tick_label

panelset,panel	         		;sets the panel position viewport

;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (IDL plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE return
xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1

control = *plotinfo.USR_PTR1
utitle = 'polar eph'

coord = control.product
coord_names = orig_names(*, 0)
coord_units = orig_names(*, 1)

extra_plotPar={xrange:[xut1, xut2], yrange:[0, 1], ystyle:5, xstyle:9, $
               xtickformat:'noticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)   

; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's xtickformat if it is something other than 'noticks'.
; "down" sets the axis labeling level. 
down=0 & p=[0,panel(1),panel(2)]
p = panel
papco_draw_time_axis, p, OVERPLOT=0, _extra=extra_plotPar  
lbl='' & FOR i=0, down-1 DO lbl=lbl+'!C'

plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels - down sets the axis labeling level
extra_plotPar.xtickformat='polar_ticks' 
tick_plotinfo = plotinfo
tick_plotinfo.USR_PTR1 = ptr_new(polar_eph_control)
*tick_plotinfo.USR_PTR1 = *plotinfo.USR_PTR1

;do for original coords
FOR i=0ll,n_elements(coord_names)-1 DO BEGIN
    IF (coord and 2l^i) eq 2l^i THEN BEGIN 
        (*tick_plotinfo.usr_ptr1).product = i
        papco_draw_time_axis, p, OVERPLOT=0, /FORCE, _extra=extra_plotPar
        lbl = lbl + tick_label + '!C'
        down=down+1
    ENDIF  
ENDFOR

;do for mag model coords
FOR i = 0, 2 DO BEGIN ;mag coords
    IF (control.mag_coords and 2^i) eq 2^i THEN BEGIN 
        (*tick_plotinfo.usr_ptr1).mag_coords = i
        (*tick_plotinfo.usr_ptr1).orig = 1
        (*tick_plotinfo.usr_ptr1).bfield_coords = 0
        papco_draw_time_axis, p, OVERPLOT=0, /FORCE, _extra=extra_plotPar
        lbl = lbl + tick_label + '!C'
        down=down+1
    ENDIF  
ENDFOR


;x_side_label puts x-axis label to the right of the axis.
x_side_label,panel, lbl

END

;-----------------------------------------------------------------------------
; ONE function for ALL polar eph ticks.
function polar_ticks, axis, index, t

COMMON mjdt, mjdt_start, mjdt_end
COMMON polar_eph_tick1, time, yray, tick_plotinfo, tick_label
COMMON polar_ephemeris 
COMMON shift_label, down             
  
down_str=''  &  for i=0,down-1 do down_str=down_str+'!C'
;get data by calling p_polar_eph_line to get data only.
IF index EQ 0 THEN p_polar_eph, panel, tick_plotinfo, OUTPUT = 2
IF tick_label EQ '' THEN return, down_str+'N/A'

;time starts at zero, papco_draw_time_axis at startsec of day. Need to add in. 
tt = time + double(mjdt_start.t)

difft=abs(tt-t) & result=min(difft,minindex) & tick=yray(minindex)

coord = tick_plotinfo.typevector(1)
coord_format = orig_names(*, 2)

exe_str = 'out_str=string(tick,format="('+coord_format(coord)+')")'
r = execute(exe_str)
  
return, down_str+out_str

END
