;*****************************************************************************
;* FILE:
;*	r_polar_eph.pro
;*
;* DESCRIPTION:
;*	Procedure to read new POLAR ephemeris info (updated mag coords)
;*
;* INPUTS:
;*      none     files are searched accoring to time in common mjdt
;*
;* OUTPUTS:
;*      none     data is in common polar_eph
;*
;* KEYWORDS:
;* 	VERBOSE  integer     be verbose or not
;*      ORBIT    integer     retrun data by orbit number
;*
;* MODIFICATION HISTORY:
;*	written  1999, R. Friedel
;*      remote wget added J. Niehof 15 Mar 2004
;* TODO:
;*      remote only gets v01--make it get latest version
;*****************************************************************************
PRO r_polar_eph, plotinfo, VERBOSE=VERBOSE, ORBIT=ORBIT

COMMON mjdt, mjdt_start, mjdt_end               ;common time limits in mjdt
COMMON polar_eph, input_header, input_data 
COMMON get_error, get_err_no, get_err_msg
  
; loop through from start to end time concatonating data files into one array.
if keyword_set(ORBIT) then begin
    result=papco_setorbit(orbit, SATELLITE='POLAR', /NOWIDGET)
endif    
  
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     

; clear data arrays for reuse
input_header=0  &  input_data=0   

; loop through from start to end time concatonating data files into one array.
time=mjdt_start & no_files_read=0

while mjdt_end.mjd ge time.mjd do BEGIN

    r_polar_eph_oneday, plotinfo, time, VERBOSE=VERBOSE

    if get_err_no eq 0 then begin          ;a file was read succesfully
        if no_files_read eq 0 then begin     ;start the array buffer
            input_data_buf=input_data
        endif else begin		     ;otherwise concatonate arrays
            existing_number=n_elements(input_data_buf)
            added_number=n_elements(input_data)
            new_number=existing_number+added_number
            input_data_buf=[input_data_buf,input_data]
        endelse 
        no_files_read=no_files_read+1
    endif
    time.mjd=time.mjd+1                   ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then BREAK

ENDWHILE

; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 2) then begin
    get_err_no=0
    get_err_msg='polar new eph read succesfully'
    input_data=temporary(input_data_buf)
endif else begin
    get_err_no=1
    get_err_msg='no new POLAR polar eph files in this timerange'
    message,get_err_msg,/info
endelse

close,/file

END 

;******************************************************************************
;*  NAME:        r_polar_eph_onefile, plotinfo, time
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       plotinfo, time 
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        July 2003
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO r_polar_eph_oneday, plotinfo, time, VERBOSE = VERBOSE

COMMON get_error, get_err_no, get_err_msg
COMMON polar_eph, input_header, input_data 
  
in_time = time

get_err_no=0 & get_err_msg='new polar eph file read successfully'
  
; check keywords and set
if keyword_set(VERBOSE) then VERBOSE=1 else VERBOSE=0

; check env. variable pointing to data directory
if not (papco_check_data_env('POLAR_EPH_DATA', path=PATH)) then return
  
;assemble the data file from the date requested
utc = {MJD:long(time.mjd), TIME:long(time.t*1000)}
datestr = ANYTIM2CAL(utc, FORM = 8, /DATE)

file = 'PO_EPH_'+datestr+'.idl' & fln=path+'PO_EPH_'+datestr+'.idl'
eph_struct_flname = path+'eph_struct.idl'

control = *plotinfo.USR_PTR1

; clear data arrays for reuse
input_header=0  &  input_data=0

;read existing file, or make it
filethere=findfile(fln, count = c)

IF c EQ 0 THEN BEGIN
    get_err_no=2 & get_err_msg='File '+fln+' not found'
    message, get_err_msg, /info
    ;then read original data, extract ephemeris
    r_polar_eph_onefile_original, plotinfo, datestr, flname

    IF get_err_no EQ 0 THEN BEGIN     
        ;if no eph_struct exists, write the original one
        result = findfile(eph_struct_flname, count = c)
        IF c EQ 0 THEN BEGIN
            eph_struct = input_data(0)
            save, file = eph_struct_flname, eph_struct
        ENDIF
    ENDIF ELSE BEGIN
        return
    ENDELSE     
ENDIF ELSE BEGIN
    restore, fln,  VERBOSE = VERBOSE
    message, 'idl saveset read succesfully', /info
    print, '  ', fln
ENDELSE

;now make/add/check mag coords in input_data.
;need to "hand over" time in TAI,coord system info, actual eph coords. 

;GPS eph in in geographic lat/long
cor_type='CAR' & cor_sys='GSM'
tai = input_data.tai
pos = input_data.GSM_XYZ

papco_add_mag, control, cor_type, cor_sys, $
               eph_struct_flname, new_struct_name, tai, pos, input_data, error

IF error EQ 2 THEN BEGIN
    time = in_time & return ;no new structures added to input_data
ENDIF

;save the new eph data to file
message, 'Writing new', /info
print, '  ', fln
save, file = fln, input_header, input_data, /COMPRESS

;save the new gps eph structure to file
message, 'Writing new structure contents to file', /info
eph_struct = input_data(0)
save, file = eph_struct_flname, eph_struct

time = in_time

END 


;******************************************************************************
;*  NAME:        r_polar_eph_onefile_original, plotinfo, datestr, flname
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first.
;*               
;*  INPUT:       plotinfo, datestr, flname
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        July 2003
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO r_polar_eph_onefile_original, plotinfo, datestr, flname, VERBOSE=VERBOSE

;read polar data eph, extract ephemeris info nneded

COMMON get_error, get_err_no, get_err_msg
COMMON polar_eph, input_header, input_data 
COMMON polar_eph_cdf, eph_cdf_header, eph_cdf_data

r_polar_eph_cdf_oneday, datestr, VERBOSE=VERBOSE

IF get_err_no NE 0 THEN BEGIN
    message,'Error reading polar cdf ephemeris',/info
    return 
ENDIF ELSE BEGIN; fold into input_data
    dmy={TAI:0.0d, RAD:0.0d, GSM_XYZ:dblarr(3), GSE_XYZ:dblarr(3), $
         EDMLT_TIME:0.0, MAG_LATITUDE:0.0, L_SHELL:0.0}
    input_data=replicate(dmy, n_elements(EPH_CDF_DATA))
    FOR i=0, n_elements(input_data) -1 DO  BEGIN
        CDF_EPOCH, EPH_CDF_DATA(i).epoch(0), $
                   yr, mo, dy, hr, mn, sc, milli, /BREAK
        utc_ext = {YEAR:yr, MONTH:mo, DAY:dy, $
                   HOUR:hr, MINUTE:mn, SECOND:sc, MILLISECOND:milli}
        input_data(i).TAI = UTC2TAI(utc_ext)
        rad = sqrt(EPH_CDF_DATA(i).GSM_POS(0)^2+$
                   EPH_CDF_DATA(i).GSM_POS(1)^2+$
                   EPH_CDF_DATA(i).GSM_POS(2)^2)
        input_data(i).RAD          = rad
        input_data(i).GSM_XYZ      = EPH_CDF_DATA(i).GSM_POS
        input_data(i).GSE_XYZ      = EPH_CDF_DATA(i).GSE_POS
        input_data(i).EDMLT_TIME   = EPH_CDF_DATA(i).EDMLT_TIME(0)
        input_data(i).MAG_LATITUDE = EPH_CDF_DATA(i).MAG_LATITUDE(0)
        input_data(i).L_SHELL      = EPH_CDF_DATA(i).L_SHELL(0)
    ENDFOR 
ENDELSE 
input_header = eph_cdf_header

END 

;------------------------------------------------------------------------------
PRO r_polar_eph_cdf_oneday, date, VERBOSE=VERBOSE

COMMON polar_eph_cdf, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
  
get_err_no=0 & get_err_msg='Read eph cdf succesfully'
  
; NEW March 2003 - all cdf's are now local, dirs per year, and zipped!

if not (papco_check_data_env('POLAR_EPH_CDF', PATH = cdf_path)) then return

year_dir = strmid(date, 0, 4)+'/'

fln = cdf_path+year_dir+'po_or_pre_'+date+'*.cdf'
file = file_search(fln, count=c)

;see if file needs unzipping
zip_flag = 0
IF c EQ 0 THEN BEGIN
    file = file_search(fln+'.gz', count=c)
    IF c NE 0 THEN BEGIN
        message, 'Found zipped file -> unzipping', /info
        spawn, 'gunzip '+file(0)
        zip_flag = 1
        file = file_search(fln, count=c)
    ENDIF 
ENDIF

;if predicitive files are not found, check for definite ones...
IF c EQ 0 THEN BEGIN
    pos = strpos(cdf_path, 'pre')
    strput, cdf_path, 'def', pos
    fln = cdf_path+year_dir+'po_or_def_'+date+'*.cdf'
    file = file_search(fln, count=c)
    IF c EQ 0 THEN BEGIN
        file = file_search(fln+'.gz', count=c)
        IF c NE 0 THEN BEGIN
            message, 'Found zipped file -> unzipping', /info
            spawn, 'gunzip '+file(0)
            zip_flag = 1
            file = file_search(fln, count=c)
        ENDIF 
    ENDIF 
ENDIF

if c eq 0 then begin
    get_err_no=2 & get_err_msg='File '+date+'*.cdf'+' not found'
    message, get_err_msg, /info & return
endif else begin
    message,'File found: ',/info
    fln = file(c-1)                ;use latest version
    print, '  ', fln
ENDELSE 

    ;try to download
;    url=papco_getenv('POLAR_EPH_CDF_FTP')+'/'+year_dir
;    download_dir=cdf_path+year_dir
;    remote_file='po_or_pre_'+date+'_v01.cdf.gz'
;    papco_get_remote_data, url, remote_file, download_dir
;    file = file_search(fln, count=c)
;    fln = file(c-1)
;    IF c eq 0 THEN BEGIN
;        get_err_no=2 & get_err_msg='File '+date+'*.cdf'+' not found and cannot download'
;        message, get_err_msg, /cont & return
;    ENDIF
;endif
;message,'File found: ',/cont
;print, '  ', fln

;get rid of older versions on local
IF c GE 2 THEN file_delete, file(0:c-2), /verb

ON_IOERROR, read_error
!QUIET = 1

id_cdf = cdf_open(fln)
cdf_info = cdf_inquire(id_cdf)

;find out what variables are in file
var_names=strarr(cdf_info.nvars,3)
for i=0, cdf_info.nvars-1 do begin
    var_info=cdf_varinq(id_cdf,i)
    var_names(i,0)=var_info.name
    var_names(i,1)=var_info.datatype
    var_names(i,2)=var_info.recvar
    if keyword_set(VERBOSE) then $
        print,string(i,var_names(i,0),var_names(i,1),var_names(i,2), $
                     format="(i2.2,a20,a12,a10)")
ENDFOR 
  
; create structure for all constant data in file!   
index=where(var_names(*,2) eq 'NOVARY',c)
struct_assign=strarr(c) & count=0
for i=0, cdf_info.nvars-1 do begin  
    if var_names(i,2) ne 'NOVARY' then goto, next1  
    cdf_varget,id_cdf,var_names(i,0),buffer,rec_count=1    
    if var_names(i,1) eq 'CDF_CHAR' then buffer=string(buffer)
    if count eq 0 then dummy1=create_struct(var_names(i,0),buffer) $
    else dummy1=create_struct(dummy1,var_names(i,0),buffer)    
    count=count+1
    next1:
ENDFOR   
  
input_header=create_struct(name='POLAR_EPH_CDF_HEAD',dummy1)
  
; create structure for all variable data in file!   
index=where(var_names(*,2) eq 'VARY',c)
struct_assign=strarr(c) & count=0
for i=0, cdf_info.nvars-1 do begin
    if var_names(i,2) ne 'VARY' then goto, next2
    cdf_varget,id_cdf,var_names(i,0),buffer,rec_count=1
    if var_names(i,0) eq 'CRN' then var_name='CRN_EARTH' $
      else var_name=var_names(i,0)

    if count eq 0 then $
      dummy2=create_struct(var_name,buffer) $
    else $
      dummy2=create_struct(dummy2,var_name,buffer)
    count=count+1
    next2:
endfor  
  
dummy3=create_struct(name='POL_EPH_CDF_'+varprt(count),dummy2)
  
IF cdf_info.maxrec EQ 0 THEN GOTO, read_error
    

input_data=replicate(dummy3,cdf_info.maxrec)
  
; Read data into structure from cdf file
count=0
for i=0, cdf_info.nvars-1 do begin
    if var_names(i,2) ne 'VARY' then goto, next3
    cdf_varget,id_cdf,var_names(i,0),buffer,rec_count=cdf_info.maxrec
    input_data.(count)=buffer
    count=count+1
    next3:
endfor  
  
cdf_close,id_cdf
!QUIET = 0
IF zip_flag THEN spawn, 'gzip '+fln+' &'

return

read_error:
get_err_no = 6
get_err_msg = 'Problem reading CDF file'
print, !ERROR_STATE.MSG

END



;for 2002 LWS proposa showing L-coverage when maglat < 30 degrees
PRO polar_eph_coverage, PS = PS
  
COMMON polar_neweph, input_header, input_data 
COMMON papco_color_names
COMMON mjdt, mjdt_start, mjdt_end               ;common time limits in mjdt

names = [black, red, green, blue, magenta, cyan, $
                            burgundy, olive, dark_green, teal, $
                            royal_blue, violet]

IF keyword_set(PS) THEN BEGIN
    plfl='/u/friedel/LaTeX/official/proposals/2002/lws/friedel/polar_mlat.ps'
    xoffset=0.5 &  yoffset=0.5
    set_plot,'PS'
    xsize=8 & ysize=6
    device, /inches, /color, bits=8, filename=plfl, $
        xoffset=xoffset, yoffset=yoffset, xsize=xsize, ysize=ysize
ENDIF 

if !D.name EQ 'X' THEN window, 6
!p.position = [0.15, 0.15, 0.75, 0.95]
plot, findgen(10), findgen(10), xrange = [1, 9], yrange = [-34, 34], $
  /nodata, color = black, xtitle = 'L (Olson Pfitzer),  Re', $
  ytitle = 'MLAT (Olson Pfitzer), deg', charsize = 1.5, xticks = 8
step = 210

FOR i = 10, 2500, step DO BEGIN
    
    r_polar_new_eph, VERBOSE=1, ORBIT=i
    
    idx = where(abs(input_data.MODEL_0_0.MLAT) LE 30.0, c)
    IF c NE 0 THEN BEGIN
        mlt = input_data(idx).MODEL_0_0.MLAT
        L = input_data(idx).MODEL_0_0.L
        col = names((i/step) MOD 12)
        oplot, L, mlt, psym = 4, symsize = 0.5, color = col
        convert_t90_to_date,mjdt_start,year,day_oy,hour,minute,second, $
                          month,cmonth,day_om,/mjdt
        date_st = string(year,month,day_om,format = "(i4.4,'/',i2.2,'/',i2.2)")
        xyouts, 0.77, 0.85-0.04*(i/step), date_st, $
          /normal,  color = col, charsize = 1.5
    ENDIF

ENDFOR

IF keyword_set(PS) THEN BEGIN
    device, /close
    set_plot, 'X'
ENDIF  


END 

