;******************************************************************************
;* FILE:
;*    $PAPCO_PATH/PAPCO_templates/HYDRA_Overv_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by 
;*    $PAPCO_PATH/papco_panel.pro (a collection of standard PAPCO widgets)
;*    
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ
;*       $PAPCO_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $PAPCO_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* COMMON-BLOCKS:
;*    PAPCO_PANEL_TEMPLATE_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_PATH/papco_panel.pro for a
;*    description). 
;*       {PAPCO_PANEL_TEMPLATE_DATA, 
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_DOC/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  PAPCO_PANEL_TEMPLATE, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*       a=PAPCO_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=HYDRA_Overv_panel(a)
;*       which calls up the window.
;*       Interrogating a with help,a,/str shows the contents. This is a good
;*       way of testing your own panel editor. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  HYDRA_Overv_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the panel  editor window.
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      swittch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, swittch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supploed widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $          	 	
;*	                xut2:0.0}             	 	 
;*
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  HYDRA_Overv_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON HYDRA_Overv_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
	 column=1, $
	 GROUP_LEADER=group)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionalty that PAPCo supports.
; They dynamically extend the panel editors widget syructure, and all the 
; events they generate are supoported by PAPCO_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.

   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

; --- Electron / Ion ----------------------------------------------------------
   panel_base1=WIDGET_BASE(panel_base, /FRAME, COLUMN=2)

   elec_ion_bt_base=WIDGET_BASE(panel_base1, /FRAME, ROW=1, /EXCLUSIVE) 
   spb_electrons=WIDGET_BUTTON(elec_ion_bt_base, $ 
			VALUE='electrons', $
			UVALUE='spb_electrons')
   spb_ions=WIDGET_BUTTON(elec_ion_bt_base, $
			VALUE='ions', $
			UVALUE='spb_ions')

   IF aPlotInfo.ioptions(17) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_electrons, SET_BUTTON=1
      WIDGET_CONTROL, spb_ions, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_electrons, SET_BUTTON=0
      WIDGET_CONTROL, spb_ions, SET_BUTTON=1
   ENDELSE



; --- SLIDERS: ----------------------------------------------------------------

; --- Energy ------------------------------------------------------------------
   energyBase=WIDGET_BASE(panel_base, COLUMN=2, /FRAME)
   energyLeftBase=WIDGET_BASE(energyBase, /EXCLUSIVE, ROW=1)
   energyRightBase=WIDGET_BASE(energyBase, COLUMN=1)
   spb_allenergies=WIDGET_BUTTON(energyLeftBase, $
				VALUE='All energies        ',$
				UVALUE='spb_allenergies')
   spb_Selectenergies=WIDGET_BUTTON(energyLeftBase, $
				VALUE='Select energy   ',$
				UVALUE='spb_selectenergies')

   IF aPlotInfo.typeVector(0) ne 0 then BEGIN
       WIDGET_CONTROL, spb_allenergies, SET_BUTTON=0 
       WIDGET_CONTROL, spb_Selectenergies, SET_BUTTON=1
   ENDIF ELSE BEGIN
       WIDGET_CONTROL, spb_allenergies, SET_BUTTON=1
       WIDGET_CONTROL, spb_Selectenergies, SET_BUTTON=0
   ENDELSE

   sl_energy=WIDGET_SLIDER(energyrightBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(1) < 10 > 1, $
                        UVALUE='sl_energy',$
                        MINIMUM=1, MAXIMUM=10)
   lbl_energy=WIDGET_LABEL(energyrightBase, $
			VALUE=strcenter('energy level',50), $
			/ALIGN_CENTER)			     

; --- Pass-Number -----------------------------------------------------------
   PvalueBase=WIDGET_BASE(panel_base, COLUMN=2, /FRAME)
   PvalueLeftBase=WIDGET_BASE(PvalueBase, /EXCLUSIVE, ROW=1)
   PvalueRightBase=WIDGET_BASE(PvalueBase, COLUMN=1)
   spb_allPvalues=WIDGET_BUTTON(PvalueLeftBase, $
				VALUE='All Passes        ',$
				UVALUE='spb_allPvalues')
   spb_SelectPvalues=WIDGET_BUTTON(PvalueLeftBase, $
				VALUE='Select Pass   ',$
				UVALUE='spb_selectPvalues')

   tmp=WIDGET_INFO(panel_base, /GEOMETRY)
   sl_Pvalue=widget_slider(PvalueRightBase, $
       MINIMUM=1, MAXIMUM=4, /DRAG, $
 	 VALUE=(aPlotinfo.typeVector(3) < 4 >1), $
	 UVALUE='sl_Pvalue')
   lbl_Pvalue =WIDGET_LABEL(PvalueRightBase, $
			VALUE=strcenter('Pass', 50), $
			/ALIGN_CENTER)
   IF aPlotinfo.typeVector(2) ne 0 THEN BEGIN
      WIDGET_CONTROL, spb_allPvalues, SET_BUTTON=0
      WIDGET_CONTROL, spb_selectPvalues, SET_BUTTON=1
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_allPvalues, SET_BUTTON=1
      WIDGET_CONTROL, spb_selectPvalues, SET_BUTTON=0
      WIDGET_CONTROL, sl_Pvalue, SENSITIVE=0      
   ENDELSE


;--- add the PAPCO panel widget for choosing the color bar for color plots ----

   widgets=PAPCO_panel_AddColorBar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widget for choosing the y-axis-ranges ----------------

   widgets=PAPCO_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widgets for choosing the z-axis-ranges ---------------

   widgets=PAPCO_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.

   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exisit in the HYDRA_Overv example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpfile:'HYDRA_Overv_panel.help', $
                    typevector:aPlotInfo.typevector, $
		     $
		    sl_energy:sl_energy, $
		    lbl_energy:lbl_energy, $
		    spb_allenergies:spb_allenergies, $
		    spb_Selectenergies:spb_Selectenergies, $
		    spb_allPvalues:spb_allPvalues, $
		    spb_selectPvalues:spb_selectPvalues, $
		    sl_Pvalue:sl_Pvalue, $
		    lbl_Pvalue:lbl_Pvalue}

   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'HYDRA_Overv_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the typeVector-tag and swittch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 

   returnData={typevector:widgetData.typevector, $
               }
   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO mics_overview_panel_refresh, $
;*                SL_PITCH=SL_PITCH, SL_ENERGY=SL_ENERGY, SL_LVALUE=SL_LVALUE $
;*                PB_PITCH=PB_PITCH, PB_ENERGY=PB_ENERGY, PB_LVALUE=PB_LVALUE
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO mics_overview_panel_event
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R.Friedel 
;*
;***************************************************************************
PRO HYDRA_Overv_panel_refresh,  $
                 SL_ENERGY=SL_ENERGY, SL_PVALUE=SL_PVALUE, $
                 PB_ENERGY=PB_ENERGY, PB_PVALUE=PB_PVALUE

   COMMON HYDRA_Overv_panel_DATA, widgetData

   IF KEYWORD_SET(SL_ENERGY) then BEGIN
      Erange=fltarr(10)
      Prange=fltarr(10)

      IF widgetData.ioptions(17) THEN BEGIN	;----------------- ions
         WIDGET_CONTROL, widgetData.sl_energy, $
        			 SET_SLIDER_MIN=1, $
        			 SET_SLIDER_MAX=10
          IF widgetData.typeVector(1) GT 10 THEN BEGIN 
        	    widgetData.typeVector(1)=10
          	    WIDGET_CONTROL, widgetData.sl_energy, $
        			    SET_VALUE=widgetData.typeVector(1)>1
          ENDIF	 
          widgetData.typeVector(1)= widgetData.typeVector(1) < 10
          range=(widgetData.typeVector(1)) > 0
          Prange=[0.02,0.05,0.1,0.2,0.5,1.0,2.0,5.0,10.0,20.0]
          ekarr=Prange
          rangeLabel='(ion) '+$
	       strtrim(string(ekarr(range-1),format='(F8.3)'),2) + ' keV'
      ENDIF ELSE BEGIN  ;------------------------------------- electrons
          WIDGET_CONTROL, widgetData.sl_energy, $
        			 SET_SLIDER_MIN=1, $
        			 SET_SLIDER_MAX=10
          IF widgetData.typeVector(1) GT 10 THEN BEGIN 
        	    widgetData.typeVector(1)=10
          	    WIDGET_CONTROL, widgetData.sl_energy, $
        			    SET_VALUE=widgetData.typeVector(1)>1
          ENDIF	 
          widgetData.typeVector(1)= widgetData.typeVector(1) < 10
          range=(widgetData.typeVector(1)) > 0
          Erange=[0.02,0.05,0.1,0.2,0.5,1.0,2.0,5.0,10.0,20.0]
          ekarr=Erange
          rangeLabel='(electron) '+$
	       strtrim(string(ekarr(range-1),format='(F8.3)'),2) + ' keV'
     ENDELSE
     WIDGET_CONTROL, widgetData.lbl_energy, $
        	SET_VALUE=strcenter('energy range '+ rangeLabel, 50)
   ENDIF

  
   IF KEYWORD_SET(PB_ENERGY)  then BEGIN
     IF widgetData.typeVector(0) eq 0 then BEGIN  ; all energy
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=0
     ENDIF ELSE BEGIN                             ; SELECT ENERGY
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
     ENDELSE
   ENDIF

   IF KEYWORD_SET(PB_PVALUE) then BEGIN
     IF widgetData.typevector(2) eq 0 then BEGIN  ; all Passes
        WIDGET_CONTROL, widgetData.spb_allPvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectPvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_Pvalue, SENSITIVE=0
        return 
     ENDIF ELSE BEGIN                            ; select Pass
        WIDGET_CONTROL, widgetData.spb_allPvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_selectPvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_Pvalue, SENSITIVE=1
     ENDELSE
   ENDIF

   IF KEYWORD_SET(SL_PVALUE) then BEGIN
      WIDGET_CONTROL, widgetData.sl_Pvalue, $
			       SET_SLIDER_MIN=1, $
			       SET_SLIDER_MAX=4
      WIDGET_CONTROL, widgetData.sl_Pvalue, $
			       SET_VALUE=(widgetData.typevector(3)>1)
      Pvalue=float(widgetData.typevector(3)-1)
      PvalueLabel=string(Pvalue,format="(f2.0)")

      WIDGET_CONTROL, widgetData.lbl_Pvalue, $
			 SET_VALUE=strcenter('Pass '+$
			         PvalueLabel,50)   
   ENDIF

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO HYDRA_Overv_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;******************************************************************************
PRO HYDRA_Overv_panel_event, event

   COMMON HYDRA_Overv_panel_DATA, widgetData

   WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'PAPCO_panel_Event', which handles the events generated
; by the PAPCO-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

   CASE uval OF
	'spb_electrons' : BEGIN
               widgetData.ioptions(17)=0
               HYDRA_Overv_panel_refresh,/SL_ENERGY
		  END
	'spb_ions' : BEGIN
               widgetData.ioptions(17)=1
               HYDRA_Overv_panel_refresh,/SL_ENERGY
		  END
	'spb_allenergies' : BEGIN
               widgetData.typeVector(0)=0
               HYDRA_Overv_panel_refresh,/PB_ENERGY
		  END
        'spb_selectenergies' :BEGIN
               widgetData.typeVector(0)=1
               HYDRA_Overv_panel_refresh,/PB_ENERGY
		  END
	'sl_energy': BEGIN
               widgetData.typeVector(1)=event.value
               HYDRA_Overv_panel_refresh, /SL_ENERGY
		  END
	'spb_allPvalues': BEGIN
               widgetData.typeVector(2)=0
               HYDRA_Overv_panel_refresh, /PB_PVALUE
                END
	'spb_selectPvalues': BEGIN
               widgetData.typeVector(2)=1
               HYDRA_Overv_panel_refresh, /PB_PVALUE
                END
        'sl_Pvalue' : BEGIN
               widgetData.typeVector(3)=event.value
               HYDRA_Overv_panel_refresh, /SL_PVALUE
		END
 
	ELSE: $
   	   widgetData=PAPCO_panel_Event(event, widgetData)

   ENDCASE
;print,'after....',widgetData.ioptions(17:19)
END
