;******************************************************************************
;* FILE: polar_hydra_overview_panel.pro
;*
;* DESCRIPTION: 
;*    This is a demonstration on how to implement a papco-panel-editor.
;*    It makes use of all the features provided by $papco_XX/papco_panel.pro
;*    (a collection of standard papco widgets). It constructs an editor for a
;*    very simple papco-plot, that has all the default-functionality.
;*
;* COMMON-BLOCKS:
;*    polar_hydra_overview_panel_data   contains the window-information
;*
;* STRUCTURES:
;*    This widget inherits a PlotInfo structure as an argument. This
;*    structure is then modified by the user according to his/her needs:
;*    Either by including standard PAPCO widgets from papco_panel.pro,
;*    which extend this structure, or by modifying its tags. The modified
;*    structure is then handed back to PAPCO.
;* 
;*    The PlotInfo structure is created by a call to papco_getPlotInfoStruct
;*    and contains a series of tags which are used to describe a given panel.
;*    These tags come in three categories:
;*       1. Tags used by PAPCO internally - don't touch!          
;*       2. Tags added by including standard PAPCO widgets. You can edit 
;*          or preset these tags if you wish.
;*       3. User Tags for control of user-specified features. There
;*          are enough user tags specified to handle most needs. 
;*    The main limitation of the User Tags is that you're stuck with
;*    what has been defined in PAPCO. If you REALLY need additional tags,
;*    this entails changing the PAPCO core, and should be done by
;*    whoever maintains the core. You can do this yourself too, by editing
;*    pro papco_getPlotInfoStruct contained in $papco_XX/papco/papco.pro
;*    but your changes will then not propagate to the next PAPCO version.
;*
;*    The PAPCO 09 set of user tags is:
;*
;*       typeVector:intarr(4), $ ; user keyword (old, for compatibility)
;*       swittch:lonarr(3),     $ ; user keyword (old, for compatibility)
;*       options:fltarr(20),   $ ; user keyword (float options)
;*       ioptions:lonarr(16),  $ ; user keyword (long options)
;*       soptions:strarr(16)   $ ; user keyword (string options)
;*
;*    NB! If you need more variables to commincate between your panel 
;*    editor and your plot routine then you can use your own common block
;*
;*    For a more complete description of the other PlotInfo tags and the
;*    tags used by PAPCO supplied widget refer to
;*    $papco_doc/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  polar_hydra_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    To test you might try this to get a stand-alone widget:
;*       a=papco_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=polar_hydra_overview_panel(a)          which calls up the panel window.
;*    Interrogating a with help,a,/str shows the contents. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;*     August 1998, modified by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  polar_hydra_overview_panel, aPlotInfo, GROUP=group, 
;*                                            ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  polar_hydra_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON polar_hydra_overview_panel_data, widgetData
  COMMON polar_hydra_overview, leg_names, Earr, type_names

;--- if GROUP is not specified, the group-leader will be the desktop
  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
  panel_base=papco_widget_base(title=aPlotInfo.panelKind, column=1, /modal, $
                               GROUP_LEADER=group)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
  widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

; --- Options base -------------------------------------------------------------
  options_base=WIDGET_BASE(panel_base, ROW=1, /ALIGN_CENTER)
  
; --- Types of data option -----------------------------------------------------
  type_base=WIDGET_BASE(options_base, /FRAME, ROW=2, /EXCLUSIVE)
  spb_type=lonarr(n_elements(type_names))
  for i=0,n_elements(type_names)-1 do $
    spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=type_names(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))
  WIDGET_CONTROL, spb_type(aPlotInfo.ioptions(0)), SET_BUTTON=1 
    
; --- Reload option -----------------------------------------------------------
  reload_base=WIDGET_BASE(options_base, /FRAME, ROW=1, /EXCLUSIVE)
  reloadNames=['Relaod',  'Adopt' ]  
  spb_reload=lonarr(n_elements(reloadNames))
  for i=0,n_elements(reloadNames)-1 do $
     spb_reload(i)=WIDGET_BUTTON(reload_base, $
			VALUE=reloadNames(i), $
			UVALUE='spb_reload'+strtrim(string(i), 2))
  WIDGET_CONTROL, spb_reload(aPlotInfo.ioptions(1)), SET_BUTTON=1      
    
; --- Orbit Pass options ------------------------------------------------------ 
  leg_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=4, /EXCLUSIVE, $
                         /ALIGN_CENTER)
  spb_leg=lonarr(n_elements(leg_names))
  for i=0,n_elements(leg_names)-1 do $
    spb_leg(i)=WIDGET_BUTTON(leg_base, $
			VALUE=leg_names(i), $
			UVALUE='spb_leg'+strtrim(string(i), 2))

  WIDGET_CONTROL, spb_leg(aPlotInfo.ioptions(2)), SET_BUTTON=1
   
; --- Energy Channel options ---------------------------------------------------
  energy_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1, /ALIGN_CENTER)
  
  lbl_energy=WIDGET_LABEL(energy_base, VALUE='Energy Channel (ev)')
  channel_base=WIDGET_BASE(energy_base, ROW=2, /EXCLUSIVE, $
                          /ALIGN_CENTER)
  channel_names=strtrim(string(Earr, format="(f8.1)"),2)
  len=max(strlen(channel_names))
  spb_channel=lonarr(n_elements(channel_names))
  for i=0,n_elements(channel_names)-1 do $
    spb_channel(i)=WIDGET_BUTTON(channel_base, $
			VALUE=strcenter(channel_names(i),len+2), $
			UVALUE='spb_channel'+strtrim(string(i), 2))

  WIDGET_CONTROL, spb_channel(aPlotInfo.ioptions(3)), SET_BUTTON=1   

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    polar_hydra_overview. If you do not want to use them,
;    just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
  ;widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo)
  ;additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
  widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  ;widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
  ;additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_panel_AddYScale_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
  widgets=papco_panel_AddZScale_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the plot_type example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
  thisWidgetsData={helpfile:'polar_hydra_overview_panel.help', $
                   ioptions:aPlotInfo.ioptions}
  widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
  papco_Panel_Realize, panel_base, widgetData
  papco_xmanager, 'polar_hydra_overview_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
  returnData={ioptions:widgetData.ioptions}
  additionalRData=papco_panel_ReturnData(widgetData)

  return, create_struct(ReturnData, additionalRData)
   
END

;***************************************************************************
;* PROCEDURE:     
;*      PRO polar_hydra_overview_panel_refresh, $
;*                SL_PITCH=SL_PITCH, SL_ENERGY=SL_ENERGY, SL_LVALUE=SL_LVALUE $
;*                PB_PITCH=PB_PITCH, PB_ENERGY=PB_ENERGY, PB_LVALUE=PB_LVALUE
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO mics_overview_panel_event
;*
;* MODIFICATION HISTORY:       
;*      May 1999, written by R.Friedel 
;*;***************************************************************************
PRO polar_hydra_overview_panel_refresh,  $
                 SL_ENERGY=SL_ENERGY, SL_PVALUE=SL_PVALUE, $
                 PB_ENERGY=PB_ENERGY, PB_PVALUE=PB_PVALUE

  COMMON polar_hydra_overview_panel_data, widgetData

  IF KEYWORD_SET(SL_ENERGY) then BEGIN
    Erange=fltarr(10)
    Prange=fltarr(10)

      IF widgetData.ioptions(17) THEN BEGIN	;----------------- ions
         WIDGET_CONTROL, widgetData.sl_energy, $
        			 SET_SLIDER_MIN=1, $
        			 SET_SLIDER_MAX=10
          IF widgetData.typeVector(1) GT 10 THEN BEGIN 
        	    widgetData.typeVector(1)=10
          	    WIDGET_CONTROL, widgetData.sl_energy, $
        			    SET_VALUE=widgetData.typeVector(1)>1
          ENDIF	 
          widgetData.typeVector(1)= widgetData.typeVector(1) < 10
          range=(widgetData.typeVector(1)) > 0
          Prange=[0.02,0.05,0.1,0.2,0.5,1.0,2.0,5.0,10.0,20.0]
          ekarr=Prange
          rangeLabel='(ion) '+$
	       strtrim(string(ekarr(range-1),format='(F8.3)'),2) + ' keV'
      ENDIF ELSE BEGIN  ;------------------------------------- electrons
          WIDGET_CONTROL, widgetData.sl_energy, $
        			 SET_SLIDER_MIN=1, $
        			 SET_SLIDER_MAX=10
          IF widgetData.typeVector(1) GT 10 THEN BEGIN 
        	    widgetData.typeVector(1)=10
          	    WIDGET_CONTROL, widgetData.sl_energy, $
        			    SET_VALUE=widgetData.typeVector(1)>1
          ENDIF	 
          widgetData.typeVector(1)= widgetData.typeVector(1) < 10
          range=(widgetData.typeVector(1)) > 0
          Erange=[0.02,0.05,0.1,0.2,0.5,1.0,2.0,5.0,10.0,20.0]
          ekarr=Erange
          rangeLabel='(electron) '+$
	       strtrim(string(ekarr(range-1),format='(F8.3)'),2) + ' keV'
     ENDELSE
     WIDGET_CONTROL, widgetData.lbl_energy, $
        	SET_VALUE=strcenter('energy range '+ rangeLabel, 50)
   ENDIF
  
   IF KEYWORD_SET(PB_ENERGY)  then BEGIN
     IF widgetData.typeVector(0) eq 0 then BEGIN  ; all energy
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=0
     ENDIF ELSE BEGIN                             ; SELECT ENERGY
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
     ENDELSE
   ENDIF

   IF KEYWORD_SET(PB_PVALUE) then BEGIN
     IF widgetData.typevector(2) eq 0 then BEGIN  ; all Passes
        WIDGET_CONTROL, widgetData.spb_allPvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectPvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_Pvalue, SENSITIVE=0
        return 
     ENDIF ELSE BEGIN                            ; select Pass
        WIDGET_CONTROL, widgetData.spb_allPvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_selectPvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_Pvalue, SENSITIVE=1
     ENDELSE
   ENDIF

   IF KEYWORD_SET(SL_PVALUE) then BEGIN
      WIDGET_CONTROL, widgetData.sl_Pvalue, $
			       SET_SLIDER_MIN=1, $
			       SET_SLIDER_MAX=4
      WIDGET_CONTROL, widgetData.sl_Pvalue, $
			       SET_VALUE=(widgetData.typevector(3)>1)
      Pvalue=float(widgetData.typevector(3)-1)
      PvalueLabel=string(Pvalue,format="(f2.0)")

      WIDGET_CONTROL, widgetData.lbl_Pvalue, $
			 SET_VALUE=strcenter('Pass '+$
			         PvalueLabel,50)   
   ENDIF

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_hydra_overview_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO polar_hydra_overview_panel_event, event
  
  COMMON polar_hydra_overview_panel_data, widgetData

  WIDGET_CONTROL,GET_UVALUE=uval, event.id
  
  IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
    widgetData.ioptions(0)=fix(strfrom(uval, 8))
    return
  ENDIF      
  
  IF strleft(uval, 8) EQ 'spb_relo' THEN BEGIN
    widgetData.ioptions(1)=fix(strfrom(uval, 10))
    return
  ENDIF   
  
  IF strleft(uval, 7) EQ 'spb_leg' THEN BEGIN
    widgetData.ioptions(2)=fix(strfrom(uval, 7))
    return
  ENDIF     
    
  IF strleft(uval, 11) EQ 'spb_channel' THEN BEGIN
    widgetData.ioptions(3)=fix(strfrom(uval, 11))
    return
  ENDIF     
  
  CASE uval OF
    'spb_allenergies' : BEGIN
      widgetData.typeVector(0)=0
      polar_hydra_overview_panel_refresh,/PB_ENERGY
    END
    'spb_selectenergies' :BEGIN
      widgetData.typeVector(0)=1
      polar_hydra_overview_panel_refresh,/PB_ENERGY
    END
    'sl_energy': BEGIN
      widgetData.typeVector(1)=event.value
      polar_hydra_overview_panel_refresh, /SL_ENERGY
    END
    'spb_allPvalues': BEGIN
      widgetData.typeVector(2)=0
      polar_hydra_overview_panel_refresh, /PB_PVALUE
    END
    'spb_selectPvalues': BEGIN
      widgetData.typeVector(2)=1
      polar_hydra_overview_panel_refresh, /PB_PVALUE
    END
    'sl_Pvalue' : BEGIN
      widgetData.typeVector(3)=event.value
      polar_hydra_overview_panel_refresh, /SL_PVALUE
    END
 
    ELSE: widgetData=PAPCO_panel_Event(event, widgetData)
  ENDCASE
    
END
