;******************************************************************************
;* file p_polar_hydra_overview.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_polar_hydra_overview, panel, ioptions, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type cel_lav for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_polar_hydra_overview, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_polar_hydra_overview, panel, ioptions, OUTPUT=OUTPUT, $
                 OVERPLOT=OVERPLOT, $
                 PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 SUBTABLE=SUBTABLE, $
                 _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; Comments here should include an explanation of the plot options used
; in type, swittch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
  common yscale, yscl		       ;man/auto yscaling
  common zscale, zscl	               ;man/auto zscaling
  common shift_label, down             ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_polar_hydra_overview . 

  common polar_hydra_lav, hyd_lav_header, hyd_lav_data
  common orbitpart, leg
  common hydra_lav_extra, time, sp, lbin, lrange
  COMMON polar_hydra_overview, leg_names, Earr, type_names
  
  papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
  panelset,panel,AXIS=1 	       ;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.
  ytickno=50*(!p.position(3)-!p.position(1))
  if ytickno gt 10 then ytickno=10
  ytickno=fix(ytickno) > 1

; for the color plot routine, you need to set up arrays containing the upper  
; and lower limits of each channel. Here we need L-value bins.
  lrange=hyd_lav_header.L_max
  lbin=hyd_lav_header.lbin
  sp=hyd_lav_header.nosp
  
  nobins=(lrange-1)/float(lbin)+1
  yarr=fltarr(nobins,2)
  for i=0,nobins-1 do begin
    yarr(i,0)=1+i*hyd_lav_header.lbin
    yarr(i,1)=1+(i+1)*hyd_lav_header.lbin
  endfor

; ch is the energy channel to be chosen
  ch=ioptions(3)
  
; leg is the orbit leg to be plotted  
  leg=ioptions(2)
  time=hyd_lav_data.info(2,leg)
   
  xut1= mjdt_start.mjd+ mjdt_start.t / 86400.0                     
  xut2= mjdt_end.mjd  + mjdt_end.t   / 86400.0
   
  ; make the plot arrays needed
  zmat=fltarr(sp,nobins)  ; total counts
  yval=fltarr(sp,nobins)  ; no of samples in bin 
  
  for i=0,sp-1 do begin   ; fill the plot arrays
    zmat(i,*)=hyd_lav_data(i).lsum(ch,*,leg)
    yval(i,*)=hyd_lav_data(i).valdat(ch,*,leg)
  endfor

  index=where(yval ne 0)
  index2=where(yval eq 0)
  zmat(index)=zmat(index)/yval(index) ; get average counts in each bin 
  nodata=-99
  zmat(index2)=nodata                 ; set nodata flag
  index=where(zmat lt 1e-10,c)
  if c ne 0 then  zmat(index)=nodata  ; set nodata flag

  ctype_names = type_names(ioptions(0))+string(Earr(ch),format="(f8.1,' eV')")
  
  uztit='flux'
  uytitle='L-value'
  utitle=ctype_names+'!CPOLAR HYDRA '+'!C'+leg_names(leg)
  
  yst=yarr(0,0)
  yen=yarr(nobins-1,1)

; restrict data to actual time range requested - this makes for faster zooming

;   index=where((time ge xut1) and (time le xut2),count)
;   if count ne 0 then begin
;      zmat=zmat(index,*)
;      time=time(index)
;   endif

; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data. Nodata sets the nodata
; flag used for the dataset.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
  if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1)
    zmax=zscl(panel(0),2)
  endif else begin
    papco_autorange,zmat,zmin,zmax, $
      log=zscl(panel(0),3), exclude=0, nodata=nodata
    zscl(panel(0),1)=zmin
    zscl(panel(0),2)=zmax
  endelse

   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yen=yscl(panel(0),2)
   endif else begin
      yscl(panel(0),1)=yst
      yscl(panel(0),2)=yen
   endelse

; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data
; structure.
   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                  ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                  ytitle:uytitle, ztitle:uztit, yticks:ytickno, $
                  xtickformat:'noticks'}  

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black.
  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse  
  
; add keyword structure set here with the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
  
; change the xtickformat if the panel is the bottom one and needs to
; plot a time axis   
  if panel(0) eq 0 then begin
    extra_plotPar.xtickformat='MJD_DEC_1'  
  endif else begin
    extra_plotPar.xtickformat='HYDRA_MJD_DEC_MLT'    
  endelse  
   
; "down" sets the axis labeling level. 
  down=0

; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
  
  ;check the routine papco_plot_colorspec in papco_XX\papco_lib\color_plot
  ;for all the possible options of papco_plot_colorspec.
  ;Hint: you might want to use the resx, resy keywords, which set the
  ;resolution to plot add. Without these, the routine will try to use
  ;device resolution, which for postscript may result in HUGE files!
  ;resx, resy are in data coordinates. If the axis is logarithmic, 
  ;resx, resy need to be too.
  resx=0.3
  resy=hyd_lav_header.lbin
    
  papco_plot_colorspec,zmat,time,yarr,resx=resx,resy=resy,nodata=nodata, $
    _extra=extra_plotPar
  papco_color_bar, _extra=extra_plotPar
  
  
; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y  

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label - 0r use any other function you define yourself.
; The routine x_side_label puts x-axis label to the right of the axis.
  extra_plotPar.color=1
  if (panel(0) eq 0) then begin
    extra_plotPar.xtickformat='MJD_DEC_2'
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='HYDRA_MJD_DEC_ORB'
    down=2
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='HYDRA_MJD_DEC_MLT'
    down=3
    plot,time,time, _extra=extra_plotPar, /NODATA
    x_side_label,panel,'Date!Ctime!COrbit!CMLT'
  endif else x_side_label,panel,'MLT'

; plot times of legs
;  dummy=fltarr(n_elements(time))
;  dummy(*)=20
;  oplot,time,dummy,psym=1,col=1
  
  
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
  right_side_label,panel,utitle,/rot90

end 


;------------------------------------------------------------------------------
; mjd_dec_orb returns POLAR orbit numbers given time input in mjd_dec
function hydra_mjd_dec_orb, axis, index, t

  COMMON mjdt, mjdt_start, mjdt_end
  COMMON shift_label, down
  common hydra_lav_extra, intime, sp, lbin, lrange
  common orbitpart, leg
  common polar_hydra_lav, hyd_lav_header, hyd_lav_data

  down_str=''
  for i=0,down-1 do down_str=down_str+'!C'

  res=min(abs(intime-t),mindex) 
  orbit=hyd_lav_data(mindex).info(3,leg)

  orbit_str=string(orbit,format="(i4.4)")

  return,down_str+orbit_str

end

;------------------------------------------------------------------------------
; mjd_dec_mlt returns POLAR MLT of leg  given time input in mjd_dec
function hydra_mjd_dec_mlt, axis, index, t

  COMMON mjdt, mjdt_start, mjdt_end
  COMMON shift_label, down
  common hydra_lav_extra, intime, sp, lbin, lrange
  common orbitpart, leg
  common polar_hydra_lav, hyd_lav_header, hyd_lav_data

  down_str=''
  for i=0,down-1 do down_str=down_str+'!C'

  res=min(abs(intime-t),mindex) 
  mlt=hyd_lav_data(mindex).info(4,leg)

  mlt_str=string(mlt,format="(f4.1)")

  return,down_str+mlt_str

end
