;******************************************************************************
;*  NAME:        r_polar_hydra_overview
;*
;*  DESCRIPTION: Routine to read hydra_overv files. Adheres to PAPCO read
;*               standard - error common block, time common block
;*
;*  ENVIRONMENTAL VARIABLES NEEDED:
;*               HYDRA_OVERVIEW_PATH  set to data directory for hydra_overview
;*                     
;*  INPUT:       type  integer  chooses which daya files to read
;*                     0  e_all  electrons, all pitch angles
;*                     1  i_all  ions, all pitch angles 
;*                     2  e_fa   electrons, field aligned
;*                     3  e_fo   electrons, field opposed
;*                     4  i_fa   ions, field aligned
;*                     5  i_fo   ions, field opposed
;*                     
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    NO_CONV  if set, do not convert time to mjdt and do not
;*                        limit data to time set in 
;*                        common mjdt, mjdt_start, mjdt_end
;*
;*  DATE:        May 1999
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     
;******************************************************************************
pro r_polar_hydra_overview, type, NO_CONV=NO_CONV

  common polar_hydra_lav, hyd_lav_header, hyd_lav_data
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
  common polar_hydra_overview, leg_names, Earr, type_names
  
  get_err_no=0  &  get_err_msg='hydra_overv read succesfully'
  
; check for env. variable pointing to data directory
  if not papco_check_data_env('HYDRA_OVERVIEW_PATH',PATH=path) then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif
  
  name=type_names(type)
  fl='hydra_lav_'+name
  fln=path+fl+'.idl'
  
  ;check if file exists.
  result=findfile(fln,count=c)
  if c eq 0 then begin      
    c_url=papco_getenv('HYDRA_OVERVIEW_FTP_ARCHIVE')
    c_file=fln  &  c_download_dir=path
    papco_get_remote_data, c_url, c_file, c_download_dir
    ;check if file "arrived" and then read it
    if get_err_no eq 0 then begin
      print,'Got it!!'
    endif else begin
      get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
      return
    endelse  
  endif         
  
  on_ioerror,err
  restore,fln
  
  message,'Restored '+fl+'.idl',/cont
    
  if keyword_set(NO_CONV) then return
  
  time_ind = n_elements(hyd_lav_data)
  time = fltarr(time_ind,4)
  for in2 = 0, 3 do begin
    for in1 = 0, time_ind-1 do begin
      seconds = long(hyd_lav_data(in1).info(2,in2))
      day     = long(hyd_lav_data(in1).info(1,in2))
      year    = long(hyd_lav_data(in1).info(0,in2))
      timedummy = convert_tzero_to_t90(seconds, day, year, /mjdt)
      time(in1,in2) =double(timedummy.mjd)+double(timedummy.t) / double(86400.)
    endfor
  endfor
  
  dt_start = double(mjdt_start.mjd) + double(mjdt_start.t) / double(86400.)
  dt_end = double(mjdt_end.mjd) + double(mjdt_end.t) / double(86400.)
  
;--Check if time requested is covered, error if not
  index=where((time(*,0) ge dt_start) and (time(*,0) lt dt_end),count)
  if count lt 2 then begin
    msg= fl+'.idl: Time range not covered in file'
    message,msg,/CONT
    get_err_no=2
    get_err_msg=msg
    RETURN
  endif else begin
    hyd_lav_data=hyd_lav_data(index)
    time=time(index,*)
    for i=0, n_elements(hyd_lav_data)-1 do $
      hyd_lav_data(i).info(2,*) = time(i,*)
    hyd_lav_header.nosp=count
  endelse
    
  return
  
  err:
    msg=fl+'.idl: binary not available.'
    message,msg,/CONT
    get_err_no=2
    get_err_msg=!ERR_STRING
    print, !ERR_STRING
    
end
