;***************************************************************************
;* file papcoadd_polar_pwi
;*      addplottype_polar_pwi
;*      addplottype_pwi_sfr_mca
;*      addplottype_pwi_fce
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a PLOT_TYPE. 
;*
;* PAPCO is written such that if the files starting with "papcoadd" exist
;* in the PAPCO_MODULES directory, then the routines contained therin
;* are utilised to add that plot type into the PAPCO package. 
;* This ensures easy extendability of PAPCO; also makes PAPCO "configurable" 
;* by adding or leaving out specific plot types.
;*
;* THIS ROUTINE AND THE AUXILLIARY READ AND PLOT ROUTINES MUST BE INCLUDED
;* IN THE FILE $PAPCO_BASE/PAPCO_modules/papco_add_init.pro IN PAPCO_MODULES 
;* SO THAT THEY CAN BE COMPILED AT STARTUP.
;*
;* If the plot type is in itself a sub-menu containing more PLOT_TYPEs, then
;* these must be listed at the top of this file, as it is read by papco.
;* the entries for each sub-plotype must come after the line 
;* "file papcoadd_PLOT_TYPE" and before the "end" line, one entry per line.
;* Each of the routines here must then be duplicated for each sub-plotype
;* used: see $PAPCO_MODULES/papcoadd_crres_plots for an example.
;*
;* Apart from editing this template the user has to write a panel 
;* editor widget for PLOT_TYPE. Use the file 
;*           $PAPCO_BASE/PAPCO_templates/papco_panel.pro.
;* The naming convention for this file is PLOT_TYPE_panel.pro.
;* If slice functions are to be supported a slice program needs to be written
;* too. Use the template file
;*           $PAPCO_BASE/PAPCO_templates/papco_slice.pro.
;* The naming convention for this file is PLOT_TYPE_slice.pro.
;* Panels and slices are kept in $PAPCO_MODULES/panels&slice.
;*
;* Finally, a data read and data plot routine are required as well. The data
;* read will be specific for each problem. An example for a standard
;* line plot is given in the file 
;*           $PAPCO_BASE/PAPCO_templates/p_data_type_line.pro
;* and for a standard color spectrogram in 
;*           $PAPCO_BASE/PAPCO_templates/p_data_type_spec.pro. 
;* These templates can also be used as a basis for more complex plots.
;*
;* The Way these add-in routines work:
;* ===================================
;*
;* At the start-up of PAPCO, all module sub-directories \papco_interface are 
;* searched for all
;* files beginning with "papcoadd_". From the names of these files the
;* PLOT_TYPE is stripped off and used to construct the procedure name
;* "addplottype_PLOT_TYPE" - this procedure must be contained in the file
;* papcoadd_PLOT_TYPE and is used to add the button for this PLOT_TYPE to
;* the papco main window. Since "addplottype_PLOT_TYPE" simply returns
;* widget ID of the button the button itself can contain further sub-menus
;* for plot sub-types (e.g. the button for SWE can contain a menu of buttons
;* for a range of SWE plots, ie meoments, spectra, etc)
;* To inform papco of all these sub-plot types (and to avoid having to
;* write a seperate file for each sub-plot type) all the types have to
;* be listed in the header of the file papcoadd_PLOT_TYPE.pro. These entries
;* have to finish off with the line ";* end" - see papcoadd_crres_plots.pro
;* for an example. All PLOT_TYPE names are stored in the string array
;* PLOT_TYPEs and are placed in a common block:
;* COMMON ADDINS, PLOT_TYPES
;*
;* If there are no sub-plot types, then the PLOT_TYPE as used in the file name
;* for papcoadd_PLOT_TYPE.pro is used to construct the calls to the procedures
;* in that file. The convention for naming these procedures is
;* "procedure_name_PLOT_TYPE" , so that papco can always construct the right
;* procedure call when needed. 
;*
;* When there are sub-plot types, their names get used to construct the
;* required calls. There are 6 procedures that need to be called for each
;* PLOT_TYPE, so that a set of these 6 procedures has to be written for each
;* sub-plot type. In the same way, you need to write a panel editor for each
;* plot type (sometimes several plot sub-types may share the same editor,
;* e.g. multiple line plots or color spectra, see papcoadd_crres_plots).
;* 
;* In papco, whenever an action for a PLOT_TYPE is taken, the information
;* stored in the structure widgetData is queried. In this structure there
;* is an array of PlotInfo structures, which contain all the relevant 
;* information for a chosen panel. The field plotInfo.panelkind is loaded
;* with the name of the PLOT_TYPE. This is set by the user in this file in
;* the routine addpanel_PLOT_TYPE ( "panelKind = 'PLOT_TYPE'" ). THIS MUST
;* BE SET CORRECTLY AS ALL THE ROUTINE NAMES USED IN THIS ADD-IN FILE ARE
;* CONSTRUCTED USING THIS INFROMATION!
;*
;* Whenever papco has to undertake some action for a
;* PLOT_TYPE, the field plotInfo.panelkind is compared to the entries in the
;* (COMMON ADDINS) PLOT_TYPES array. If a match is found, the required
;* procedure name is constructed and called If not, an error occurs.
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_polar_pwi,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_pwi_fce, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO plottable_pwi_sfr_mca, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO descriptor_pwi_fce, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO descriptor_pwi_sfr_mca, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO descriptor_pwi_fce, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_pwi_fce
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO addpanel_pwi_sfr_mca
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_pwi_fce, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           PAPCO_EditPlot (papco_paneledit.pro).
;*
;*    PRO editor_pwi_sfr_mca, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           PAPCO_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_pwi_fce, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;*
;*    PRO  draw_pwi_sfr_mca, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;*
;*    PRO  draw_pwi_mcae_spec, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;*
;*    PRO  draw_pwi_mcab_spec, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;*
;*    PRO  draw_pwi_sfra_spec, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;*
;*    PRO  draw_pwi_sfrb_spec, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;***************************************************************************

;***************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_polar_pwi, plottypes_base, buttonGeometry, pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this PLOT_TYPE to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      PLOT_TYPEs.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      add_plottype_polar_pwi,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addplottype_polar_pwi, plottypes_base, buttonGeometry, pdm_polar_pwi

; VALUE assigns a label for the button
; The plot_type portion of UVALUE is used to construct the calls to all
; other procedures in this add-in needed by PAPCO.

; example for simple button

;  pb_plot_type = WIDGET_BUTTON(plottypes_base,  $
;                                 VALUE = strcenter('plot_type',14), $
;                                 UVALUE = 'pb_plot_type', $
;                                 _extra = buttonGeometry)

;example for simple pull-down for sub-types

    MenuDesc = [ { CW_PDMENU_S, 3, strcenter('POLAR_PWI',10) }, $ 
	{ CW_PDMENU_S, 0, ' PWI_SFR_MCA   ' }, $
	{ CW_PDMENU_S, 0, ' PWI_Fce   ' } ]

    pdm_polar_pwi = CW_PDMENU(plottypes_base, MenuDesc, UVALUE='pdm_polar_pwi')

;note the convention for naming this widget "pdm_..." for pull-down menu.
;this obviously needs to replace "pb_..." in the call.

END

;***************************************************************************
;* PROCEDURE:     
;*      PRO plottable_pwi_fce, plottable
;*      PRO plottable_pwi_sfr_mca, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable	a boolean 0 if plottable only by CRRES orbit
;*				  1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_PLOT_TYPE, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;***************************************************************************
pro plottable_pwi_fce, plottable

   plottable = 1

END

pro plottable_pwi_sfr_mca, plottable

   plottable = 1

END

;***************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_pwi_fce, plotInfo, descriptStr
;*      PRO descriptor_pwi_sfr_mca, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_PLOT_TYPE, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro descriptor_pwi_fce, plotInfo, descriptStr

   descriptStr = ' Polar PWI Fce'

END

pro descriptor_pwi_sfr_mca, plotInfo, descriptStr

print, 'descriptor typeVec', plotInfo.typeVector

    case plotInfo.typeVector(1) of
	0: type = 'Polar PWI MCA E Spec'
	1: type = 'Polar PWI MCA B Spec'
	2: type = 'Polar PWI SFR A Spec'
	3: type = 'Polar PWI SFR B Spec'
    endcase

   descriptStr = type

END

;***************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_polar_pwi
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      using an instance of the PAPCO_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled.
;*      panelEditor must be set to the name of the panel editor routine
;*
;* INPUTS:       
;*      event              event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_PLOT_TYPE
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_polar_pwi, event

    panelKind = ''	&	panelEditor = ''
print, 'Event.Value', Event.Value
    Case Event.Value of

	1: begin
	    panelKind = 'pwi_sfr_mca'
	    panelEditor = 'pwi_sfr_mca_panel'
	end

	2: begin
	    panelKind = 'pwi_fce'
	    panelEditor = 'pwi_fce_panel'
	end

    endCase
    defaults = PAPCO_getPlotInfoStruct()
    defaults.typeVector = [0,0,0,0]
    PAPCO_Add_Plot, panelEditor, panelKind, initialPlotInfo = defaults

END

;***************************************************************************
;* PROCEDURE:     
;*      PRO editor_pwi_fce, plotNumber, newPlotData
;*      PRO editor_pwi_sfr_mca, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      PAPCO_EditPlot (papco_paneledit.pro).
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0 <= plotNumber <= widgetData.numberOfPlots
;*                  the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_PLOT_TYPE, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
PRO editor_pwi_Fce, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo = widgetData.plotInfos(plotNumber)
   newPlotData = pwi_fce_panel(plotInfo, $
        ACTION = 'Update '+plotInfo.panelKind, $
      GROUP = widgetData.main_base)

END

PRO editor_pwi_sfr_mca, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo = widgetData.plotInfos(plotNumber)
   newPlotData = pwi_sfr_mca_panel(plotInfo, $
        ACTION = 'Update '+plotInfo.panelKind, $
      GROUP = widgetData.main_base)

END

;***************************************************************************
;* PROCEDURES:     
;*      PRO draw_pwi_fce, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;*      PRO draw_pwi_mcab_spec, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;*      PRO draw_pwi_mcae_spec, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;*      PRO draw_pwi_sfra_spec, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;*      PRO draw_pwi_sfrb_spec, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;*      PRO draw_pwi_sfr_mca, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype.
;*      
;*
;* INPUTS:       
;*      panelVector      position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo      structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call   a string containing the data call. Used by other
;*                      routines that need to know how to get the data
;*                      in a panel, e.g. to draw spectra, distribution functions
;*      IsRead          a boolean which is set when the the data is read the
;*                      first time. Useful if a given data-read puts data
;*                      into commons which can be plotted in many ways.
;*                      When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 swittches. Currently
;*                      These are supported (this may be extended later)
;*                       1 = mouse zooms 
;*                       2 = mouse slices
;*                       4 = mouse calls level zero
;*                       8 = mouse calls polar slice
;*                       16 = mouse views ephemers
;*                       32, 64, 128 = not supported yet
;*                       0 = disable all
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type = 5  support zoom and level zero
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      draw_PLOT_TYPE, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro draw_pwi_fce, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead

    COMMON extra, extra_PlotInfo  ; passes in PAPCO-set plot keywords structure
    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2

    get_data_call = "r_pwi_fce"

    n = size(FCE_lp)
    if (n(0) GT 0) then begin
	mt = FCE_lp.nt - 1
	Overall_sec = FCE_lp.time(mt).T90 - FCE_lp.time(0).T90
	fac = 0.01 * Overall_sec

	if (FCE_lp.time(0).t90 LE xut1 + fac and $
		FCE_lp.time(mt).t90 GE xut2 - fac) then begin

; Overall_sec = #sec of data in common area
; Total_sec = # sec of data that we want
; At this point we know that the time period that we want is contained in the
; time period that we have

	    Total_sec = xut2 - xut1

;;; Re-read if we need HR data

	    if (Overall_sec LE 3600) then	IsRead = 1 $
	    else begin
		temp = Total_sec / Overall_sec
		if (temp LE 0.5) then IsRead = 0 $
		else	IsRead = 1
	    endelse

	endif else	IsRead = 0
    endif else	IsRead = 0

    print, 'Fce IsRead', IsRead

;    IsRead = 0
    slice_type = -1
    if (currPlotInfo.typeVector(0) GT 0) then	do_local = 1 $
    else	do_local = 0

    clr = currPlotInfo.color	; + 238
    case clr OF
	0: clr = 1	; black
	1: clr = 239	; bright red 60
	2: clr = 240	; lime	30
	3: clr = 241	; yellow	40
	4: clr = 242	; royal	8
	5: clr = 243	; magenta
	6: clr = 244	; cyan	18
	7: clr = 245	; white
	8: clr = 246	; maroon
	9: clr = 247	; olive?  mustard?
	10: clr = 248	; green
	11: clr = 249	; turquoise
	12: clr = 250	; dk. blue	3
	13: clr = 251	; purple
	14: clr = 252	; gray	209
	else: begin
	    print, 'Else clr = ', clr
	    if (clr GT 255) then	clr = clr Mod 256
	end
    endcase

    n = size(extra_PlotInfo)
    if (n(0) GT 0) then $
	extra_PlotInfo = cat_struct( { color:clr }, extra_PlotInfo) $
    else	extra_PlotInfo = { color: clr }
; print, 'extra_PlotInfo ', extra_PlotInfo

    IF (IsRead) THEN BEGIN
	if (currPlotInfo.overplot) then $
	    p_pwi_fce, panelVector, currPlotInfo.typeVector(1)+1, /OVERPLOT, $
			_EXTRA = extra_PlotInfo $
	else	p_pwi_fce, panelVector, currPlotInfo.typeVector(1)+1, $
			_EXTRA = extra_PlotInfo
    ENDIF ELSE BEGIN
	r_pwi_fce, do_local
      
	IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
				panelVector, orbitNr, $
				currPlotInfo.typeVector, $
				currPlotInfo.channel) THEN BEGIN
	    IsRead = 1
	    if (currPlotInfo.overplot) then $
		p_pwi_fce, panelVector, currPlotInfo.typeVector(1)+1, $
			/OVERPLOT, _EXTRA = extra_PlotInfo $
	    else	p_pwi_fce, panelVector, currPlotInfo.typeVector(1)+1, $
				_EXTRA = extra_PlotInfo
	ENDIF
    ENDELSE

END

pro draw_pwi_mcab_spec, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead, do_local

    COMMON extra, extra_PlotInfo  ; passes in PAPCO-set plot keywords structure
    COMMON MCABspctrmstuff, MCABspctrm
    COMMON time, xut1, xut2

    get_data_call = "r_pwi_mcab_spec"

    n = size(MCABspctrm)

    if (n(0) GT 0) then begin
	mt = MCABspctrm.nt - 1
	Overall_sec = MCABspctrm.time(mt).T90 - MCABspctrm.time(0).T90
	fac = 0.01 * Overall_Sec

	if (MCABspctrm.time(0).t90 LE xut1 + fac and $
		MCABspctrm.time(mt).t90 GE xut2 - fac) then	begin
	    Total_sec = xut2 - xut1
;;; Re-read if we need HR data

	    if (Overall_sec LE 1800) then	IsRead = 1 $
	    else begin
		temp = Total_sec/Overall_sec
		if (temp LE 0.5) then	IsRead = 0 $
		else	IsRead = 1
	    endelse

	endif else	IsRead = 0
    endif else	IsRead = 0

;    print, 'MCAB IsRead', IsRead
;    IsRead = 0

    IF (IsRead) THEN BEGIN
	p_pwi_mcab_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	_EXTRA = extra_PlotInfo
    ENDIF ELSE BEGIN
	r_pwi_mcab_spec, do_local
      
	IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
				panelVector, orbitNr, $
				currPlotInfo.typeVector, $
				currPlotInfo.channel) THEN BEGIN
	    IsRead = 1
	    p_pwi_mcab_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	    _EXTRA = extra_PlotInfo
	ENDIF
    ENDELSE

END

;-------------------------------------------------

pro draw_pwi_mcae_spec, panelVector, currPlotInfo, orbitNo, $
				get_data_call, slice_type, IsRead, do_local

    COMMON extra, extra_PlotInfo  ; passes in PAPCO-set plot keywords structure
    COMMON MCAEspctrmstuff, MCAEspctrm
    COMMON time, xut1, xut2

    get_data_call = "r_pwi_mcae_spec"

    n = size(MCAEspctrm)
    if (n(0) GT 0) then begin
	mt = MCAEspctrm.nt - 1
	Overall_sec = MCAEspctrm.time(mt).T90 - MCAEspctrm.time(0).T90
	fac = 0.01 * Overall_sec

	if (MCAEspctrm.time(0).t90 LE xut1 + fac and $
			MCAEspctrm.time(mt).t90 GE xut2 - fac) then	begin
	    Total_sec = xut2 - xut1
;;; Re-read if we need HR data

	    if (Overall_sec LE 1800) then	IsRead = 1 $
	    else begin
		temp = Total_sec/Overall_sec
		if (temp LE 0.5) then	IsRead = 0 $
		else	IsRead = 1
	    endelse

	endif else	IsRead = 0
    endif else	IsRead = 0

;    print, 'MCAE IsRead', IsRead
;    IsRead = 0

    IF (IsRead) THEN BEGIN
	p_pwi_mcae_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	_EXTRA = extra_PlotInfo
    ENDIF ELSE BEGIN
	r_pwi_mcae_spec , do_local
      
	IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) THEN BEGIN
	    IsRead = 1
	    p_pwi_mcae_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	    _EXTRA = extra_PlotInfo
	ENDIF
    ENDELSE

END

;-------------------------------------------------

pro draw_pwi_sfra_spec, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead, do_local

    COMMON extra, extra_PlotInfo  ; passes in PAPCO-set plot keywords structure
    COMMON SFRAspctrmstuff, SFRAspctrm
    COMMON time, xut1, xut2

    get_data_call = "r_pwi_sfra_spec"

    n = size(SFRAspctrm)
    if (n(0) GT 0) then begin
	mt = SFRAspctrm.nt - 1
	Overall_sec = SFRAspctrm.time(mt).T90 - SFRAspctrm.time(0).T90
	fac = 0.01 * Overall_sec

	if (SFRAspctrm.time(0).t90 LE xut1 + fac and $
		SFRAspctrm.time(mt).t90 GE xut2 - fac) then	begin
	    Total_sec = xut2 - xut1
;;; Re-read if we need HR data

	    if (Overall_sec LE 4500) then	IsRead = 1 $
	    else begin
		temp = Total_sec/Overall_sec
		if (temp LE 0.5) then	IsRead = 0 $
		else	IsRead = 1
	    endelse

	endif else	IsRead = 0
    endif else	IsRead = 0
;    print, 'A IsRead', IsRead

;    IsRead = 0

    IF (IsRead) THEN BEGIN
	p_pwi_sfra_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	_EXTRA = extra_PlotInfo
    ENDIF ELSE BEGIN
	r_pwi_sfra_spec, do_local
      
	IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) THEN BEGIN
	    IsRead = 1
	    p_pwi_sfra_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	    _EXTRA = extra_PlotInfo
	ENDIF
    ENDELSE

END

;-------------------------------------------------

pro draw_pwi_sfrb_spec, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead, do_local

    COMMON extra, extra_PlotInfo  ; passes in PAPCO-set plot keywords structure
    COMMON SFRBspctrmstuff, SFRBspctrm
    COMMON time, xut1, xut2

    get_data_call = "r_pwi_sfrb_spec"

    n = size(SFRBspctrm)
    if (n(0) GT 0) then begin
	mt = SFRBspctrm.nt - 1
	Overall_sec = SFRBspctrm.time(mt).T90 - SFRBspctrm.time(0).T90
	fac = 0.01 * Overall_sec

	if (SFRBspctrm.time(0).t90 LE xut1 + fac and $
		SFRBspctrm.time(mt).t90 GE xut2 - fac) then	begin
	    Total_sec = xut2 - xut1
;;; Re-read if we need HR data
;	    if (Total_sec GT 4500.0) then	IsRead = 1 $
;	    else if (Overall_sec GT 4500.0) then	IsRead = 0 $
;	    else	IsRead = 1

	    if (Overall_sec LE 4500) then	IsRead = 1 $
	    else begin
		temp = Total_sec/Overall_sec
		if (temp LE 0.5) then	IsRead = 0 $
		else	IsRead = 1
	    endelse

	endif else	IsRead = 0
    endif else	IsRead = 0

;    print, 'B IsRead', IsRead

;    IsRead = 0

    IF (IsRead) THEN BEGIN
	p_pwi_sfrb_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	_EXTRA = extra_PlotInfo
    ENDIF ELSE BEGIN
	r_pwi_sfrb_spec, do_local
      
	IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) THEN BEGIN
	    IsRead = 1
	    p_pwi_sfrb_spec, panelVector, currPlotInfo.typeVector(1)+1	;, $
;	    _EXTRA = extra_PlotInfo
	ENDIF
    ENDELSE

END

pro draw_pwi_sfr_mca, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead

    COMMON extra, extra_PlotInfo  ; passes in PAPCO-set plot keywords structure
    COMMON MCAEspctrmstuff, MCAEspctrm
    COMMON MCABspctrmstuff, MCABspctrm
    COMMON SFRAspctrmstuff, SFRAspctrm
    COMMON SFRBspctrmstuff, SFRBspctrm
    COMMON time, xut1, xut2

    if (currPlotInfo.typeVector(0) GT 0) then	do_local = 1 $
    else	do_local = 0

    slice_type = 2

    plot_type = currPlotInfo.typeVector(1) + 1
print, 'draw_pwi_spec typeVec', currPlotInfo.typeVector

    Case plot_type of
	1:  draw_pwi_mcae_spec, panelVector, currPlotInfo, orbitNo, $
			get_data_call, slice_type, IsRead, do_local
	2:  draw_pwi_mcab_spec, panelVector, currPlotInfo, orbitNo, $
			get_data_call, slice_type, IsRead, do_local
	3:  draw_pwi_sfra_spec, panelVector, currPlotInfo, orbitNo, $
			get_data_call, slice_type, IsRead, do_local
	4:  draw_pwi_sfrb_spec, panelVector, currPlotInfo, orbitNo, $
			get_data_call, slice_type, IsRead, do_local
	else:  print, '	*Unknown plot_type:', plot_type
    endCase

end
