;***************************************************************************
;* FILE:
;*    $PAPCO_PATH/PAPCO_modules/POLAR_pwi/pwi_sfra_spec_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by 
;*    $PAPCO_PATH/papco_panel.pro (a collection of standard PAPCO widgets)
;*    
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ
;*       $PAPCO_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $PAPCO_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for PAPCO to run, read the 
;*    file addpapco_init.PRO in $PAPCO_MODULES
;*
;* COMMON-BLOCKS:
;*    PAPCO_PANEL_TEMPLATE_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_PATH/papco_panel.pro for a
;*    description). 
;*       {PAPCO_PANEL_TEMPLATE_DATA, 
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  PAPCO_PANEL_TEMPLATE, aPlotInfo, GROUP = group, ACTION = ACTION
;*    You might try this:
;*       a = PAPCO_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a = plot_type_panel(a)
;*       which calls up the window.
;*       Interrogating a with help, a, /str shows the contents. This is a good
;*       way of testing your own panel editor. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;***************************************************************************

;***************************************************************************
;* FUNCTION:     
;*      FUNCTION  pwi_sfra_spec_panel, aPlotInfo, GROUP = group, ACTION = ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,	
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[ 0, 0, 0, 0 ], $ 
;*                      swittch:[ 0, 0, 0 ], $
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	newPlotData = PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION =, GROUP =)
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;***************************************************************************
FUNCTION  pwi_sfra_spec_panel, aPlotInfo, GROUP = group, ACTION = ACTION

   COMMON pwi_sfra_spec_panel_data, widgetData

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP = 0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION = 'Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.

   additionalWData = { canceled:1 }

    if (strleft(ACTION, 6) NE 'Update') then begin
	aPlotInfo.zScaleMin = 1.0e-18
	aPlotInfo.zScaleMax = 1.0e-08

	aPlotInfo.yScaleMin = 2.677e+01
	aPlotInfo.yScaleMax = 8.0781425e+05
    endif

;--- create the base-window ----------------------------------------------------

    panel_base = WIDGET_BASE(title = aPlotInfo.panelKind, $
	 column = 1, GROUP_LEADER = group)

;--- the first part of the window will consist in the widgets ------------------
;    that control the height of that plot
   widgets = PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData = cat_struct(widgets, additionalWData)

;
; <<<<< insert your widgets ! (replace the label below) >>>>>>>>>
;
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $PAPCO_MODULES/panels&slice/more_panel_widgets
; 
; As the same panel is called when you edit a plot, ineterrogate the tags in the
; plotInforStruct (aplotInfo) and set your widgets accordingly. In this way the 
; widget is always reinitialized to its last state.
;
    type_base = WIDGET_BASE(panel_base, /FRAME, COLUMN = 2, /exclusive)

    typeNames = ['Import Data', 'Use Local Data']
    spb_type = lonarr (n_elements(typeNames))

    for i = 0, n_elements (typeNames) - 1 do $
 	spb_type (i) = WIDGET_BUTTON (type_base, VALUE = typeNames(i), $
 		UVALUE = 'spb_type' + strtrim (string(i), 2))

    WIDGET_CONTROL, spb_type (aPlotInfo.typevector(0)), SET_BUTTON = 1
 
;--- add the papco panel widget for choosing the color bar for color plots ----
    widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
    additionalWData=cat_struct(widgets, additionalWData)

;--- add the widgets for choosing the plot-style -------------------------------
;   widgets = PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
;   additionalWData = cat_struct(widgets, additionalWData)
   
;--- now, add the widgets for choosing the y-axis-ranges -----------------------
   widgets = PAPCO_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData = cat_struct(widgets, additionalWData)
   
;--- now, add the widgets for choosing the z-axis-ranges -----------------------
   widgets = PAPCO_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData = cat_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ----------
;    editor window.
   widgets = PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData = cat_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in $PAPCO_BASE/PAPCO/help and describe your data and your 
;    choices in your widget. Put in here also the ID's of the widgets you 
;    create so that you can use them in the events routine.
   thisWidgetsData = { helpfile:'pwi_spec_panel.help', $
                    typeVector:aPlotInfo.typeVector }
   widgetData = PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)
   widgetData.ioptions(17:19) = [ 0, 0, 0 ]

;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.

   xmanager, 'pwi_sfra_spec_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the typeVector-tag and swittch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 

   returnData = { typeVector:widgetData.typeVector, $
               }
   additionalRData = PAPCO_panel_ReturnData(widgetData)

   return, cat_struct(ReturnData, additionalRData)
END

;***************************************************************************
;* PROCEDURE:     
;*      PRO pwi_sfra_spec_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;***************************************************************************
PRO pwi_sfra_spec_panel_event, event

   COMMON pwi_sfra_spec_panel_data, widgetData

   WIDGET_CONTROL, GET_UVALUE = uval, event.id

; in this editor, we don't need a special event-processing, so we just
; pass the event on to 'PAPCO_panel_Event'. If you need to process
; events, you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your new widgets

   if (strleft(uval, 8) EQ 'spb_type') then $
	widgetData.typeVector(0) = fix(strfrom(uval, 8)) $
   else	widgetData = PAPCO_panel_Event(event, widgetData)

END
