;***************************************************************************
;* PROCEDURE:     
;*      p_pwi_fce, panel, type, $
;*           OVERPLOT = OVERPLOT, PLOTS_ATTHISPOSITION = PLOTS_ATTHISPOSITION,$
;*           _EXTRA = extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type DATA_TYPE for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale     
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              of the typeVector of the PAPCO structure PAPCO_PLOTINFO which
;*              is set by the panel editor. Additional fields that can be used
;*              if needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              For this data type has the following meaning:   
;*      type(0) 0 for electrons
;*              1 for ions
;*              2 for both in one plot (iowa style).
;*    type(1:2) not used      
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_data_type, panel, type, $
;*           OVERPLOT = OVERPLOT, PLOTS_ATTHISPOSITION = PLOTS_ATTHISPOSITION,$
;*           _EXTRA = extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;***************************************************************************
pro p_pwi_fce, panel, type, $
           OVERPLOT = OVERPLOT, PLOTS_ATTHISPOSITION = PLOTS_ATTHISPOSITION, $
           _EXTRA = extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT = OVERPLOT, PLOTS_ATTHISPOSITION = PLOTS_ATTHISPOSITION,
;_EXTRA = extra_par are not used at all here, but are in the call for
; consistency...

; Comments here should include an explanation of the plot options used
; in type, swittch or channel. Replace type in the call with whatever variables
; you need

   COMMON FCEstuff, FCE_lp

; One of the common blocks should contain the plot data as returned by the
; data read procedure. 
; The following common blocks are needed:

   COMMON time, xut1, xut2		;common time limits in T90
   COMMON yscale, yscl			;man/auto yscaling
   COMMON shift_label, down
   COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords, used by mouse
                                        ;                   functions
   COMMON PAPCO_color_names

   subtable = 0

   panelset, panel			;sets the panel position viewport
   PAPCO_set_subtable, subtable

   if (!d.name EQ 'Z') then	font = '!5' $
   else	font = '!3'

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

;   ytickno = 50*(!p.position(3)-!p.position(1))
;   if ytickno gt 10 then ytickno = 10

; since there is a common time axis, plots look better if they all use the same
; number of xticks. In PAPCO, this number is mostly 6

   oplot = KEYWORD_SET(OVERPLOT)

;print, 'oplot', oplot

   x_range = [ xut1, xut2 ]
   forward_function pwi_timeticks
   forward_function pwi_ticks
   xtickno = 5
   x_tickv = dblarr (30)
   if (NOT oplot) then	pwi_timeticks, x_range, xtickno, x_tickv, x_minor

; for the color plot routine, you need to set up arrays containing the upper  
; and lower limits of each channel. This needs to be changed if the y-range is 
; restricted as the dimensions between this array and tha data array need to
; check out. Here is an example for CRRES MEB energy channels.

; for color plots, the y-axis normally refers to a channel. You have to 
; re-scale the data array to the range required before passing it to the
; color plot routine, also the ymin and ymax arrays.

; the plot-calls normally use the arrays T90 for the time axis and yarr (zmat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

   T90 = FCE_lp.time(*).T90
   n = FCE_lp.nt

   CASE type OF
	1:	begin
	    nf = 14	&	PWI_freq = 10^(findgen (14)/4.0 + 0.75)
	end
	2:	begin
	    nf = 20	&	PWI_freq = 10^(findgen (20)/4.0 + 0.75)
	end
	3: begin
	    nf = 320	&	PWI_freq = 10^(findgen (320)/319.0 * 4.47966 + 1.42765)
	end
	4: begin
	    nf = 2
	    PWI_freq = intarr(nf)
	    if (yscl(panel(0)) GT 0) then $
		PWI_freq = [ yscl(panel(0), 1), yscl(panel(0), 2) ] $
	    else	PWI_freq = [ min(FCE_lp.fce), max(FCE_lp.fce) ]
	end
	else: begin
	    nf = 2
	    PWI_freq = intarr(nf)
	    PWI_freq = [ min(FCE_lp.fce), max(FCE_lp.fce) ]
	end
   endCASE

   if (yscl(Panel(0), 0) GT 0) then begin
        ymin = yscl(panel(0), 1)
        ymax = yscl(panel(0), 2)
        min_indx = 0    &       done = 0 
        while (NOT done) do begin 
            if (min_indx LT nf) then begin
                if (PWI_freq(min_indx) LT ymin) then $
                    min_indx = min_indx + 1 $ 
                else    done = 1
            endif else begin 
                get_err_msg = '**Improper Minimum frequency chosen ' + ymin 
                print, get_err_msg 
                return
            endelse
        endwhile
        max_indx = min_indx     &       done = 0 
        while (NOT done) do begin
            if (max_indx LT nf) then begin
                if (PWI_freq(max_indx) LT ymax) then $
                    max_indx = max_indx + 1 $
                else    done = 1
            endif else begin
		done = 1 	&	max_indx = max_indx - 1
	    endelse
        endwhile 
        freq = PWI_freq(min_indx:max_indx)
	index = where (T90 GE xut1 AND T90 LE xut2, kount)
        fce_y = FCE_lp.fce(index) 
	T90 = T90(index)
	nf = max_indx - min_indx + 1
   endif else begin
        yscl(panel(0), 1) = PWI_freq(0)
        yscl(panel(0), 2) = PWI_freq(nf - 1)
        freq = PWI_freq
	index = where (T90 GE xut1 AND T90 LE xut2, kount)
	fce_y = FCE_lp.fce(index) 
	T90 = T90(index)
   endelse

   if (NOT oplot) then	T90 = T90 - xut1

   yst = freq(0)	&	yf = freq(nf - 1)
   y_log = yscl(Panel(0), 3)

   this_par = {ylog:y_log}

    ctit2 = FCE_lp.title	&	cytit = FCE_lp.ytitle

    ctit2 = font + ctit2	&	cytit = font + cytit

    if (panel(0) eq 0) then begin
	convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, $
				smonth, scmonth, sday

	convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, $
				emonth, ecmonth, eday
 
	if (syear EQ eyear AND sdoy EQ edoy) then $
            xtitle = string (syear, '-', smonth, '-', sday, format = $
                                "(i4.4, a1, i2.2, a1, i2.2)") $
	else xtitle = string (syear, '-', smonth, '-', sday,  ' to ', $
			eyear, '-', emonth, '-', eday, format = $
                "(i4.4, a1, i2.2, a1, i2.2, a4, i4.4, a1, i2.2, a1, i2.2)")

	if (NOT oplot) then	cxtit = xtitle $
	else	cxtit = ''
    endif else 	cxtit = ''
    cxtit = font + cxtit

; NOTE: If you have converted your time array to T90, it is a long array
; with LARGE numbers. The IDL plot routine currently converts everything
; to FLOAT format internally: This can lead to rounding errors when you
; want to plot high resolotion times in this format. For plotting data with 
; second resolution or below you need to subtract the starting time from
; the time array and plot without this offset:

; reduce time to seconds since the beginning of the year 1990.
; idl plot routine can't handle second time resolution of long integers!

; If you want to plot data below second resolution you need to write your
; own means of doing so. PAPCO can only hand over start amd end times in 
; LONG seconds since the start of 1990.

;restrict array sizes to actual time range needed. this makes for faster zooming

; the variable used above have the following meaning:
; 	cytit : the y-axis label
; 	cxtit : the x-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; yscl of common block yscale is an array of dimensions (*, 3) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*, 0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*, 1)  is then ymin (or zmin)
;         yscl(*, 2)  is then ymax (or zmax)
;         yscl(*, 3)  is then ylog (0 = linear plot, 1 = logarithmic)
; when scaling is set to automatic, the automatic y-limits are loaded in.
; for color plots, the y-limits are used for the color (z-axis) scaling

; set up extra plot keywords, common ones first

  y_range = [ yst, yf ]

  extra_plotPar_common = { xrange: x_range, yrange: y_range, noerase: 1, $
	ticklen: -0.01, xtitle: cxtit, ytitle: cytit }

; check if the passed-in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then	extra_par = { color: 1 }

; add keyword structure specific to the subtype of plot chosen here
  
   extra_plotPar_common = cat_struct(this_par, extra_plotPar_common)

   color1_plotPar = cat_struct({ color: 1 }, extra_plotPar_common)

; add keyword structure set here with the one passed in

   extra_plotPar_common = cat_struct(extra_par, extra_plotPar_common)

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot 

   if panel(0) eq 0 then begin
     if (NOT oplot) then $
	extra_plotPar = cat_struct({ xticks:xtickno, xminor:x_minor, $
		xtickv:x_tickv, xtickformat:'pwi_ticks', ytitle:cytit }, $
			color1_plotPar)
   endif else 	extra_plotPar = cat_struct({ xtickformat: 'noticks', $
				ytitle:cytit }, color1_plotPar)

; now do the actual data plot. For the color plots, you need to plot the axis
; and then store the coordinate sytem in the common block. The draw the
; color plot and then redraw the axis once more.

    down = 0
    plot, x_range, y_range, _extra = extra_plotPar_common, /NODATA, $
		xstyle = 5, ystyle = 5

; store the coordinate information into common block

   plotted_x = !x	&	plotted_y = !y

;do line plot

    plot, T90, fce_y, xrange = x_range, yrange = Y_range, _Extra = $
		extra_plotPar_common, xstyle = 5, ystyle = 5	; inhibit axes

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color = 1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; if you have reduced the time base to seconds since 1990 use the tick routine
; xticksdate3 to still get the correct date ticks.

  if (not oplot) then begin	; plot the axes
    plot, x_range, y_range, _extra = extra_plotPar, /NODATA, xstyle = 1, $
		ystyle = 1

; plot info right of the panel. If you want overplots to have their own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION. Since here colorbars are plotted, labels are rotated
; by 90 degrees and then plotted.

     right_side_label, panel, ctit2, /rot90
     if (panel(0) eq 0) then begin
	down = 1
	x_side_label, panel, 'time (UT)!CDate'
     endif

  endif

end 

