;+
; NAME: pwi_COLORBAR
;
; PURPOSE:
; This IDL-procedure plots a colorbar on the right side of the screen
;	
; CATEGORY:
;       Plotting (Color)
;
; CALLING SEQUENCE:
;
;       COLORBAR,ZMIN,ZMAX,[ztitle], zlog=zlog
;
; INPUTS:     in accordance with PLOT_CLR
;             ZMIN     float bottom value of color axis
;             ZMAX     float top value of color axis
;
;	
; PROCEDURE:  The color sequence is (and must be) linear, even for
;	      logarithmic z-scaling.
;             Axes are drawn using the AXIS- and LOGAXIS-procedures.
;             Viewport and data-screen relation are restored at the end.
;
; MODIFICATION HISTORY:
;       written jan 92 M.Fraenz,N.Krupp
;       customized (generalized, really) for hydra 9/20/96 by J.B.Faden
;       customized for pwi 05/97 by Julie Dowell
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO pwi_COLORBAR, ZMIN, ZMAX, ztitle, zlog = zlog

   COMMON PAPCO_color, PAPCO_color, r_papco, g_papco, b_papco

   zlog = keyword_set(zlog)
   
;  double stroke font for gif output.
   if !d.name eq 'Z' then	font='!5' else font=''

; find colorbar range

   cindex = where (PAPCO_color(*, 2) EQ 1)
   if (cindex(0) EQ -1) then begin
	print, 'Color system failure, contact jhd@space.physics.uiowa.edu'
	print, '	Using color set 0'
	cindex = 0
   endif

   color_range = PAPCO_get_Colorindices (cindex)
   cbase = color_range(0)
   ctop = color_range(1)
   clen = ctop - cbase

; save current viewport and data-screen-relation:
   pa = !p	&	xa = !x
   ya = !y	;	&	colrat = clen/255.0
   !x.title = ''	&	!p.TITLE = ''
   !y.title = ''	&	!p.ticklen = -0.2

   if (zlog) then begin
	if (ZMIN LT 1.0) then begin
	    tx = 0	&	done = 0
	    while (NOT done) do begin
		tzmin = 10.0^tx
		if (tzmin LT zmin) then	done = 1 $
		else	tx = tx - 1
	    endwhile
	endif else begin
	    tx = 0	&	done = 0
	    while (NOT done) do begin
		tzmin = 10^tx
		if (tzmin GE zmin) then	done = 1 $
		else	tx = tx + 1
	    endwhile
	    tx = tx - 1
	endelse
	tzmin = 10.0^tx
	!y.range = [ tzmin, ZMAX ]
   endif else begin
	tzmin = ZMIN
	!y.range = [ ZMIN, ZMAX ]
   endelse

   scalex = (!p.position(2) - !p.position(0))
   scaley = (!p.position(3) - !p.position(1))
   !p.position(0) = !p.position(2) + scalex/11.
   !p.position(2) = !p.position(0) + scalex/30.0
   SIZE = SQRT(scaley/scalex)*2.0
   !x.range = [0.,80.]
   !x.style = 4 + 1	&	!y.style = 1 + 4

   zfac = double (zmax - zmin)/clen
   PLOT, [0, 80], [zmin, zmin], color = !p.background, /noerase
   
;*******************create color bar (linear color sequence)
   FOR I = 1, clen DO BEGIN
       ZINT1 = (I - 1)*zfac + zmin	&	ZINT2 = (I)*zfac + zmin
       color = cbase + I - 1
       POLYFILL, [0., 0., 80., 80.], [ZINT1, ZINT2, ZINT2, ZINT1], col = color
   ENDFOR
   
;*******************axes and annotation (only here do we have to distinguish
;                   linear and logarithmic scaling)
   IF zlog THEN BEGIN ; logarithmic
       logmin = ALOG10(tzmin)	&	logmax = ALOG10(ZMAX)
       !y.ticks = (logmax - logmin)	&	!y.crange = alog10(!y.crange)
;	this is a trick to get around the range finding algorithm in LOGAXIS
       !p.color = 1	&	LOGAXIS, 3, ticklen = -0.6
;       the main tick length is 0.6 times the width of the color bar
   ENDIF ELSE BEGIN ; linear
       !y.ticks = FIX(zfac)/20		&	!x.style = 4
       !p.color = 1	&	AXIS, yaxis = 1, ystyle = 1, charsize = 1.0

  endelse
  
  !p.font = -1		&	!p.position(0) = !p.position(2)
  !p.position(2) = !p.position(0) + scalex/8.0
  !y.range = [0., 100.]
  x0 = 0.95
  y0 = (!p.position(1) + !p.position(3))/2
  PLOT, [0., 80.], [80.,  80.], /noerase, /nodata,  ystyle = 5,  xstyle = 5
  xyouts, x0, y0, /normal, ztitle, alig = 0.5, chars = 1.0, orient = -90., color = 1
  !x = xa	&	!y = ya		&	!p = pa

  return
end
