;***************************************************************************
;* 
;* PROCEDURE:     
;*      PWI_PLOT_CLR, Spect, time, frequency, zmin, zmax, linlogbit, smbit, $
;*					ylogbit
;* 
;* DESCRIPTION:
;*	plot a two-dimensional representation of three-dimensional data
;*
;* RESTRICTIONS:
;*      The x-coordinates are expected to be linear.
;*      The y-coordinates can be linear or logarithmic, but the exact
;*      y-field boundaries must be specified and be the same for all x.
;*
;*      If the output-device is not the screen or a printer with scalable pixel
;*	sizes, the routine 'PLOT_CLR_ALT' is used, which is extreme slow !
;*
;* HOW DOES THIS PROCEDURE DO ITS WORK - AND WHY ?
;*      Instead of plotting the data to the screen using the POLYFILL-Routines,
;*	the data is plotted into a two-dimensional data-array, 'image', which
;*	is output to the current device afterwards. 
;*	Therefore, the dimension of 'image' is the effective resolution the data
;*	is plotted with. As the available RAM is finite, and because the dimension
;*      of image are almost proportional to the size of created Postscript-files,
;*      image can't have the size XResolution X YResolution. 
;*	The height of 'image' is calculated by the function 'crres_calculate_Height'.
;*	The width of 'image' is
;*
;* INPUTS:     
;*      Spect    FLTARR(xdim, ydim) the spectrogram
;*      time     FLTARR(xdim) the x-axis values
;*	frequency FLTARR(ydim) the y-axis values
;*      ZMIN     float bottom value of color axis
;*      ZMAX     float top value of color axis
;*               linlogbit = 0: linear z-scaling
;*                         = 1: logarithmic z-scaling
;*	smbit	  = 0: no smoothing
;*		  = 1: smoothing (default)
;*	ylogbit	  = 1 if y-axis is to be logarithmic
;*		  = 0 if y-axis is to be linear
;*
;* OPTIONAL INPUTS:
;* 
;* KEYWORDS:
;*      USEOLD	  when specified, the old routine is used.
;*		  e.g.: PLOT_CLR,..., /USEOLD        
;*
;* OUTPUT: 
;*      a color-coded plot of ZARR on the current
;*      viewport (limits taken from !x.crange,!y.crange).
;*      Axes are drawn depending on the current
;*      value of the !type-variable. Usually x- and y-axes
;*      should be generated by the AXIS- or LOGAXIS-procedures,
;*      the z-axis by the COLORBAR-procedure.
;*      Color density is influenced by the last parameter
;*      in POLYFILLS (25 for PHASERPS).
;*      Additionally the ZARR is smoothed and interpolated in
;*      y-direction. (the variable ZARR is not changed)
;*	This feature can be turned off by calling with smbit = 0
;*
;*      
;* MODIFICATION HISTORY:       
;*	january 1992, M.Fraenz and N.Krupp
;*             	      wrote the routine 'plot_clr_alt' at the end of this
;*		      file 
;*      august 1995,  written by A.Keese, based on 'plot_clr_alt'
;*		      performance-enhancement by using the TV-Routine instead of the
;*		      polyfill-routine
;*      18th august 95, A.Keese
;*	              corrected the calculation of the vertical resolution
;*      21st august 95, A.Keese
;*		      corrected the calculation of the horiz resolution
;*		      corrected rounding error while calculating width of color-bars
;***************************************************************************
pro pwi_plot_clr, Spect, time, frequency, zmin, zmax, linlogbit, smbit, ylogbit

; the bottom 2 colors are reserved
; 0 = black	1 = 200 (background)

; set starting offset for color table. Offset to Offset + n defines range
; of color indices this routine may use

; if the print device is postscript, allow full range for color table

    common papco_color
    common papco_color_names
    common colors, r_orig, g_orig, b_orig, r_curr, g_curr, b_curr

; this procedure works with X-Display and Postscript only !
; if the printer does not supply resizable pixels, the older and slower routine
; must be used.

   if !d.name NE 'X' THEN BEGIN
      print, '% pwi_plot_clr: plotting data to printer...'
   endif else print, '% pwi_plot_clr: plotting data to display...'

   cindex = where (papco_color (*, 2) EQ 1)	; search the active vector
   if (cindex(0) EQ -1) then begin
	print, '% pwi_plot_clr: Color system failure; contact jhd@space.physics.uiowa.edu'
	print, '% pwi_plot_clr: Using color set 0'
	cindex = 0
   endif

   color_range = PAPCO_get_Colorindices (cindex)
   color_base = color_range(0)
   ncolors = color_range (1) - color_range(0)
   colortop = color_range(1) - 2

   ycrange = !y.crange

; check the dimensions of the data-fields. This is done, cause we don't want
; to be suspected of being wrong when an error occurs :-) Besides, this 
; simplifies debugging...

   dimX   = size(time)		&	dimZ   = size(Spect)

   if dimX(0) NE 1 OR dimZ(0) NE 2 then begin
      print, replicate('-',30)
      print, '% pwi_plot_clr: PARAMETER Error !'
      print, 'dimZ(0) ', dimZ(0), ', dimX(0) ', dimX(0)
      print, '% pwi_plot_clr: one or more of the dimensions of '
      print, '% pwi_plot_clr: Spect are wrong.'
      print, replicate('-',30)
      help, /traceback
      print, replicate('-',30)
      return  
   endif

   if dimZ(1) NE dimX(1) then begin
      print, replicate('-',30)
      print, '% pwi_plot_clr: PARAMETER-RANGE-Error !'
      print, 'dimZ(1) ', dimZ(1), ', dimX(1) ', dimX(1)
      print, '% pwi_plot_clr: one or more ranges of the dimensions of '
      print, '% pwi_plot_clr: Spect are wrong.'
      print, replicate('-',30)
      help, /traceback
      print, replicate('-',30)
      return      
   endif

   ydim = n_elements(frequency)	&	xdim = n_elements(time)
   y_min = frequency(0)		&	y_max = frequency(ydim - 1)
   x_min = 0.0		&	x_max = time(xdim - 1) - time(0)

; To turn on smoothing in the vertical direction, the bounds of the color-fields,

  if smbit then begin
     smdim = FIX(320.0/ydim)           ; interpolation (1 = no interpolation)

     if (smdim GT 1) then	newydim = smdim*(ydim) $
     else	newydim = ydim
     freq_out = fltarr(newydim)	&	spect_out = fltarr(newydim, xdim)
     if (smdim GT 1) then begin
	freq_out = rebin (frequency, newydim)
	spect_out = rebin (Spect, xdim, newydim)
     endif else begin
	freq_out = frequency	&	spect_out = Spect
     endelse
  endif else begin
    freq_out = frequency	&	spect_out = Spect
    newydim = ydim
  endelse ; (smbit)
  
; if the colors shall be plotted logarithmic, adjust the ranges...

   if (linlogbit) then begin
     zfac = double(alog10(zmax) - alog10(zmin))
     logmin = alog10(zmin)	&	logmax = alog10(zmax)
   endif else	zfac = double(zmax - zmin)

; get the rectangle in which to plot (in Device-coordinates)
   NormalXY0 = [!p.position(0), !p.position(1)]  ; Bottom left corner in normal coords
   NormalXY1 = [!p.position(2), !p.position(3)]  ; Top right corner in normal coordinates
   DeviceXY0 = round(convert_coord(NormalXY0,/normal,/to_device)) ;bl corner (dev-coords)
   DeviceXY1 = round(convert_coord(NormalXY1,/normal,/to_device)) ;tr corner (dev-coords)
   DeviceDX = DeviceXY1(0) - DeviceXY0(0)	; width of the plot-rectangle
   DeviceDY = DeviceXY1(1) - DeviceXY0(1)	; its height
   print, '% pwi_plot_clr:  Device WxH ', DeviceDX, ' x ', DeviceDY

;-- Calculate the size of the box in which the image is to be calculated------------
; the image is plotted using the TV-command. The two-dimensional array 
; image will be filled with the plotted bitmap. 
; we want the width of 'image' to be a multiple or a divisor of DeviceDX, and we
; want it to be close to N. If we would choose imageWidth to be not a multiple or 
; divisor of DeviceDX, the output might be influenced by rounding errors.
; The same applies to the image height and the maximal Y-coordinate YMAXOutMax

; calWidth is the horizontal resolution we wish to use. It is choosen so that
; each Element of XARR will be visible. 

; calculate image-width

  imageWidth = xdim		; number of elements to be plotted

  xrange = double (!x.crange(1) - !x.crange(0))

  imageHeight = newydim

  heightFactor = DOUBLE(DeviceDY)/DOUBLE(imageHeight)

  if (imageWidth LT imageHeight) then $
	wfact = fix (imageHeight/imageWidth) + 1 $
  else	wfact = 1

  image = bytarr (imageWidth, imageHeight)

  print, '% pwi_plot_clr: Image resolution (WxH)', imageWidth,'x',imageHeight

; calculate vertical scaling. We use the IDL-routines, so that set coordinate
; systems will be applied. Afterwards, the scaling is transformed to our
; imageHeight by dividing by heightFactor

;  xDummy = indgen(N_ELEMENTS(freq_out))                  
;  YY = convert_coord(xDummy, freq_out, /DATA, /TO_DEVICE) - DeviceXY0(1)
;  YMINout = round(YY(1,*)/heightFactor) < imageHeight - 1  ; go sure, that
;  YMINout = YMINout > 0			          ; ranges in YMAXout are ok.
;  YY = convert_coord(xDummy, freq_out, /DATA, /TO_DEVICE) - DeviceXY0(1)
;  YMAXout = round(yy(1,*)/heightFactor) < imageHeight - 1   ; make sure the
;  YMAXout = YMAXout > 0			          ; ranges in YMAXout are ok.

  lo_index = where (spect_out LT ZMIN, lo_count)
  if (lo_count GT 0) then	spect_out(lo_index) = ZMIN

  hi_index = where (spect_out GT ZMAX, hi_count)
  if (hi_count GT 0) then	spect_out(hi_index) = ZMAX

  if (linlogbit) then $
    image = bytscl (alog10(spect_out), min = logmin, max = logmax, top = colortop) + 2 $
  else	image = bytscl (spect_out, min = ZMIN, max = ZMAX, top = colortop) + 2

    xmax = xdim - 1	&	ymax = ydim - 1

    print, '% pwi_plot_clr: time(0) ', time(0), ', time(xmax) ', time(xmax)
;	print, 'ycrange ', ycrange
    if (ylogbit) then begin
	lld = convert_coord (x_min, y_min, /data, /to_device)
	urd = convert_coord (x_max, y_max, /data, /to_device)
    endif else begin
	lld = convert_coord (x_min, y_min, /data, /to_device)
	urd = convert_coord (x_max, y_max, /data, /to_device)
    endelse
;	print, 'x_min, y_min', x_min, y_min
;	print, 'x_max, y_max', x_max, y_max
    print, '% pwi_plot_clr: urd(0) ', urd(0), ', lld(0) ', lld(0)
    print, '% pwi_plot_clr: urd(1) ', urd(1), ', lld(1) ', lld(1)
    xsized = long(urd(0) - lld(0))
    ysized = long(urd(1) - lld(1))
    print, '% pwi_plot_clr: xsized ', xsized, ', ysized ', ysized

;-- now, the image has been created, and is being plotted after being scaled to
;   the display size

    if (!d.name EQ 'PS') then begin

	xpx_per_cm = float(!d.x_px_cm)
	ypx_per_cm = float(!d.y_px_cm)
	print, '% pwi_plot_clr:  converting pixels to cm'
	xsizec = xsized/xpx_per_cm	; convert to cm
	ysizec = ysized/ypx_per_cm
	llc0 = lld(0)/xpx_per_cm	; convert to cm
	llc1 = lld(1)/ypx_per_cm
	print, '% pwi_plot_clr: xsizec ', xsizec, ', ysizec ', ysizec
	if (xpx_per_cm EQ 1000) then begin
	    xsized = xsized*72/2450
	    ysized = ysized*72/2450
	    if (xsized LT ImageWidth) then	txsize = xsized $
	    else	txsize = ImageWidth
	    if (ysized LT ImageHeight) then	tysize = ysized $
	    else	tysize = ImageHeight
;	    print, 'txsize, tysize', txsize, tysize
	    image_out = congrid(image, txsize, tysize)
	endif else image_out = image
;	lln = convert_coord (lld, /device, /to_normal)
;	urn = convert_coord (urd, /device, /to_normal)
;	xsizen = urn(0) - lln(0)
;	ysizen = urn(1)	- lln(1)
	tv, image_out, llc0, llc1, xsize = xsizec, ysize = ysizec, /centimeters
    endif else $
	tv, congrid(image, DeviceDX, DeviceDY), DeviceXY0(0), DeviceXY0(1), /device

   return
end

