;****************************************************************************
;*  NAME:        r_pwi_fce_net
;*
;*  DESCRIPTION: Routine to read data_type files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       none
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_fce_net

    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    max_components = 2000
    FCE = fltarr (max_components)
    PWI_time = dblarr (max_components)

    ideal_components = 1000	&	fill_val = 1.0e-32
    FCE_Sec = 9.2	&	prev_offset = 0.0

    aPWI_offset = 0.0	&	FCE_set = fltarr (4)

    get_err_no = 0

    total_sec = xut2 - xut1

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

; the PWI read routines all use Gnu wget to retrieve calibrated spectral
; densities via the internet, eliminating the need for files.

; the latest version of wget (as of 05/97) is available at
;       <URL:ftp://prep.ai.mit.edu/pub/gnu/wget-1.4.3.tar.gz>.
; if wget is installed in the system path, you should not need to do
; anything special to get this procedure to work.

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
				scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
				ecmonth, eday
    end_sod = start_sod + total_sec

    if (total_sec GT 9200) then begin
	total_spec = long(total_sec/FCE_Sec + 0.5)
	if (total_spec LT 1) then	total_spec = 1
	num_to_av = total_spec / ideal_components
	if (num_to_av LT 1) then	num_to_av = 1
	sec_interval = num_to_av * FCE_Sec
	if (sec_interval LT FCE_Sec) then	sec_interval = FCE_Sec
	delta_t = 1.5 * sec_interval

    endif else begin
	sec_interval = FCE_Sec	&	delta_t = 1.5 * sec_interval
    endelse

    start_offset = start_sod

    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
		sminute, ':', ssecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
		eminute, ':', esecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    command = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=fce&begin_time= ' + $
		beg_timestr + '&end_time= ' + end_timestr + '"'

    spawn, command, unit = lun

    DataOK = 1	&	i = 0L
    PWI_time(*) = 0.0

    readu, lun, FCE_set
    on_ioerror, bad_data0

    limit = delta_t	&	FCE_sum = 0.0	&	FCE_count = 0L
    FCE_off_sum = 0.D0
    aPWI_offset = FCE_set(0)

    while (DataOK) do begin
	prev_offset = aPWI_offset

	if (aPWI_offset GE limit) then begin

	    if (FCE_count GT 0) then begin
		FCE(i) = FCE_sum / FCE_count
		off_avg = FCE_off_sum/FCE_count
		PWI_time(i) = start_offset + off_avg
		FCE_count = 0L	&	FCE_sum = 0.0
		FCE_off_sum = 0.D0
		limit = off_avg + delta_t
		i = i + 1

		if (i GE max_components) then begin
		    print, '% r_pwi_fce_net:  need more columns (i = ', i, ')'
		    get_err_no = 2
		    get_err_msg = '*Time overflow in FCE'
		    if (end_sod - PWI_time(i - 1) LE 60.0) then begin
			print, get_err_msg
			goto, Finished0
		    endif else	goto, bad_data
		endif
	    endif

	    while (aPWI_offset GT limit) do begin	; fill gap with 0.0

;		print, i, ' offset ', aPWI_offset, ', limit ', limit
		if (i GT 0) then $
		    PWI_time(i) = PWI_time(i - 1) + delta_t $
		else	PWI_time(i) = start_offset
		FCE(i) = 0.0
		i = i + 1	&	limit = limit + delta_t

		if (i GE max_components) then begin
		    print, '% r_pwi_fce_net:  need more columns (i = ', i, ')'
		    get_err_no = 2
		    get_err_msg = '**Time overflow in FCE'
		    if (end_sod - PWI_time(i - 1) LE 60.0) then begin
			print, get_err_msg
			goto, Finished0
		    endif else	goto, bad_data
		endif

	    endwhile
	endif

	FCE_off_sum = FCE_off_sum + aPWI_offset
	FCE_sum = FCE_sum + FCE_set(1)
	FCE_count = FCE_count + 1

	readu, lun, FCE_set
	on_ioerror, Finished
        aPWI_offset = FCE_set(0)
    endwhile

    Finished: close, lun	&	free_lun, lun

    if (prev_offset LT total_sec) then begin
	prev_offset = prev_offset + delta_t
	while (prev_offset LT total_sec) do begin
	    PWI_time(i) = start_offset + prev_offset
	    FCE(i) = 0.0
	    prev_offset = prev_offset + sec_interval
	    i = i + 1

	    if (i GT max_components) then begin
		print, '% r_pwi_fce_net:  need more columns (i = ', i, ')'
		get_err_no = 2
		get_err_msg = '****Time overflow in FCE'
		if (end_sod - PWI_time(i - 1) LE 60.0) then begin
		    print, get_err_msg
		    goto, Finished0
		endif else	goto, bad_data
	    endif
	endwhile
	Finished0:
    endif
    nt = i

    print, '% r_pwi_fce_net:  #values ', nt

    time0 = { PB5:lonarr(3), t90:0.0D }

    FCE_lp = { nt:0L, title:'Polar FCE', ytitle:'Frequency <Hz>', $
		fce:fltarr(nt), time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time is in SOD:

    t90str = string (smonth, '/', sday, '/', syear, ' 00:00:00', $
		format = '(i2.2, a1, i2.2, a1, i4.4, a9)')

    t90base = convert_timetoseconds(t90str, /t90)

    for j = 0L, i - 1 do begin
	FCE_lp.time(j).PB5(0) = syear
	FCE_lp.time(j).PB5(1) = sdoy
	FCE_lp.time(j).PB5(2) = PWI_time(j)
;change time formats to t90
	FCE_lp.time(j).t90 = PWI_time(j) + t90base.value
    endfor

    FCE_lp.nt = nt

    print,'% r_pwi_fce_net: converting time to T90'

    for j = 0L, nt - 1 do $
	FCE_lp.fce(j) = FCE(j)

    if (get_err_no EQ 0) then $
	get_err_msg='pwi_fce read successfully'

    print, '% r_pwi_fce_net: ', get_err_msg
    return

    bad_data0:	get_err_no = 1
		get_err_msg = '*NO Network data returned'

    bad_data:
	print, get_err_msg
	print, '** No FCE Data Returned'
end

;****************************************************************************
;*  NAME:        r_pwi_fce, do_local
;*
;*  DESCRIPTION: Routine to read Polar Fce data from PWI files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       do_local from draw_pwi_fce
;*
;*  OUTPUT:      none - data written to common blocks
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*		 J. Dowell
;*
;*  CHANGES: 
;****************************************************************************
 
pro r_pwi_fce, do_local

    COMMON FCEstuff, FCE_lp
    COMMON MCAEspectmstuff, MCAEspctrm
    COMMON MCABspectmstuff, MCABspctrm
    COMMON SFRAspectmstuff, SFRAspctrm
    COMMON SFRBspectmstuff, SFRBspctrm
    COMMON time ,xut1,xut2
    COMMON get_error, get_err_no, get_err_msg

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, $
				smonth, scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, $
				emonth, ecmonth, eday
    end_sod = start_sod + total_sec

    sec_per_day = 86400.0	&	get_err_no = 0

    cyear = syear       &       cmonth = smonth &       cday = sday
    if (syear mod 4 EQ 0) then  m2 = 29 $
    else        m2 = 28
    month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]

    if (do_local) then begin

	file_name = ''	&	err = 0

	PWI_path = getenv ('PWI_DATA')
	L = strlen(PWI_path)
	if (L EQ 0) then begin
	    get_err_no = 1
	    get_err_msg = 'Path to PWI data not found.  Env. variable Not set'
	    goto, bad_data
	endif else if (strmid(PWI_path, L - 1, 1) NE '/') then $
	    PWI_path = PWI_path + '/'
 
	result = findfile (PWI_PATH, count = cc)
	ccmax = cc - 1

	date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')

	if (total_sec GT 1800) then begin
            file_name = 'po_pwi_mca_' + date_string + '_v01.sv.cdf'

	    found = 0
	    for t = 0, ccmax do $
		if (result(t) EQ file_name) then	found = 1

	    if (found) then begin
	        forward_function r_pwi_mca_spec_sv_local
                r_pwi_mca_spec_sv_local
	    endif else begin

        	file_name = 'po_pwi_sfr_' + date_string + '_v01.sv.cdf'
 
		found = 0
		for t = 0, ccmax do $
		    if (result(t) EQ file_name) then	found = 1


		if (found) then begin
		    forward_function r_pwi_sfr_spec_sv_local
        	    r_pwi_sfr_spec_sv_local
		endif else begin
        	    file_name = 'po_h0_pwi_' + date_string + '_v01.cdf'
 
		    found = 0
		    for t = 0, ccmax do $
			if (result(t) EQ file_name) then	found = 1
 
		    if (found) then begin
			forward_function r_pwi_mca_spec_hr_local
			r_pwi_mca_spec_hr_local
		    endif else begin
        		file_name = 'po_h1_pwi_' + date_string + '_v01.cdf'
 
			found = 0
			for t = 0, ccmax do $
			    if (result(t) EQ file_name) then	found = 1
 
			if (found) then begin
			    forward_function r_pwi_sfr_spec_hr_local
        		    r_pwi_sfr_spec_hr_local
			endif else begin
			    get_err_no = 1
			    get_err_msg = '* No Data file found for ' + date_string
			    goto, bad_data
			endelse
		    endelse
		endelse
	    endelse
 
	endif else begin

            file_name = 'po_h0_pwi_' + date_string + '_v01.cdf'
 
	    found = 0
	    for t = 0, ccmax do $
		if (result(t) EQ file_name) then	found = 1
 
	    if (found) then begin
		forward_function r_pwi_mca_spec_hr_local
        	r_pwi_mca_spec_hr_local
	    endif else begin

        	file_name = 'po_h1_pwi_' + date_string + '_v01.cdf'
 
		found = 0
		for t = 0, ccmax do $
		    if (result(t) EQ file_name) then	found = 1
 
		if (found) then begin
		    forward_function r_pwi_sfr_spec_hr_local
        	    r_pwi_sfr_spec_hr_local
		endif else begin

		    file_name = 'po_pwi_sfr_' + date_string + '_v01.sv.cdf'
 
		    found = 0
		    for t = 0, ccmax do $
			if (result(t) EQ file_name) then	found = 1

		    if (found) then begin
			forward_function r_pwi_mca_spec_sv_local
                	r_pwi_mca_spec_sv_local
		    endif else begin

			file_name = 'po_pwi_sfr_' + date_string + '_v01.sv.cdf'
 
			found = 0
			for t = 0, ccmax do $
			    if (result(t) EQ file_name) then	found = 1


			if (found) then begin
			    forward_function r_pwi_sfr_spec_sv_local
			    r_pwi_sfr_spec_sv_local
			endif else begin
			    get_err_no = 1
			    get_err_msg = '* No Data file found for ' + date_string
			    goto, bad_data
			endelse
		    endelse
		endelse
	    endelse
        endelse

    endif else	r_pwi_fce_net
    return

bad_data:	print, get_err_msg
		print, '**No FCE data returned'
end
