
;****************************************************************************
;*  NAME:        mca_time_overflow
;*
;*  DESCRIPTION: Routine to handle array overflow
;*
;*  INPUT:       i, max_components, proc_name, err_msg, end_sod, PWI_time
;*			(array of times)
;*
;*  OUTPUT:      places data into common block and returns -1 for
;*		 fatal error, 0 for non-fatal error, 1 for no error
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
function mca_time_overflow, i, max_components, proc_name, err_msg, end_sod, PWI_time

    COMMON get_error, get_err_no, get_err_msg

    if (i GE max_components) then begin	; 1st check for room in array
	print, '% ', proc_name, ':  need more columns (i = ', i, ')'
	get_err_no = 2
	get_err_msg = err_msg
	if (end_sod - PWI_time(i - 1) LE 60.0) then begin
	    print, get_err_msg
	    return, 0
	endif else	return, -1
    endif

    return, 1
end

;****************************************************************************
;*  NAME:        r_pwi_mcab_spec_net
;*
;*  DESCRIPTION: Routine to read data_type files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       none
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_mcab_spec_net

    COMMON MCABspctrmstuff, MCABspctrm
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    max_components = 2000	&	get_err_no = 0
    PWI_spec = fltarr (max_components, 14)
    PWI_time = dblarr (max_components)

    fill_val = 1.0e-31
    ideal_components = 1000	&	MCA_Sec = 1.3248

    aPWI_offset = 0.0	&	aPWI_spec = fltarr (14)

    start_offset = 0.0	&	prev_offset = 0.0

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

; the PWI read routines all use Gnu wget to retrieve calibrated spectral
; densities via the internet, eliminating the need for files.

; the latest version of wget (as of 05/97) is available at 
;	<URL:ftp://prep.ai.mit.edu/pub/gnu/wget-1.4.3.tar.gz>.
; if wget is installed in the system path, you should not need to do
; anything special to get this procedure to work.

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    if (total_sec GT 1800) then begin
	total_spec = long(total_sec/MCA_Sec + 0.5)
	if (total_spec LT 1) then	total_spec = 1
	num_to_avg = total_spec / ideal_components
	if (num_to_avg LT 1)	then	num_to_avg = 1
	sec_interval = num_to_avg * MCA_Sec
	if (sec_interval LE MCA_Sec) then	sec_interval = MCA_Sec
	if (total_sec GT 4000) then	delta_t = 1.5 * sec_interval $
	else if (total_sec GT 2700) then	delta_t = 1.85 * sec_interval $
	else	delta_t = 2.55 * sec_interval
    endif else begin
	sec_interval = MCA_Sec	&	delta_t = 1.5 * sec_interval
    endelse

    start_offset = start_sod

    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
		sminute, ':', ssecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')
    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
		eminute, ':', esecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    command = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=papco_mcab&begin_time= ' + $
		beg_timestr + '&end_time= ' + end_timestr + '"'

    spawn, command, unit=lun

    DataOK = 1	&	i = 0L
    PWI_time(*) = 0.0

    readu, lun, aPWI_offset, aPWI_spec
    on_ioerror, bad_data0		; no data
    limit = delta_t
    fill_count = 0L

     while (DataOK) do begin

	prev_offset = aPWI_offset

	while (aPWI_offset GT limit) do begin
;	print, 'offset ', aPWI_offset, ', limit ', limit
	    if (i GT 0) then $
		PWI_time(i) = PWI_time(i - 1) + delta_t $
	    else	PWI_time(i) = start_offset
	    PWI_spec(i, *) = 0.0
	    i = i + 1
	    limit = limit + delta_t
	    fill_count = fill_count + 1
	    t = mca_time_overflow (i, max_components, 'r_pwi_mcab_spec_net', $
			'Time overflow in mcab_spec', end_sod, PWI_time)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
	
	PWI_time(i) = start_offset + aPWI_offset
	PWI_spec(i, *) = aPWI_spec
	i = i + 1	&	limit = aPWI_offset + delta_t

	t = mca_time_overflow (i, max_components, 'r_pwi_mcab_spec_net', $
			'*Time overflow in mcab_spec', end_sod, PWI_time)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
	
	readu, lun, aPWI_offset, aPWI_spec
	on_ioerror, Finished
    endwhile
    Finished: close, lun	&	free_lun, lun

    if (prev_offset LT total_sec) then begin
	prev_offset = prev_offset + delta_t
	while (prev_offset LT total_sec) do begin
	    PWI_time(i) = start_offset + prev_offset
	    PWI_spec(i, *) = 0.0
	    prev_offset = prev_offset + delta_t
	    i = i + 1

	    t = mca_time_overflow (i, max_components, 'r_pwi_mcab_spec_net', $
			'**Time overflow in mcab_spec', end_sod, PWI_time)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
	Finished0:
    endif
    nt = i	&	nf = 14L
    i = i - 1

    print, '% r_pwi_mcab_spec_net:  spec dimensions ', nt, ' x ', nf
    time0 = { PB5:lonarr(3), t90:0.0D }

    MCABspctrm = { nt:0L, nf:0L, frequency:fltarr(nf), $
		title:'PWI MCA B/Bu', ytitle:'Frequency <Hz>', $
		ztitle:'nT!e2!n/Hz', spec:fltarr(nt, nf), $
		time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time is in SOD
    t90str = string (smonth, '/', sday, '/', syear, ' 00:00:00', $
		format = '(i2.2, a1, i2.2, a1, i4.4, a9)')
    t90base = convert_timetoseconds(t90str, /t90)

    for j = 0L, i do begin
	MCABspctrm.time(j).PB5(0) = syear
	MCABspctrm.time(j).PB5(1) = sdoy
	MCABspctrm.time(j).PB5(2) = PWI_time(j)
; change time formats to t90
	MCABspctrm.time(j).t90 = PWI_time (j) + t90base.value
	MCABspctrm.spec(j, *) = PWI_spec(j, *)
    endfor

    MCABspctrm.nt = nt	&	MCABspctrm.nf = nf

    print,'% r_pwi_mcab_spec_net: converting time to T90'

    if (get_err_no EQ 0) then $
	get_err_msg = 'pwi_mcab_spec read successfully'

    print, '% r_pwi_mcab_spec_net: ', get_err_msg

    return

    bad_data0:	get_err_no = 1
		get_err_msg = '*NO Network data returned'

    bad_data:
	print, get_err_msg
	print, '**No PWI Data Returned'
end


;****************************************************************************
;*  NAME:        r_pwi_mca_spec_hr_local
;*
;*  DESCRIPTION: Routine to read MCAB survey files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       none
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_mca_spec_hr_local

    COMMON MCAEspctrmstuff, MCAEspctrm
    COMMON MCABspctrmstuff, MCABspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    total_sec = xut2 - xut1

    n = size (FCE_lp)
    if (n(0) GT 0) then begin
	mt = FCE_lp.nt - 1
	fac = 0.02 * total_sec
	if (FCE_lp.time(0).t90 LE xut1 + fac AND FCE_lp.time(mt).t90 GE $
			xut2 - fac) then	do_Fce = 0 $
	else	do_Fce = 1
    endif else	do_Fce = 1

    max_components = 2000

    PWI_mcae = fltarr(max_components, 20)
    PWI_mcab = fltarr(max_components, 14)
    aPWI_time = dblarr(max_components)
    bPWI_time = dblarr(max_components)

    if (do_Fce) then begin
	FCE = fltarr(max_components)
	Fce_time = dblarr(max_components)
    endif

    PWI_path = getenv ('PWI_DATA')
    L = strlen(PWI_path)
    if (L EQ 0) then begin
	get_err_no = 1
	get_err_msg = '**PWI Data Path NOT Found.  Env. Var. not set'
	goto, bad_data
    endif
    if (strmid(PWI_path, L - 1, 1) NE '/') then PWI_path = PWI_path + '/'

    ideal_components = 1000	&	fill_val = 1.0e-31
    MCA_Sec = 1.3248	&	SV_Sec = 36.8

    aPWI_spec = fltarr (20)	&	bPWI_spec = fltarr (14)

    taPWI_spec = fltarr (20)	&	tbPWI_spec = fltarr (14)
    aspec_sum = fltarr (20)	&	bspec_sum = fltarr(14)

    PB5 = lonarr(3)	&	file_name = ''
    AAntenna = 0L	&	BAntenna = 0L

    get_err_no = 0	&	prev_offset = 0.0

    AAnt_list = [ 'Eu', 'Ev', 'Ez', 'L ' ]
    BAnt_list = [ 'Eu', 'Bu', 'Bv', 'Bz' ]
    sec_per_day = 86400.0

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, $
				smonth, scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    end_sod = start_sod + total_sec

    start_offset = 0.0		&	FCE_in = 0.0

    if (total_sec GT 1800.0) then begin
	total_spec = long(total_sec/MCA_Sec + 0.5)
	if (total_spec LT 1) then	total_spec = 1
	num_to_avg = long(float(total_spec)/ideal_components)
	if (num_to_avg LT 1) then	num_to_avg = 1
	sec_interval = num_to_avg*MCA_Sec
	if (sec_interval LT MCA_Sec) then	sec_interval = MCA_Sec
    endif else	sec_interval = MCA_sec

    delta_t = 1.1 * sec_interval

    PB5_name = 'Time_PB5'
    MCAE_name = 'MCAE_Elec'
    MCAB_name = 'MCAB_Mag'
    AAnt_name = 'MCAE_Antenna'
    BAnt_name = 'MCAB_Antenna'
    FCE_name = 'Fce'

    cyear = syear	&	cmonth = smonth
    cday = sday
    if (syear mod 4 EQ 0) then m2 = 29 $
    else	m2 = 29
    month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]

    datestring = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
    file_name = PWI_path + 'po_h0_pwi_' + datestring +  '_v01'
    print, file_name
    CDF_ID = CDF_open (file_name)

    if (CDF_ID LT 0) then begin
	get_err_no = 1
	get_err_msg = '*Unable to Open ' + file_name + '.  File not found'
	goto, bad_data
    endif

    CDF_Result = CDF_inquire(CDF_ID)
    MaxRec = CDF_Result.MaxRec + 1

    VarNum_PB5 = CDF_varnum (CDF_ID, PB5_name)
    VarNum_MCAE = CDF_varnum (CDF_ID, MCAE_name, IsZVar)
    VarNum_MCAB = CDF_varnum (CDF_ID, MCAB_name, IsZVar)
    VarNum_AAnt = CDF_varnum (CDF_ID, AAnt_name)
    VarNum_BAnt = CDF_varnum (CDF_ID, BAnt_name)
    if (do_Fce) then	VarNum_FCE = CDF_varnum (CDF_ID, FCE_name)

    RecNum = 0L

    aPWI_time(*) = 0.0	&	bPWI_time(*) = 0.0
    aspec_sum(*) = 0.0	&	bspec_sum(*) = 0.0
    aspec_count = 0L	&	bspec_count = 0L
    aoff_sum = 0.0	&	boff_sum = 0.0
    Fce_sum = 0.0	&	Fce_count = 0L
    FCE_in = 0.0	&	foff_sum = 0.0
    Done = 0	&	i = 0L

    limit = start_sod + delta_t

    while (NOT Done) do begin
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start=RecNum
	toffset = start_offset + double(PB5(2))/1000.0
	if (toffset GE start_sod) then begin
	    Done = 1
	    CDF_varget, CDF_ID, VarNum_AAnt, AAntenna, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, BAntenna, Rec_Start=RecNum
	    if (do_Fce) then $
		CDF_varget, CDF_ID, VarNum_FCE, FCE_in, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAE, aPWI_spec, /ZVARIABLE, $
					Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAB, bPWI_spec, /ZVARIABLE, $
					Rec_Start=RecNum
	endif else begin
	    prev_aant = AAntenna(0)	&	prev_bant = BAntenna(0)
	    prev_offset = toffset
	endelse
	RecNum = RecNum + 1
    endwhile

    Done = 0
    while (NOT Done) do begin
	if (toffset GT limit) then begin
	    if (aspec_count GT 0) then begin
		aPWI_time(i) = aoff_sum/aspec_count
		taPWI_spec = aspec_sum / aspec_count
		PWI_mcae(i, *) = taPWI_spec
		aspec_sum(*) = 0.0
		aoff_sum = 0.0	&	aspec_count = 0	
	    endif else begin
		PWI_mcae(i, *) = 0.0
		if (i GT 0) then $
		    aPWI_time(i) = aPWI_time(i - 1) + delta_t $
		else	aPWI_time(i) = start_sod
	    endelse
	    if (bspec_count GT 0) then begin
		bPWI_time(i) = boff_sum / bspec_count + MCA_Sec
		tbPWI_spec = bspec_sum / bspec_count
		PWI_mcab(i, *) = tbPWI_spec
		bspec_sum(*) = 0.0
		boff_sum = 0.0	&	bspec_count = 0	
	    endif else begin
		PWI_mcab(i, *) = 0.0
		if (i GT 0) then $
		    bPWI_time(i) = bPWI_time(i - 1) + delta_t $
		else	bPWI_time(i) = start_sod
	    endelse
	    if (do_Fce) then begin
		if (Fce_count GT 0) then begin
		    Fce_time(i) = foff_sum / Fce_count
		    FCE(i) = Fce_sum / Fce_count
		    Fce_sum = 0.0	&	Fce_count = 0	
		endif else begin
		    if (i GT 0) then	Fce_time(i) = Fce_time(i - 1) + delta_t $
		    else	Fce_time(i) = start_sod
		    FCE(i) = -1.0
		endelse
	    endif
	    i = i + 1	&	limit = limit + delta_t
	    if (limit GT end_sod) then	goto, Finished
	    t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_hr_local', $
			'*Time overflow in mca_spec', end_sod, aPWI_time)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	    while (toffset GT limit) do begin
;		print, '(i, limit, toffset)', i, limit, toffset
		aPWI_time(i) = aPWI_time(i - 1) + delta_t
		bPWI_time(i) = aPWI_time(i) + MCA_Sec
		PWI_mcae(i, *) = 0.0	&	PWI_mcab(i, *) = 0.0
		i = i + 1
		t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_hr_local', $
			'**Time overflow in mca_spec', end_sod, aPWI_time)
		if (t LT 0) then	goto, bad_data $
		else if (t LT 1) then	goto, Finished0
		aspec_sum(*) = 0.0	&	bspec_sum(*) = 0.0
		aoff_sum = 0.0	&	aspec_count = 0
		boff_sum = 0.0	&	bspec_count = 0
		limit = limit + delta_t
		if (limit GT end_sod) then	goto, Finished
	    endwhile
	    if (toffset GT end_sod) then goto, Finished
	endif
	if (toffset + sec_interval GT end_sod) then	goto, Finished
	if (BAntenna(0) EQ 1) then begin
	    bspec_sum = bspec_sum + bPWI_spec(*)
	    bspec_count = bspec_count + 1
	    boff_sum = boff_sum + toffset
	endif
	if (AAntenna(0) EQ 0) then begin
	    aspec_sum = aspec_sum + aPWI_spec(*)
	    aspec_count = aspec_count + 1
	    aoff_sum = aoff_sum + toffset
	endif
	if (do_Fce) then begin
	    Fce_sum = Fce_sum + FCE_in
	    foff_sum = foff_sum + toffset
	    Fce_count = Fce_count + 1
	endif
	prev_offset = toffset
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start=RecNum
	toffset = start_offset + double(PB5(2))/1000.0
	if (toffset GT end_sod) then	Done = 1 $
	else begin
	    prev_aant = AAntenna(0)
	    prev_bant = BAntenna(0)
	    CDF_varget, CDF_ID, VarNum_AAnt, AAntenna, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, BAntenna, Rec_Start=RecNum
	    if (do_Fce) then $
		CDF_varget, CDF_ID, VarNum_FCE, FCE_in, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAE, aPWI_spec, /ZVARIABLE, $
					Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAB, bPWI_spec, /ZVARIABLE, $
					Rec_Start=RecNum
	    RecNum = RecNum + 1
	    if (RecNum GE MaxRec) then begin
		if (end_sod EQ start_offset + sec_per_day) then goto, Finished
		CDF_close, CDF_ID

		cday = cday + 1
		if (cday GT month_tab(cmonth - 1)) then begin
		    cday = 1	&	cmonth = cmonth + 1
		    if (cmonth GT 12) then begin
			cmonth = 1	&	cyear = cyear + 1
		    endif
		endif
		datestring = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
		file_name = PWI_path + 'po_h0_pwi_' + datestring +  '_v01'
		CDF_ID = CDF_open (file_name)

		if (CDF_ID LT 0) then begin
		    get_err_no = 3
		    get_err_msg = '*Unable to Open ' + file_name + '.  File not found'
		    if (end_sod - aPWI_time(i - 1) LE 60.0) then begin
			print, get_err_msg
			goto, Finished0
		    endif else	goto, bad_data
		endif

		CDF_Result = CDF_inquire(CDF_ID)
		MaxRec = CDF_Result.MaxRec + 1
		RecNum = 0L
		start_offset = start_offset + sec_per_day
	    endif
	endelse
    endwhile

    Finished:	CDF_close, CDF_ID

    if (aspec_count GT 0) then begin
	t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_hr_local', $
			'***Time overflow in mca_spec', end_sod, aPWI_time)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
	aPWI_time(i) = start_offset + (aoff_sum/aspec_count)
	taPWI_spec = aspec_sum / aspec_count
	PWI_mcae(i, *) = taPWI_spec
    endif else begin
	if (bspec_count GT 0) then begin
	    aPWI_time(i) = aPWI_time(i) + delta_t
	    PWI_mcae(i, *) = 0.0
	endif
    endelse

    if (do_Fce) then begin
	if (Fce_count GT 0) then begin
	    FCE(i) = Fce_sum/Fce_count
	    Fce_time(i) = start_offset + foff_sum/Fce_count
	endif else begin
	    FCE(i) = -1.0
	    Fce_time(i) = Fce_time(i - 1) + delta_t
	endelse
    endif

    if (bspec_count GT 0) then begin
	t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_hr_local', $
			'****Time overflow in mca_spec', end_sod, bPWI_time)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
	bPWI_time(i) = start_offset + (boff_sum/bspec_count)
	tbPWI_spec = bspec_sum / bspec_count
	PWI_mcab(i, *) = tbPWI_spec
	i = i + 1
    endif else begin
	if (aspec_count GT 0) then begin
	    bPWI_time(i) = bPWI_time(i - 1) + delta_t
	    PWI_mcab(i, *) = 0.0
	    i = i + 1
	endif
    endelse

    if (end_sod - toffset GT delta_t) then begin
	toffset = toffset + delta_t
	while (end_sod - toffset GT delta_t) do begin
	    aPWI_time(i) = toffset
	    bPWI_time(i) = toffset + MCA_Sec
	    PWI_mcae(i, *) = 0.0	&	PWI_mcab(i, *) = 0.0
	    if (do_Fce) then begin
		FCE(i) = -1.0
		Fce_time(i) = toffset
		print, 'toffset', toffset
	    endif
	    i = i + 1
	    t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_hr_local', $
			'*****Time overflow in mca_spec', end_sod, aPWI_time)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
    endif

    Finished0:	nt = i	&	nfb = 14L	&	nfa = 20L
    end_indx = nt - 1
    if (do_Fce) then $
	print, end_indx, ') Last Fce', aPWI_time(end_indx), FCE(end_indx)

    print, '% r_pwi_mca_spec_hr_local:  spec dimensions ', nt, ' x ', nfb

    time0 = { PB5:lonarr(3), t90:0.0D }

    MCAEspctrm = { nt:0L, nf:0L, title:'PWI MCA E/Eu', $
		frequency:fltarr(nfa), ytitle:'Frequency <Hz>', $
		ztitle:'(V/m)!e2!n/Hz', spec:fltarr(nt, nfa), $
		time:replicate(time0, nt) }

    MCABspctrm = { nt:0L, nf:0L, title:'PWI MCA B/Bu', $
		frequency:fltarr(nfb), ytitle:'Frequency <Hz>', $
		ztitle:'nT!e2!n/Hz', spec:fltarr(nt, nfb), $
		time:replicate(time0, nt) }

    print,'% r_pwi_mca_spec_hr_local: converting time to T90'

; use 00:00:00 here because PWI_time is in SOD:
    t90str = string (smonth, '/', sday, '/', syear, ' 00:00:00', $
		format = '(i2.2, a1, i2.2, a1, i4.4, a9)')

    t90base = convert_timetoseconds(t90str, /t90)

    max = nt - 1

    for j = 0L, max do begin
	MCAEspctrm.time(j).PB5(0) = syear
	MCABspctrm.time(j).PB5(0) = syear
	MCAEspctrm.time(j).PB5(1) = sdoy
	MCABspctrm.time(j).PB5(1) = sdoy
	MCAEspctrm.time(j).PB5(2) = aPWI_time(j)
	MCABspctrm.time(j).PB5(2) = bPWI_time(j)
;change time formats to t90
	MCAEspctrm.time(j).t90 = aPWI_time(j) + t90base.value
	MCABspctrm.time(j).t90 = bPWI_time(j) + t90base.value
	MCAEspctrm.spec(j, *) = PWI_mcae(j, *)
	MCABspctrm.spec(j, *) = PWI_mcab(j, *)
    endfor

    MCAEspctrm.nt = nt	&	MCAEspctrm.nf = nfa
    MCABspctrm.nt = nt	&	MCABspctrm.nf = nfb
    if (do_Fce) then begin

	index = where (FCE(0:end_indx) GT 0.0, kount)
	nt_f = kount

	FCE_lp = { nt:0L, title:'Polar Fce', ytitle:'Frequency <Hz>', $
		fce:fltarr(nt_f), time:replicate(time0, nt_f) }
	FCE_lp.nt = nt_f	&	FCE_lp.fce = FCE(index)
	FCE_lp.time(*).PB5(0) = syear
	FCE_lp.time(*).PB5(1) = sdoy
	FCE_lp.time(*).PB5(2) = aPWI_time(index)
	FCE_lp.time(*).T90 = aPWI_time(index) + t90base.value
	print, 'Last times (FEB):', FCE_lp.time(nt_f - 1).T90, $
		MCAEspctrm.time(end_indx).T90, MCABspctrm.time(end_indx).T90
	print, FCE_lp.time(nt_f - 2).T90
    endif

    if (get_err_no EQ 0) then $
	get_err_msg='pwi_mca_spec read successfully'

    print, '% r_pwi_mca_spec_hr_local: ', get_err_msg

    return

    bad_data:
	print, get_err_msg
	print, '** No PWI Data Returned'

end

;****************************************************************************
;*  NAME:        r_pwi_mca_spec_sv_local
;*
;*  DESCRIPTION: Routine to read MCAB survey files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       none (uses common block time from PAPCO)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_mca_spec_sv_local

    COMMON MCAEspctrmstuff, MCAEspctrm
    COMMON MCABspctrmstuff, MCABspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time ,xut1,xut2
    COMMON get_error, get_err_no, get_err_msg

    max_components = 2000
    PWI_mcae = fltarr(max_components, 20)
    PWI_mcab = fltarr(max_components, 14)
    aPWI_time = dblarr(max_components)
    bPWI_time = dblarr(max_components)

    total_sec = xut2 - xut1

    n = size (FCE_lp)
    if (n(0) GT 0) then begin
	mt = FCE_lp.nt - 1
	fac = 0.02 * total_sec
	if (FCE_lp.time(0).t90 LE xut1 + fac AND FCE_lp.time(mt).t90 GE $
			xut2 - fac) then	do_Fce = 0 $
; We could have gotten the Fce elsewhere (e.g., from SFR data)
	else	do_Fce = 1
    endif elsE	do_Fce = 1

    if (do_Fce) then begin
	FCE = fltarr(max_components)
	Fce_time = dblarr(max_components)
    endif

    PWI_path = getenv ('PWI_DATA')
    L = strlen(PWI_path)
    if (L EQ 0) then begin
	get_err_no = 1
	get_err_msg = '**PWI data path NOT found.  Env. Var. not set'
	goto, bad_data
    endif
    if (strmid(PWI_path, L - 1, 1) NE '/') then PWI_path = PWI_path + '/'

    ideal_components = 1000	&	fill_val = 1.0e-31
    MCA_Sec = 1.3248	&	SV_Sec = 36.8

    aPWI_spec = fltarr (20)	&	bPWI_spec = fltarr (14)
    taPWI_spec = fltarr (20)	&	tbPWI_spec = fltarr (14)

    aspec_sum = fltarr (20)	&	bspec_sum = fltarr(14)
    file_name = ''		&	PB5 = lonarr(3)
    AAntenna = 0L	&	BAntenna = 0L

    get_err_no = 0	&	prev_offset = 0.0

    AAnt_list = [ 'Eu', 'Ev', 'Ez', 'L ' ]
    BAnt_list = [ 'Eu', 'Bu', 'Bv', 'Bz' ]
    sec_per_day = 86400.0

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, $
				smonth, scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    end_sod = start_sod + total_sec

    total_spec = long(total_sec / SV_Sec + 0.5)
    if (total_spec LT 1) then	total_spec = 1
    num_to_avg = fix(float(total_spec)/float(ideal_components) + 0.5)
    if (num_to_avg LT 1) then	num_to_avg = 1
    sec_interval = num_to_avg * SV_Sec
    if (sec_interval LE SV_Sec) then	sec_interval = SV_Sec

    delta_t = 1.1 * sec_interval

    start_offset = 0.0

    PB5_name = 'Time_PB5'
    MCAE_name = 'MCAE_Elec'
    MCAB_name = 'MCAB_Mag'
    AAnt_name = 'MCAE_Antenna'
    BAnt_name = 'MCAB_Antenna'
    FCE_name = 'Fce'

    cyear = syear	&	cmonth = smonth
    cday = sday
    if (syear mod 4 EQ 0) then m2 = 29 $
    else	m2 = 29
    month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]

    datestring = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
    file_name = PWI_path + 'po_pwi_mca_' + datestring +  '_v01.sv'
    print, file_name
    CDF_ID = CDF_open (file_name)

    if (CDF_ID LT 0) then begin
	get_err_no = 1
	get_err_msg = '*Unable to Open ' + file_name + '.  File not found'
	goto, bad_data
    endif

    CDF_Result = CDF_inquire(CDF_ID)
    MaxRec = CDF_Result.MaxRec + 1

    VarNum_PB5 = CDF_varnum (CDF_ID, PB5_name)
    VarNum_MCAE = CDF_varnum (CDF_ID, MCAE_name, IsZVar)
    VarNum_MCAB = CDF_varnum (CDF_ID, MCAB_name, IsZVar)
    VarNum_AAnt = CDF_varnum (CDF_ID, AAnt_name)
    VarNum_BAnt = CDF_varnum (CDF_ID, BAnt_name)
    VarNum_FCE = CDF_varnum (CDF_ID, FCE_name)

    RecNum = 0L

    aPWI_time(*) = 0.0	&	bPWI_time(*) = 0.0
    aspec_sum(*) = 0.0	&	bspec_sum(*) = 0.0
    aspec_count = 0L	&	bspec_count = 0L
    aoff_sum = 0.0	&	boff_sum = 0.0
    Fce_sum = 0.0	&	Fce_count = 0L
    FCE_in = 0.0	&	foff_sum = 0.0
    Done = 0	&	i = 0L

    while (NOT Done) do begin
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start=RecNum
	toffset = start_offset + double(PB5(2))/1000.0
	if (toffset GE start_SOD) then begin
	    Done = 1
	    CDF_varget, CDF_ID, VarNum_AAnt, AAntenna, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, BAntenna, Rec_Start=RecNum
	    if (do_Fce) then $
		CDF_varget, CDF_ID, VarNum_FCE, FCE_in, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAE, aPWI_spec, /ZVARIABLE, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAB, bPWI_spec, /ZVARIABLE, Rec_Start=RecNum
	endif else begin
	    prev_aant = AAntenna(0)	&	prev_bant = BAntenna(0)
	    prev_offset = toffset
	endelse
	RecNum = RecNum + 1
    endwhile

    Done = 0	&	limit = start_sod + sec_interval
    while (NOT Done) do begin
	if (toffset GE limit) then begin
	    if (aspec_count GT 0) then begin
		aPWI_time(i) = aoff_sum/aspec_count
		taPWI_spec = aspec_sum / aspec_count
		PWI_mcae(i, *) = taPWI_spec
		aspec_sum(*) = 0.0
		aoff_sum = 0.0	&	aspec_count = 0	
	    endif else begin
		PWI_mcae(i, *) = 0.0
		if (i GT 0) then $
		    aPWI_time(i) = aPWI_time(i - 1) + delta_t $
		else	aPWI_time(i) = start_sod
	    endelse
	    if (bspec_count GT 0) then begin
		bPWI_time(i) = boff_sum / bspec_count + MCA_Sec
		tbPWI_spec = bspec_sum / bspec_count
		PWI_mcab(i, *) = tbPWI_spec
		bspec_sum(*) = 0.0
		boff_sum = 0.0	&	bspec_count = 0	
	    endif else begin
		PWI_mcab(i, *) = 0.0
		if (i GT 0) then $
		    bPWI_time(i) = bPWI_time(i - 1) + delta_t $
		else	bPWI_time(i) = start_sod
	    endelse
	    if (do_Fce) then begin
		if (Fce_count GT 0) then begin
		    Fce_time(i) = foff_sum / Fce_count
		    FCE(i) = Fce_sum / Fce_count
		    Fce_sum(*) = 0.0	&	Fce_count = 0	
		endif else begin
		    if (i GT 0) then	Fce_time(i) = Fce_time(i - 1) + delta_t $
		    else	Fce_time(i) = start_sod
		    FCE(i) = -1.0
		endelse
	    endif
	    i = i + 1
	    t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_sv_local', $
			'Time overflow in mca_spec', end_sod, aPWI_time)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	    limit = aPWI_time(i - 1) + delta_t
	    while (toffset GT limit) do begin
;		print, 'sv: (i, limit, toffset)', i, limit, toffset
		aPWI_time(i) = aPWI_time(i - 1) + delta_t
		bPWI_time(i) = aPWI_time(i) + MCA_Sec
		PWI_mcae(i, *) = 0.0	&	PWI_mcab(i, *) = 0.0
		i = i + 1
		t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_sv_local', $
			'*Time overflow in mca_spec', end_sod, aPWI_time)
		if (t LT 0) then	goto, bad_data $
		else if (t LT 1) then	goto, Finished0
		aspec_sum(*) = 0.0	&	bspec_sum(*) = 0.0
		aoff_sum = 0.0	&	aspec_count = 0
		boff_sum = 0.0	&	bspec_count = 0
		limit = limit + delta_t
		if (limit GT end_sod) then	goto, Finished
	    endwhile
	    if (toffset GT end_sod) then goto, Finished
	endif
	if (BAntenna(0) EQ 1) then begin
	    bspec_sum = bspec_sum + bPWI_spec(*)
	    bspec_count = bspec_count + 1
	    boff_sum = boff_sum + toffset
	endif
	if (AAntenna(0) EQ 0) then begin
	    aspec_sum = aspec_sum + aPWI_spec(*)
	    aspec_count = aspec_count + 1
	    aoff_sum = aoff_sum + toffset
	endif
	if (do_Fce) then begin
	    Fce_sum = Fce_sum + FCE_in
	    Fce_count = Fce_count + 1
	    foff_sum = foff_sum + toffset
	endif
	prev_offset = toffset
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start=RecNum
	toffset = start_offset + double(PB5(2))/1000.0
	if (toffset GT end_SOD) then	Done = 1 $
	else begin
	    prev_aant = AAntenna(0)
	    prev_bant = BAntenna(0)
	    CDF_varget, CDF_ID, VarNum_AAnt, AAntenna, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, BAntenna, Rec_Start=RecNum
	    if (do_Fce) then $
		CDF_varget, CDF_ID, VarNum_FCE, FCE_in, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAE, aPWI_spec, /ZVARIABLE, Rec_Start=RecNum
	    CDF_varget, CDF_ID, VarNum_MCAB, bPWI_spec, /ZVARIABLE, Rec_Start=RecNum
	    RecNum = RecNum + 1
	    if (RecNum GE MaxRec) then begin
		if (end_sod EQ start_offset + sec_per_day) then goto, Finished
		CDF_close, CDF_ID

		cday = cday + 1
		if (cday GT month_tab(cmonth - 1)) then begin
		    cday = 1	&	cmonth = cmonth + 1
		    if (cmonth GT 12) then begin
			cmonth = 1	&	cyear = cyear + 1
		    endif
		endif

		datestring = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
		file_name = PWI_path + 'po_pwi_mca_' + datestring + '_v01.sv'
		print, file_name
		CDF_ID = CDF_open (file_name)

		if (CDF_ID LT 0) then begin
		    get_err_no = 3
		    get_err_msg = '*Unable to Open ' + file_name + '.  File not found'
		    if (end_sod - aPWI_time(i - 1) LE 60.0) then begin
			print, get_err_msg
			goto, Finished0
		    endif else	goto, bad_data
		endif

		CDF_Result = CDF_inquire(CDF_ID)
		MaxRec = CDF_Result.MaxRec + 1
		RecNum = 0L
		start_offset = start_offset + sec_per_day
	    endif
	endelse
    endwhile

    Finished:	CDF_close, CDF_ID

    if (aspec_count GT 0) then begin
	t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_sv_local', $
			'**Time overflow in mca_spec', end_sod, aPWI_time)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
	aPWI_time(i) = start_offset + (aoff_sum/aspec_count)
	taPWI_spec = aspec_sum / aspec_count
	PWI_mcae(i, *) = taPWI_spec
    endif else begin
	if (bspec_count GT 0) then begin
	    aPWI_time(i) = aPWI_time(i) + delta_t
	    PWI_mcae(i, *) = 0.0
	endif
    endelse

    if (do_Fce) then begin
	if (Fce_count GT 0) then begin
	    FCE(i) = Fce_sum/Fce_count
	    Fce_time(i) = start_offset + foff_sum/Fce_count
	endif else begin
	    FCE(i) = 0.0
	    Fce_time(i) = Fce_time(i - 1) + delta_t
	endelse
    endif

    if (bspec_count GT 0) then begin
	t = mca_time_overflow (i, max_components, 'r_pwi_mca_spec_sv_local', $
			'***Time overflow in mca_spec', end_sod, bPWI_time)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
	bPWI_time(i) = start_offset + (boff_sum/bspec_count)
	tbPWI_spec = bspec_sum / bspec_count
	PWI_mcab(i, *) = tbPWI_spec
	i = i + 1
    endif else begin
	if (aspec_count GT 0) then begin
	    bPWI_time(i) = bPWI_time(i - 1) + delta_t
	    PWI_mcab(i, *) = 0.0
	    i = i + 1
	endif
    endelse
    Finished0:	nt = i	&	nf_b = 14L	&	nf_a = 20L
    end_indx = nt - 1
    if (do_Fce) then $
	print, 'Last Fce', Fce_time(end_indx), FCE(end_indx)

    print, '% r_pwi_mca_spec_sv_local:  spec dimensions ', nt, ' x ', nf_b

    time0 = { PB5:lonarr(3), t90:0.0D }

    MCABspctrm = { nt:0L, nf:0L, title:'PWI MCA B/Bu', $
		frequency:fltarr(nf_b), ytitle:'Frequency <Hz>', $
		ztitle:'nT!e2!n/Hz', spec:fltarr(nt, nf_b), $
		time:replicate(time0, nt) }

    MCAEspctrm = { nt:0L, nf:0L, title:'PWI MCA E/Eu', $
		frequency:fltarr(nf_a), ytitle:'Frequency <Hz>', $
		ztitle:'(V/m)!e2!n/Hz', spec:fltarr(nt, nf_a), $
		time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time is in SOD:
    print,'% r_pwi_mca_spec_sv_local: converting time to T90'

    t90str = string (smonth, '/', sday, '/', syear, ' 00:00:00', $
		format = '(i2.2, a1, i2.2, a1, i4.4, a9)')

    t90base = convert_timetoseconds(t90str, /t90)

    max = nt - 1

    for j = 0L, max do begin
	MCABspctrm.time(j).PB5(0) = syear
	MCAEspctrm.time(j).PB5(0) = syear
	MCABspctrm.time(j).PB5(1) = sdoy
	MCAEspctrm.time(j).PB5(1) = sdoy
	MCABspctrm.time(j).PB5(2) = bPWI_time(j)
	MCAEspctrm.time(j).PB5(2) = aPWI_time(j)
;change time formats to t90
	MCABspctrm.time(j).t90 = bPWI_time(j) + t90base.value
	MCAEspctrm.time(j).t90 = aPWI_time(j) + t90base.value
	MCABspctrm.spec(j, *) = PWI_mcab(j, *)
	MCAEspctrm.spec(j, *) = PWI_mcae(j, *)
    endfor

    MCABspctrm.nt = nt	&	MCABspctrm.nf = nf_b
    MCAEspctrm.nt = nt	&	MCAEspctrm.nf = nf_a

    if (do_Fce) then begin

        index = where (FCE(0:max) GT 0.0, nt_f)

	FCE_lp = { nt:0L, title:'Polar Fce', ytitle:'Frequency <Hz>', $
		fce:fltarr(nt_f), time:replicate(time0, nt_f) }

	FCE_lp.nt = nt_f	&	FCE_lp.fce = FCE(index)
	FCE_lp.time(*).PB5(0) = syear
	FCE_lp.time(*).PB5(1) = sdoy
	FCE_lp.time(*).PB5(2) = aPWI_time(index)
	FCE_lp.time(*).T90 = aPWI_time(index) + t90base.value
    endif

    if (get_err_no EQ 0) then $
	get_err_msg='pwi_mca_spec read successfully'

    print, '% r_pwi_mca_spec_sv_local: ', get_err_msg
    return

    bad_data:
	print, get_err_msg
	print, '** No PWI Data Returned'

end

;****************************************************************************
;*  NAME:        r_pwi_mcab_spec, do_local
;*
;*  DESCRIPTION: Routine to read data_type files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       do_local from draw_pwi_mcab_spec
;*
;*  OUTPUT:      none
;*
;*  DATE:        June 1996
;*
;*  AUTHOR:       R. Friedel
;*		 J. Dowell
;*
;*  CHANGES: 
;****************************************************************************

pro r_pwi_mcab_spec, do_local

    COMMON MCAEspctrmstuff, MCAEspctrm
    COMMON MCABspctrmstuff, MCABspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    sec_per_day = 86400.0

    if (do_local) then begin
        cyear = syear       &       cmonth = smonth &       cday = sday
        if (syear mod 4 EQ 0) then  m2 = 29 $
        else        m2 = 28
        month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]

        file_name = ''

	PWI_path = getenv ('PWI_DATA')
	L = strlen(PWI_path)
	if (L EQ 0) then begin
	    get_err_no = 1
	    get_err_msg = 'Path to PWI data not found.  Env. variable Not set'
	    goto, bad_data
	endif else if (strmid(PWI_path, L - 1, 1) NE '/') then $
	    PWI_path = PWI_path + '/'
	date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
	no_survey = 0
 
	result = findfile(PWI_PATH, count = cc)
	ccmax = cc - 1

	if (total_sec GT 1800) then begin

	    file_name = 'po_pwi_mca_' + date_string + '_v01.sv.cdf'
	    found = 0
	    for i = 0, ccmax do $
		if (file_name EQ result(i)) then	found = 1

	    if (NOT found) then begin
		no_survey = 1
		goto, do_hires
	    endif
	    r_pwi_mca_spec_sv_local
	endif else begin

	do_hires:

	    file_name = 'po_h0_pwi_' + date_string + '_v01.cdf'
	    ccmax = cc - 1
	    found = 0
	    for i = 0, ccmax do $
		if (file_name EQ result(i)) then	found = 1
 
	    if (NOT found) then	goto, do_survey

            r_pwi_mca_spec_hr_local

            return
 
do_survey: if (no_survey) then begin
		get_err_msg = '** no local data found for ' + date_string
		goto, bad_data
	   endif
	   print, '*Unable to open ' + file_name + '.  File not found.'
           print, 'Calling survey reader'
           r_pwi_mca_spec_sv_local
        endelse

    endif else	r_pwi_mcab_spec_net

    return

bad_data:	print, get_err_msg
		print, '**No PWI data returned'
end

