;****************************************************************************
;*  NAME:        r_pwi_mcae_spec_net
;*
;*  DESCRIPTION: Routine to read data_type files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       none
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_mcae_spec_net

    COMMON MCAEspctrmstuff, MCAEspctrm
    COMMON time ,xut1,xut2
    COMMON get_error, get_err_no, get_err_msg

    max_components = 2000
    PWI_spec = fltarr (max_components, 20)
    PWI_time = dblarr (max_components)

    ideal_components = 1000	&	fill_val = 1.0e-32
    MCA_Sec = 1.3248	&	prev_offset = 0.0

    aPWI_offset = 0.0	&	aPWI_spec = fltarr (20)

    get_err_no = 0

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

; the PWI read routines all use Gnu wget to retrieve calibrated spectral
; densities via the internet, eliminating the need for files.

; the latest version of wget (as of 05/97) is available at
;       <URL:ftp://prep.ai.mit.edu/pub/gnu/wget-1.4.3.tar.gz>.
; if wget is installed in the system path, you should not need to do
; anything special to get this procedure to work.

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
				scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
				ecmonth, eday
    end_sod = start_sod + total_sec

    if (total_sec GT 1800) then begin
	total_spec = long(total_sec/MCA_Sec + 0.5)
	if (total_spec LT 1) then	total_spec = 1
	num_to_avg = total_spec / ideal_components
	if (num_to_avg LT 1) then	num_to_avg = 1
	sec_interval = MCA_Sec*num_to_avg
	if (sec_interval LT MCA_Sec) then	sec_interval = MCA_Sec
	if (total_sec GT 4000) then	delta_t = sec_interval*1.5 $
	else if (total_sec GT 2700) then	delta_t = sec_interval*1.85 $
	else	delta_t = sec_interval*2.55

    endif else begin
	sec_interval = MCA_Sec	&	delta_t = sec_interval*1.5
	num_to_avg = 1
    endelse

    start_offset = start_sod

    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
		sminute, ':', ssecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
		eminute, ':', esecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    command = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=papco_mcae&begin_time= ' + $
		beg_timestr + '&end_time= ' + end_timestr + '"'

    spawn, command, unit = lun

    DataOK = 1	&	i = 0L
    PWI_time(*, *) = 0.0

    readu, lun, aPwi_offset, aPWI_spec
    on_ioerror, bad_data0

    limit = delta_t

    while (DataOK) do begin
	prev_offset = aPWI_offset

	while (aPWI_offset GT limit) do begin	; fill gap with 0.0
	    if (i GT 0) then $
		PWI_time(i) = PWI_time(i - 1) + delta_t $
	    else	PWI_time(i) = start_offset
	    PWI_spec(i, *) = 0.0
	    i = i + 1	&	limit = limit + sec_interval

	    t = mca_time_overflow(i, max_components, 'r_pwi_mcae_spec_net', $
			'Time overflow in mcae_spec', end_sod, PWI_time)
	    if (t LT 0) then	goto, bad_data $
	    else if (T LT 1) then	goto, Finished0
	endwhile

	PWI_time(i) = start_offset + aPWI_offset
	PWI_spec(i, *) = aPWI_spec
	i = i + 1	&	limit = aPWI_offset + delta_t

	t = mca_time_overflow(i, max_components, 'r_pwi_mcae_spec_net', $
			'*Time overflow in mcae_spec', end_sod, PWI_time)
	if (t LT 0) then	goto, bad_data $
	else if (T LT 1) then	goto, Finished0

	readu, lun, aPwi_offset, aPWI_spec
	on_ioerror, Finished
    endwhile

    Finished: close, lun	&	free_lun, lun

    if (prev_offset LT total_sec) then begin
	prev_offset = prev_offset + delta_t
	while (prev_offset LT total_sec) do begin
	    PWI_time(i) = start_offset + prev_offset
	    PWI_spec(i, *) = 0.0
	    prev_offset = prev_offset + sec_interval
	    i = i + 1

	    t = mca_time_overflow(i, max_components, 'r_pwi_mcae_spec_net', $
			'**Time overflow in mcae_spec', end_sod, PWI_time)
	    if (t LT 0) then	goto, bad_data $
	    else if (T LT 1) then	goto, Finished0
	endwhile
    endif

    Finished0:	nt = i	&	nf = 20L

    print, '% r_pwi_mcae_spec_net:  spec dimensions ', nt, ' x ', nf

    time0 = { PB5:lonarr(3), t90:0.0D }

    MCAEspctrm = { nt:0L, nf:0L, frequency:fltarr(nt), $
		title:'PWI MCA E/Eu', ytitle:'Frequency <Hz>', $
		ztitle:'(V/m)!e2!n/Hz', spec:fltarr(nt, nf), $
		time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time is in SOD:

    t90str = string (smonth, '/', sday, '/', syear, ' 00:00:00', $
		format = '(i2.2, a1, i2.2, a1, i4.4, a9)')

    t90base = convert_timetoseconds(t90str, /t90)

    for j = 0L, i - 1 do begin
	MCAEspctrm.time(j).PB5(0) = syear
	MCAEspctrm.time(j).PB5(1) = sdoy
	MCAEspctrm.time(j).PB5(2) = PWI_time(j)
;change time formats to t90
	MCAEspctrm.time(j).t90 = PWI_time(j) + t90base.value
    endfor

    MCAEspctrm.nt = nt	&	MCAEspctrm.nf = nf

    print,'% r_pwi_mcae_spec_net: converting time to T90'

    for j = 0L, nt - 1 do $
	MCAEspctrm.spec(j, *) = PWI_spec(j, *)

    if (get_err_no EQ 0) then $
	get_err_msg='pwi_mcae_spec read successfully'

    print, '% r_pwi_mcae_spec_net: ', get_err_msg
    return

    bad_data0:	get_err_no = 1
		get_err_msg = '**Error reading network data'
    bad_data:
	print, get_err_msg
	print, '** No PWI Data Returned'

end

;****************************************************************************
;*  NAME:        r_pwi_mcae_spec, do_local
;*
;*  DESCRIPTION: Routine to read MCAE survey files. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       do_local from draw_pwi_mcae_spec
;*
;*  OUTPUT:      none - writes to common blocks
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*		 J. Dowell
;*
;*  CHANGES: 
;****************************************************************************
 
pro r_pwi_mcae_spec, do_local

    COMMON MCAEspctrmstuff, MCAEspctrm
    COMMON MCABspctrmstuff, MCABspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time ,xut1,xut2
    COMMON get_error, get_err_no, get_err_msg

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, $
				smonth, scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, $
				emonth, ecmonth, eday
    end_sod = start_sod + total_sec

    sec_per_day = 86400.0

    if (do_local) then begin

	file_name = ''

	PWI_path = getenv ('PWI_DATA')
	L = strlen(PWI_path)
	if (L EQ 0) then begin
	    get_err_no = 1
	    get_err_msg = 'Path to PWI data not found.  Env. variable Not set'
	    goto, bad_data
	endif else if (strmid(PWI_path, L - 1, 1) NE '/') then $
	    PWI_path = PWI_path + '/'
	cyear = syear       &       cmonth = smonth &       cday = sday
	if (syear mod 4 EQ 0) then  m2 = 29 $
	else        m2 = 28
	month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]
	date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
	no_survey = 0
 
	result = findfile(PWI_PATH, count = cc)
	ccmax = cc - 1

	if (total_sec GT 1800) then begin
            file_name = 'po_pwi_mca_' + date_string + '_v01.sv.cdf'
	    found = 0
	    for i = 0, ccmax do $
		if (result(i) EQ file_name) then	found = 1

	    if (NOT found) then begin
		no_survey = 1
		goto, do_hires
	    endif

	    r_pwi_mca_spec_sv_local
	endif else begin
	    do_hires:
	    file_name = 'po_h0_pwi_19' + date_string + '_v01.cdf'
 
	    found = 0
	    for i = 0, ccmax do $
		if (result(i) EQ file_name) then	found = 1

	    if (NOT found) then	goto, do_survey

            r_pwi_mca_spec_hr_local
            return
 
do_survey:  if (no_survey) then begin
		get_err_msg = 'No local data found for ' + date_string
		goto, bad_data
	    endif
	    print, '*Unable to open ' + file_name + '.  File not found.'
	    print, 'Calling survey reader'
	    r_pwi_mca_spec_sv_local
        endelse

    endif else	r_pwi_mcae_spec_net
    return

bad_data:	print, get_err_msg
		print, '**No PWI data returned'
end
