
;****************************************************************************
;*  NAME:        err_check
;*
;*  DESCRIPTION: Routine to increment i and check for end of array
;*
;*  INPUT:       i, max_components, proc_name, message (error message
;*		 to be displayed), PWI_time (array of spectrum times),
;*		 end_sod
;*
;*  OUTPUT:      places data into common block and Function returns
;*		 -1 if error is fatal, 0 if not fatal, 1 if no error
;*
;*  DATE:        June 1998
;*
;*  AUTOR:       J. Dowell
;*
;*  CHANGES: 
;****************************************************************************
FUNCTION err_check, i, max_components, proc_name, message, PWI_time, end_sod

    COMMON get_error, get_err_no, get_err_msg

    if (i GE max_components) then begin
	print, '% ', proc_name, ':  need more columns (i = ', i, ')'
	get_err_no = 2
	get_err_msg = message
	if (end_sod - PWI_time(i - 1) LE 60.0) then begin
	    print, get_err_msg
	    return, 0
	endif else	return, -1
    endif
    return, 1
end

;****************************************************************************
;*  NAME:        r_pwi_sfra_spec_sv_net
;*
;*  DESCRIPTION: Routine to read SFR A survey data from the internet. Adheres
;*		 to PAPCO conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       none - gets data from common blocks
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
PRO r_pwi_sfra_spec_sv_net

   COMMON SFRAspctrmstuff, SFRAspctrm
   COMMON time, xut1, xut2
   COMMON get_error, get_err_no, get_err_msg

   max_components = 2000	&	get_err_no = 0
   PWI_spec = fltarr (max_components, 320)
   PWI_time = dblarr (max_components)

   ideal_components = 1000	&	fill_val = 1.0e-32

   spect_sec = 2.3552		&	SFR_interval = 33.0

   aPWI_offset = 0.0	&	aPWI_spec = fltarr (320)

   start_offset = 0.0	&	prev_offset = 0.0

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

; the PWI read routines all use Gnu wget to retrieve calibrated spectral
; densities via the internet, eliminating the need for files.

; the latest version of wget (as of 05/97) is available at
;       <URL:ftp://prep.ai.mit.edu/pub/gnu/wget-1.4.3.tar.gz>.
; if wget is installed in the system path, you should not need to do
; anything special to get this procedure to work.

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    total_spec = long(total_sec / spect_sec + 0.5)
    if (total_spec LT 1) then	total_spec = 1
    num_to_avg = total_spec / ideal_components
    if (num_to_avg LT 1) then	num_to_avg = 1
    sec_interval = num_to_avg * spect_sec

    if (sec_interval LT SFR_interval) then	sec_interval = SFR_interval

    sec_incr = sec_interval / 2.0
    if (sec_incr LE SFR_interval) then	sec_incr = SFR_interval

    delta_t = 1.25 * sec_interval	&	start_offset = start_sod

    mid_timestr = '1996-05-27T00:06:00'
    max_timestr = '1997-02-10T00:00:00'

    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
			sminute, ':', ssecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
			eminute, ':', esecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    command1 = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=papco_sfra&begin_time= ' + $
		beg_timestr + '&end_time= ' + end_timestr + '"'

    if (beg_timestr LT mid_timestr) then begin

	if (end_timestr LT mid_timestr) then	Ant = 0 $
	else	Ant = 2

    endif else if (beg_timestr LT max_timestr) then begin

      if (end_timestr LT max_timestr) then	Ant = 1 $
      else	Ant = 3

    endif else	Ant = 0

    print, command1
    spawn, command1, unit = lun

    PWI_time(*) = 0.0

    DataOK = 1	&	i = 0L
    readu, lun, aPWI_offset, aPWI_spec
    on_ioerror, bad_data0

    limit = delta_t	&	prev_offset = aPWI_offset

    while (DataOK) do begin

	while (aPWI_offset GT limit) do begin
	    if (i GT 0) then $
		PWI_time(i) = PWI_time(i - 1) + sec_interval $
	    else	PWI_time(i) = start_offset
	    PWI_spec(i, *) = 0.0
	    i = i + 1	&	limit = limit + sec_interval
	    t = err_check (i, max_components, 'r_pwi_sfra_spec_sv_net', $
			'Time overflow in sfra_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile

	PWI_time(i) = start_offset + aPWI_offset
	PWI_spec(i, *) = aPWI_spec
	i = i + 1	&	limit = aPWI_offset + delta_t
	t = err_check (i, max_components, 'r_pwi_sfra_spec_sv_net', $
			'*Time overflow in sfra_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0

	prev_offset = aPWI_offset
	readu, lun, aPWI_offset, aPWI_spec
	on_ioerror, Finished
    endwhile

    Finished:

    if (end_sod - PWI_time(i - 1) GT delta_t) then begin
	toffset = PWI_time(i - 1) + delta_t
	while (end_sod - toffset GT delta_t) do begin
	    PWI_time(i) = toffset
	    PWI_spec(i, *) = 0.0
	    toffset = toffset + delta_t
	    i = i + 1
	    t = err_check (i, max_components, 'r_pwi_sfra_spec_sv_net', $
			'**Time overflow in sfra_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
    endif
    Finished0:	nt = i	&	nf = 320L

    print, '% r_pwi_sfra_spec_sv_net:  spec dimensions ', nt, ' x ', nf

    time0 = { PB5:lonarr(3), t90:0.0D }

    if (Ant EQ 0) then $
	ttitle = 'PWI SFR A/Eu' $
    else if (Ant EQ 1) then $
	ttitle = 'PWI SFR A/Ez' $
    else if (Ant EQ 2) then $
	ttitle = 'PWI SFR A/Eu/Ez' $
    else if (Ant EQ 3) then $
	ttitle = 'PWI SFR A/Ez/Eu' $
    else 	ttitle = 'PWI SFR A'

    SFRAspctrm = { nt:0L, nf:nf, title:ttitle, $
		ytitle:'Frequency <Hz>', ztitle:'(V/m)!e2!n/Hz', $
		frequency:fltarr(nf), spec:fltarr(nt, nf), $
		time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time() is in SOD
    t90base = convert_timetoseconds(string (smonth, '/', sday, '/', syear, $
		' 00:00:00', format = '(i2.2, a1, i2.2, a1, i4.4, a9)'), /t90)

    max = nt - 1
    for j = 0L, max do begin
	SFRAspctrm.time(j).PB5(0) = syear
	SFRAspctrm.time(j).PB5(1) = sdoy
	SFRAspctrm.time(j).PB5(2) = PWI_time(j)
;change time formats to t90
	SFRAspctrm.time(j).t90 = PWI_time(j) + t90base.value
	SFRAspctrm.spec(j, *) = PWI_spec(j, *)
    endfor

    SFRAspctrm.nt = nt

; once you are herel, the data has been correctly read. At this point you
; may restrict the arrays passed back to the time chosen using xut1 and xut2,
; you need to convert the time information to T90 (use the routines in
; $PAPCO_BASE/PAPCO_Lib/convert.pro) - if you intend to use the PAPCO
; plotting templates for your data. If you use your own plot
; routines this step is not needed, and you also don't have to re-save the
; converted file.

    print, '% r_pwi_sfra_spec_sv_net: converted time to T90'

    if (get_err_no EQ 0) then	get_err_msg = 'pwi_sfra_spec read succesfully'
    print, '% r_pwi_sfra_spec_sv_net: ', get_err_msg

    return

    bad_data0:  get_err_no = 1
    get_err_msg = '**No network data returned'

    bad_data:
	print, get_err_msg
	print, '**No PWI data returned'
end

;****************************************************************************
;*  NAME:        r_pwi_sfra_spec_hr_net
;*
;*  DESCRIPTION: Routine to read high resolution SFR A data over the internet.
;*		 Adheres to PAPCO conventions: Error common block, changes
;*		 time format to T90
;*
;*  INPUT:       none - gets data from common blocks
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
PRO r_pwi_sfra_spec_hr_net

    COMMON SFRAspctrmstuff, SFRAspctrm
    COMMON time ,xut1,xut2
    COMMON get_error, get_err_no, get_err_msg

    max_components = 6000	&	get_err_no = 0
    PWI_spec = fltarr (max_components, 320)
    PWI_time = dblarr (max_components)

    ideal_components = 4075.0	&	fill_val = 1.0e-32

    aPWI_offset = 0.0	&	spect_sec = 2.3552

    aPWI_spec = fltarr (320)	&	prev_offset = 0.0

    spec_sum = fltarr (320)	&	start_offset = 0.0
    start_offset0 = 0.0

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

; the PWI read routines all use Gnu wget to retrieve calibrated spectral
; densities via the internet, eliminating the need for files.

; the latest version of wget (as of 05/97) is available at
;       <URL:ftp://prep.ai.mit.edu/pub/gnu/wget-1.4.3.tar.gz>.
; if wget is installed in the system path, you should not need to do
; anything special to get this procedure to work.

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    if (total_sec GT 4600) then begin
	get_err_no = 4
	get_err_msg = '**Time range too large for HR reader'
	goto, bad_data
    endif

    sec_interval = 2.37

    sec_incr = sec_interval / 2.0
    if (sec_incr LE 2.36) then	sec_incr = 0.0

    start_offset = start_sod	&	delta_t = 1.5 * sec_interval

    mid_timestr = '1996-05-27T00:06:00'
    max_timestr = '1997-02-10T00:00:00'

    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
			sminute, ':', ssecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
			eminute, ':', esecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

; This reader will not be called unless total_sec <= 4500 (1 hour, 15 minutes)

    if (beg_timestr LT mid_timestr) then begin
	if (end_timestr LT mid_timestr) then begin
; antenna was Eu for all data
	    command1 = $
	'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfra_eu&begin_time= ' + $
		beg_timestr + '&end_time= ' + end_timestr + '"'
	    command2 = ''
	    Ant = 0	&	total_sec0 = total_sec
	endif else begin
; antenna was Eu for beginning data & Ez for ending data
	    command1 = $
	'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfra_eu&begin_time= ' + $
		beg_timestr + '&end_time= ' + mid_timestr + '"'
	    command2 = $
	'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfra_ez&begin_time= ' + $
		mid_timestr + '&end_time= ' + end_timestr + '"'
	    tyear = 1996	&	tdoy = 148	&	tsod = 360.0
; tsod = SOD when antenna was swittched to Ez

	    Ant = 2

; total_sec0 = amount of time in this data when antenna was Eu

	    if (sdoy EQ tdoy) then	total_sec0 = tsod  - start_sod $
	    else begin		; tyear = syear = 1996
		ddiff = tdoy - sdoy
		diff = 86400.0*ddiff + tsod - start_sod
		total_sec0 = diff
	    endelse
	    start_offset0 = start_offset + total_sec0
	endelse
    endif else if (beg_timestr LT max_timestr) then begin
      if (end_timestr LT max_timestr) then begin
; antenna was Ez for all data
	command1 = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfra_ez&begin_time= ' + $
		beg_timestr + '&end_time= ' + end_timestr + '"'
	command2 = ''
	Ant = 1 	&	total_sec0 = total_sec
      endif else begin
; antenna was Ez for beginning data & Eu for ending data
	command1 = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfra_ez&begin_time= ' + $
		beg_timestr + '&end_time= ' + max_timestr + '"'
	command2 = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfra_eu&begin_time= ' + max_timestr + $
		'&end_time= ' + end_timestr + '"'

	Ant = 3

; tsod = SOD when antenna was swittched bac to Eu
	tyear = 1997	&	tdoy = 41	&	tsod = 0.0
	if (syear EQ tyear) then	ddiff = tdoy - sdoy $
	else begin
	    if (syear mod 4 EQ 0) then	ydays = 366 $
	    else	ydays = 365
	    ddiff = ydays - sdoy + tdoy
	endelse
	diff = 86400.0*ddiff + tsod - start_sod
; here total_sec0 = amount of time the antenna was Ez
	total_sec0 = diff
	start_offset0 = start_offset + total_sec0
      endelse
    endif else begin
; antenna was Eu for all data
	command1 = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfra_eu&begin_time= ' + $
		beg_timestr + '&end_time= ' + end_timestr + '"'
	command2 = ''
	Ant = 0
	total_sec0 = total_sec
    endelse

    print, command1
    spawn, command1, unit = lun

    PWI_time(*) = 0.0

    spec_count = 0L	&	spec_sum(*) = 0.0	&	off_sum = 0.0

    DataOK = 1	&	i = 0L
    readu, lun, aPWI_offset, aPWI_spec
    on_ioerror, bad_data0

    limit = delta_t

    while (DataOK) do begin
	prev_offset = aPWI_offset

	while (aPWI_offset GT limit) do begin
	    if (i GT 0) then $
		PWI_time(i) = PWI_time(i - 1) + sec_interval $
	    else	PWI_time(i) = start_offset
	    PWI_spec(i, *) = 0.0
	    i = i + 1	&	limit = limit + sec_interval
	    t = err_check (i, max_components, 'r_pwi_sfra_spec_hr_net', $
			'Time overflow in sfra_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
	PWI_spec (i, *) = aPWI_spec
	PWI_time(i) = start_offset + aPWI_offset
	i = i + 1	&	limit = aPWI_offset + delta_t
	t = err_check (i, max_components, 'r_pwi_sfra_spec_sv_net', $
			'*Time overflow in sfra_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
	readu, lun, aPWI_offset, aPWI_spec
	on_ioerror, Finished
    endwhile

    Finished: close, lun	&	free_lun, lun

    if (prev_offset LT total_sec0) then begin
	prev_offset = prev_offset + sec_interval
	while (prev_offset LT total_sec0) do begin
	    PWI_time(i) = PWI_time(i - 1) + sec_interval
	    PWI_spec(i, *) = 0.0
	    i = i + 1	&	prev_offset = prev_offset + sec_interval
	    t = err_check (i, max_components, 'r_pwi_sfra_spec_hr_net', $
			'**Time overflow in sfra_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
    endif

    Finished1:

    if (strlen(command2) GT 0) then begin
	print, command2
	spawn, command2, unit=lun

	if (strlen(command1) GT 0) then begin
	    readu, lun, aPWI_offset, aPWI_spec
	    on_ioerror, Finished2
	endif else begin
	    readu, lun, aPWI_offset, aPWI_spec
	    on_ioerror, bad_data0
	endelse
	limit = total_sec0 + delta_t
	prev_offset = total_sec0 + aPWI_offset

	while (DataOK) do begin

	    while (prev_offset GT limit) do begin
		PWI_time(i) = PWI_time(i - 1) + sec_interval
		PWI_spec(i, *) = 0.0
		i = i + 1	&	limit = limit + sec_interval
		t = err_check (i, max_components, 'r_pwi_sfra_spec_hr_net', $
			'***Time overflow in sfra_spec', PWI_time, end_sod)
		if (t LT 0) then	goto, bad_data $
		else if (t LT 1) then	goto, Finished0
	    endwhile
	    PWI_spec(i, *) = aPWI_spec
	    PWI_time(i) = start_offset0 + aPWI_offset
	    i = i + 1	&	limit = total_sec0 + aPWI_offset + delta_t
	    t = err_check (i, max_components, 'r_pwi_sfra_spec_hr_net', $
			'****Time overflow in sfra_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	    prev_offset = total_sec0 + aPWI_offset
	    readu, lun, aPWI_offset, aPWI_spec
	    on_ioerror, Finished2
	endwhile

	Finished2: close, lun	&	free_lun, lun

	if (prev_offset LT total_sec) then begin
	    prev_offset = prev_offset + sec_interval
	    while (prev_offset LT total_sec) do begin
		PWI_time(i) = PWI_time(i - 1) + sec_interval
		PWI_spec(i, *) = 0.0
		i = i + 1
		prev_offset = prev_offset + sec_interval
		t = err_check (i, max_components, 'r_pwi_sfra_spec_hr_net', $
			'*****Time overflow in sfra_spec', PWI_time, end_sod)
		if (t LT 0) then	goto, bad_data $
		else if (t LT 1) then	goto, Finished0
	    endwhile
	endif
    endif

    Finished0:	nt = i	&	nf = 320L

    print, '% r_pwi_sfra_spec_hr_net:  spec dimensions ', nt, ' x ', nf

    time0 = { PB5:lonarr(3), t90:0.0D }

    if (Ant EQ 0) then $
	ttitle = 'PWI SFR A/Eu' $
    else if (Ant EQ 1) then $
	ttitle = 'PWI SFR A/Ez' $
    else if (Ant EQ 2) then $
	ttitle = 'PWI SFR A/Eu/Ez' $
    else if (Ant EQ 3) then $
	ttitle = 'PWI SFR A/Ez/Eu' $
    else 	ttitle = 'PWI SFR A'

    SFRAspctrm = { nt:0L, nf:nf, title:ttitle, $
		ytitle:'Frequency <Hz>', ztitle:'(V/m)!e2!n/Hz', $
		frequency:fltarr(nf), spec:fltarr(nt, nf), $
		time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time() is in SOD
    t90base = convert_timetoseconds(string (smonth, '/', sday, '/', syear, $
		' 00:00:00', format = '(i2.2, a1, i2.2, a1, i4.4, a9)'), /t90)

    max = nt - 1
    for j = 0L, max do begin
	SFRAspctrm.time(j).PB5(0) = syear
	SFRAspctrm.time(j).PB5(1) = sdoy
	SFRAspctrm.time(j).PB5(2) = PWI_time(j)
;change time formats to t90
	SFRAspctrm.time(j).t90 = PWI_time(j) + t90base.value
	SFRAspctrm.spec(j, *) = PWI_spec(j, *)
    endfor

    SFRAspctrm.nt = nt

; once you are herel, the data has been correctly read. At this point you
; may restrict the arrays passed back to the time chosen using xut1 and xut2,
; you need to convert the time information to T90 (use the routines in
; $PAPCO_BASE/PAPCO_Lib/convert.pro) - if you intend to use the PAPCO
; plotting templates for your data. If you use your own plot
; routines this step is not needed, and you also don't have to re-save the
; converted file.

    print, '% r_pwi_sfra_spec_hr_net: converted time to T90'

    if (get_err_no EQ 0) then	get_err_msg = 'pwi_sfra_spec read succesfully'
    print, '% r_pwi_sfra_spec_hr_net: ', get_err_msg

    return

    bad_data0: get_err_no = 1
	get_err_msg = '*No network data returned'

    bad_data:  print, '% r_pwi_sfra_spec_hr_net: ' + get_err_msg
	print, '**No PWI data returned'
end

;****************************************************************************
;*  NAME:        fill_spec
;*
;*  DESCRIPTION: Will divide the sum of spectra and the sum of time
;*		 offsets by the non-zero value in spec_count and will
;*		 place the results in PWI_spec(i) and PWI_time(i),
;*		 respectively
;*
;*  INPUT:       i (current time imdex), spec_count, spec_sum, off_sum
;*
;*  CALLED by:	 r_pwi_sfr_spec_sv_local, r_pwi_sfr_spec_hr_local
;*
;*  OUTPUT:      PWI_spec (modified), PWI_time (modified)
;*		 spec_count, spec_sum, and off_sum are re-initialized
;*
;*  DATE:        June 1998
;*
;*  AUTOR:       J. Dowell
;*
;*  CHANGES: 
;****************************************************************************
pro fill_spec, PWI_spec, PWI_time, i, spec_count, spec_sum, off_sum

	tPWI_spec = fltarr(320)

	PWI_time(i) = off_sum / spec_count
	tPWI_spec = spec_sum/spec_count
	PWI_spec(i, *) = tPWI_spec
	off_sum = 0.0	&	spec_count = 0L
	spec_sum(*) = 0.0
end

;****************************************************************************
;*  NAME:        fill_Fce
;*
;*  DESCRIPTION: Will divide the sum of Fce values and the sum of time
;*		 offsets by the non-zero value in spec_count and will
;*		 place the results in FCE(i) and Fce_time(i),
;*		 respectively
;*
;*  INPUT:       i (current time imdex), Fce_count, Fce_sum, foff_sum
;*
;*  CALLED by:	 r_pwi_sfra_spec_sv_local, r_pwi_sfr_spec_hr_local.
;*
;*  OUTPUT:      FCE (modified), Fce_time (modified)
;*		 Fce_count, Fce_sum, and foff_sum are re-initialized
;*
;*  DATE:        June 1998
;*
;*  AUTOR:       J. Dowell
;*
;*  CHANGES: 
;****************************************************************************
pro fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count

	Fce_time(i) = foff_sum/Fce_count
	FCE(i) = Fce_sum/Fce_count
	foff_sum = 0.0	&	Fce_count = 0L
	Fce_sum = 0.0
end

;****************************************************************************
;*  NAME:        r_pwi_sfr_spec_sv_local
;*
;*  DESCRIPTION: Routine to read SFR A survey data from a local CDF file.
;*		 Adheres to PAPCO conventions: Error common block, changes
;*		 time format to T90
;*
;*  INPUT:       none - gets data from common blocks
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
PRO r_pwi_sfr_spec_sv_local

    COMMON SFRAspctrmstuff, SFRAspctrm
    COMMON SFRBspctrmstuff, SFRBspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    total_sec = xut2 - xut1

    n = size(FCE_lp)
    if (n(0) GT 0) then begin
	mt = FCE_lp.nt - 1
	fac = 0.02 * total_sec
	if (FCE_lp.time(0).t90 LE xut1 + fac AND FCE_lp.time(mt).t90 $
			GE xut2 - fac) then	do_Fce = 0 $
	else	do_Fce = 1
    endif else	do_Fce = 1

    max_components = 3000	&	get_err_no = 0
    PWI_spec_a = fltarr(max_components, 320)
    PWI_spec_b = fltarr(max_components, 320)
    aPWI_time = dblarr(max_components)
    bPWI_time = dblarr(max_components)

    if (do_Fce) then begin
	FCE = fltarr(max_components)
	Fce_time = dblarr(max_components)
    endif

    ideal_components = 1000	&	fill_val = 1.0e-32
    SFR_sv_offset = 32.9728		&	Sec_per_day = 86400.0

    max_SFR_sec = 64000

    spect_sec = 32.9728		&	SFR_interval = 33.0

    aPWI_spec = fltarr (320)	&	bPWI_spec = fltarr (320)
    taPWI_spec = fltarr (320)	&	tbPWI_spec = fltarr (320)
    aspec_sum = fltarr(320)	&	bspec_sum = fltarr(320)
    aPWI_offset = 0.0	&	PB5 = lonarr(3)

    SFR1A = fltarr(32)	&	SFR4A = fltarr(64)
    SFR2A = fltarr(32)	&	SFR5A = fltarr(64)
    SFR3A = fltarr(32)
    SFR1B = fltarr(32)	&	SFR4B = fltarr(64)
    SFR2B = fltarr(32)	&	SFR5B = fltarr(64)
    SFR3B = fltarr(32)
    SFRA_Ant = 0L	&	SFRB_Ant = 0L

    start_offset = 0.0	&	prev_offset = 0.0

    PWI_path = getenv ('PWI_DATA')
    L = strlen(PWI_path)
    if (L EQ 0) then begin
	get_err_no = 1
	get_err_msg = 'Path to PWI data not found.  Env. variable Not set'
	goto, bad_data
    endif else if (strmid(PWI_path, L - 1, 1) NE '/') then $
		PWI_path = PWI_path + '/'

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    total_spec = long(total_sec / spect_sec + 0.5)
    if (total_spec LT 1) then	total_spec = 1
    num_to_avg = fix(float(total_spec)/float(ideal_components) + 0.5)
    if (num_to_avg LT 1) then	num_to_avg = 1
    sec_interval = num_to_avg * spect_sec

    if (sec_interval LT spect_sec) then	sec_interval = spect_sec

    delta_t = 1.5 * sec_interval	&	start_offset = 0.0

    mid_timestr = '1996-05-27T00:06:00'
    max_timestr = '1997-02-10T00:00:00'

    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
			sminute, ':', ssecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
			eminute, ':', esecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    if (beg_timestr LT mid_timestr) then begin

	if (end_timestr LT mid_timestr) then	AAnt = 0 $
	else	AAnt = 2

    endif else if (beg_timestr LT max_timestr) then begin

      if (end_timestr LT max_timestr) then	AAnt = 1 $
      else	AAnt = 3

    endif else	AAnt = 0

    PB5_name = 'Time_PB5'
    SFR1A_name = 'SFR1A'	&	SFR2A_name = 'SFR2A'
    SFR3A_name = 'SFR3A'	&	SFR4A_name = 'SFR4A'
    SFR5A_name = 'SFR5A'

    SFR1B_name = 'SFR1B'	&	SFR2B_name = 'SFR2B'
    SFR3B_name = 'SFR3B'	&	SFR4B_name = 'SFR4B'
    SFR5B_name = 'SFR5B'

    AAnt_name = 'SFRA_Antenna'	&	BAnt_name = 'SFRB_Antenna'
    FCE_name = 'Fce'

    cyear = syear	&	cmonth = smonth	&	cday = sday
    if (syear mod 4 EQ 0) then	m2 = 29 $
    else	m2 = 28
    month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]
    date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
    result = findfile(PWI_PATH, count = cc)
    ccmax = cc - 1
    tfile_name = 'po_pwi_sfr_' + date_string + '_v01.sv.cdf'
    file_ok = 0
    for t = 0, ccmax do $
	if (result(t) EQ tfile_name) then	file_ok = 1

    file_name = PWI_path + tfile_name

    if (NOT file_ok) then begin
	get_err_no = 1
	get_err_msg = '% r_pwi_sfr_spec_local: Unable to open ' + $
			file_name + '.  File not found.'
	goto, bad_data
    endif

    print, file_name
    CDF_ID = CDF_open (file_name)

    CDF_Result = CDF_inquire (CDF_ID)
    MaxRec = CDF_Result.MaxRec + 1

    VarNum_PB5 = CDF_varnum(CDF_ID, PB5_name)
    VarNum_AAnt = CDF_varnum(CDF_ID, AAnt_name)
    VarNum_BAnt = CDF_varnum(CDF_ID, BAnt_name)
    VarNum_SFR1A = CDF_varnum(CDF_ID, SFR1A_name, IsZvar)
    VarNum_SFR2A = CDF_varnum(CDF_ID, SFR2A_name, IsZvar)
    VarNum_SFR3A = CDF_varnum(CDF_ID, SFR3A_name, IsZvar)
    VarNum_SFR4A = CDF_varnum(CDF_ID, SFR4A_name, IsZvar)
    VarNum_SFR5A = CDF_varnum(CDF_ID, SFR5A_name, IsZvar)
    VarNum_SFR1B = CDF_varnum(CDF_ID, SFR1B_name, IsZvar)
    VarNum_SFR2B = CDF_varnum(CDF_ID, SFR2B_name, IsZvar)
    VarNum_SFR3B = CDF_varnum(CDF_ID, SFR3B_name, IsZvar)
    VarNum_SFR4B = CDF_varnum(CDF_ID, SFR4B_name, IsZvar)
    VarNum_SFR5B = CDF_varnum(CDF_ID, SFR5B_name, IsZvar)
    VarNum_FCE = CDF_varnum(CDF_ID, FCE_name)

    RecNum = 0L	&	i = 0L	&	Done = 0

    aPWI_time(*) = 0.0	&	bPWI_time(*) = 0.0
    aspec_count = 0L	&	bspec_count = 0L
    aoff_sum = 0.0	&	boff_sum = 0.0
    Fce_sum = 0.0	&	Fce_count = 0L
    FCE_in = 0.0	&	foff_sum = 0.0

    limit = delta_t

    while (NOT Done) do begin
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start = RecNum
	toffset = start_offset + double(PB5(2))/1000.0
	if (toffset GE start_sod) then begin
	    Done = 1
	    CDF_varget, CDF_ID, VarNum_AAnt, SFRA_Ant, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, SFRB_Ant, Rec_Start = RecNum
	    if (do_Fce) then $
		CDF_varget, CDF_ID, VarNum_Fce, FCE_in, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR1A, SFR1A, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR2A, SFR2A, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR3A, SFR3A, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR4A, SFR4A, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR5A, SFR5A, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR1B, SFR1B, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR2B, SFR2B, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR3B, SFR3B, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR4B, SFR4B, /ZVARIABLE, $
					Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR5B, SFR5B, /ZVARIABLE, $
					Rec_Start = RecNum
	endif else begin
	    prev_offset = toffset
	endelse
	RecNum = RecNum + 1
    endwhile

    prev_aant = SFRA_Ant(0)	&	prev_bant = SFRB_Ant(0)

    limit = start_sod + delta_t
    taPWI_spec(*) = 0.0	&	tbPWI_spec(*) = 0.0

    Done = 0

    while (NOT Done) do begin
	if (aspec_count GE num_to_avg) then begin
	    fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, aoff_sum
	    if (bspec_count GT 0) then $
		fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum $
	    else begin
		PWI_spec_b(i, *) = 0.0
		if (i GT 0) then	bPWI_time(i) = bPWI_time(i - 1) + sec_interval $
		else	bPWI_time(i) = start_sod
	    endelse

	    if (do_Fce) then begin
		if (Fce_count GT 0) then $
		    fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count $
		else begin
		    if (i GT 0) then	Fce_time(i) = Fce_time(i - 1) + sec_interval $
		    else	Fce_time(i) = start_sod
		    FCE(i) = -1.0
		endelse
	    endif
	    limit = aPWI_time(i) + delta_t
	    if (limit GT end_sod) then	goto, Finished
	    i = i + 1
	    t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'Time overflow in sfr_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endif else if (bspec_count GE num_to_avg) then begin
	    fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum
	    if (aspec_count GT 0) then $
		fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, aoff_sum $
	    else begin
		PWI_spec_a(i, *) = 0.0
		if (i GT 0) then	aPWI_time(i) = aPWI_time(i - 1) + sec_interval $
		else	aPWI_time(i) = start_sod
	    endelse

	    if (do_Fce) then begin
		if (Fce_count GT 0) then $
		    fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count $
		else begin
		    if (i GT 0) then	Fce_time(i) = Fce_time(i - 1) + sec_interval $
		    else	Fce_time(i) = start_sod
		    FCE(i) = -1.0
		endelse
	    endif
	    limit = aPWI_time(i) + delta_t
	    if (limit GT end_sod) then	goto, Finished
	    i = i + 1
	    t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'*Time overflow in sfr_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endif else if (toffset GE limit) then begin

	    if (aspec_count GT 0) then $
		fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, aoff_sum $
	    else begin
		PWI_spec_a(i, *) = 0.0
		if (i GT 0) then	aPWI_time(i) = aPWI_time(i - 1) + sec_interval $
		else	aPWI_time(i) = start_sod
	    endelse

	    if (bspec_count GT 0) then $
		fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum $
	    else begin
		PWI_spec_b(i, *) = 0.0
		if (i GT 0) then	bPWI_time(i) = bPWI_time(i - 1) + sec_interval $
		else	bPWI_time(i) = start_sod
	    endelse

	    if (do_Fce) then begin
		if (Fce_count GT 0) then $
		    fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count $
		else begin
		    if (i GT 0) then	Fce_time(i) = Fce_time(i - 1) + sec_interval $
		    else	Fce_time(i) = start_sod
		    FCE(i) = -1.0
		endelse
	    endif
	    limit = aPWI_time(i) + delta_t
	    if (limit GT end_sod) then	goto, Finished
	    i = i + 1
	    t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'**Time overflow in sfr_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	    while (toffset GT limit) do begin
		if (i GT 0) then begin
		    aPWI_time(i) = aPWI_time(i - 1) + sec_interval
		    bPWI_time(i) = bPWI_time(i - 1) + sec_interval
		    if (do_Fce) then	Fce_time(i) = Fce_time(i - 1) + sec_interval
		endif else begin
		    aPWI_time(i) = start_sod
		    bPWI_time(i) = start_sod
		    if (do_Fce) then	Fce_time(i) = start_sod
		endelse
		PWI_spec_a(i, *) = 0.0	&	PWI_spec_b(i, *) = 0.0
		if (do_Fce) then	FCE(i) = -1.0
;		i = i + 1	&	limit = limit + sec_interval
		i = i + 1	&	limit = limit + delta_t
		if (limit GT end_sod) then	goto, Finished
		t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'***Time overflow in sfr_spec', PWI_time, end_sod)
		if (t LT 0) then	goto, bad_data $
		else if (t LT 1) then	goto, Finished0
	    endwhile
	    if (toffset GT end_sod) then goto, Finished
	endif

	if (SFRA_Ant(0) LT 2) then begin
	    for j = 0, 63, 2 do begin
		k = j + 1	&	m = j/2
		aPWI_spec(j) = SFR1A(m)
		aPWI_spec(k) = SFR1A(m)
		aPWI_spec(64 + j) = SFR2A(m)
		aPWI_spec(64 + k) = SFR2A(m)
		aPWI_spec(128 + j) = SFR3A(m)
		aPWI_spec(128 + k) = SFR3A(m)
	    endfor

	    for j = 0, 63 do begin
		k = j + 192	&	m = k + 64
		aPWI_spec(k) = SFR4A(j)
		aPWI_spec(m) = SFR5A(j)
	    endfor

; We need to NOT add Spectra together if Antennae are mixed, but what
; shall we do about SFRB and FCE in this case???
;	    if (SFRA_Ant(0) EQ prev_aant) then begin
		aoff_sum = aoff_sum + toffset
		aspec_count = aspec_count + 1
		aspec_sum = aspec_sum + aPWI_spec(*)
;	    endif else begin
;	    endelse
	endif

	if (SFRB_Ant(0) GT 2) then begin
	    for j = 0, 63, 2 do begin
		k = j + 1	&	m = j/2
		bPWI_spec(j) = SFR1B(m)
		bPWI_spec(k) = SFR1B(m)
		bPWI_spec(64 + j) = SFR2B(m)
		bPWI_spec(64 + k) = SFR2B(m)
		bPWI_spec(128 + j) = SFR3B(m)
		bPWI_spec(128 + k) = SFR3B(m)
	    endfor

	    for j = 0, 63 do begin
		k = j + 192	&	m = k + 64
		bPWI_spec(k) = SFR4B(j)
		bPWI_spec(m) = SFR5B(j)
	    endfor

	    boff_sum = boff_sum + toffset
	    bspec_count = bspec_count + 1
	    bspec_sum = bspec_sum + bPWI_spec(*)
	endif

	if (do_Fce) then begin
	    Fce_sum = Fce_sum + FCE_in(0)
	    Fce_count = Fce_count + 1
	    foff_sum = foff_sum + toffset
	endif

	prev_offset = toffset
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start = RecNum
	toffset = start_offset + double(PB5(2))/1000.0
	if (toffset GT end_sod) then Done = 1 $
	else begin
	    prev_aant = SFRA_Ant	&	prev_bant = SFRB_Ant
	    CDF_varget, CDF_ID, VarNum_AAnt, SFRA_Ant, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, SFRB_Ant, Rec_Start = RecNum
	    if (do_Fce) then $
		CDF_varget, CDF_ID, VarNum_Fce, FCE_in, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR1A, SFR1A, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR2A, SFR2A, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR3A, SFR3A, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR4A, SFR4A, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR5A, SFR5A, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR1B, SFR1B, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR2B, SFR2B, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR3B, SFR3B, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR4B, SFR4B, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR5B, SFR5B, /ZVARIABLE, $
				Rec_Start = RecNum
	    RecNum = RecNum + 1
	    if (RecNum GE MaxRec) then begin
		if (end_sod EQ start_offset + sec_per_day) then goto, Finished
		CDF_close, CDF_ID

		cday = cday + 1
		if (cday GT month_tab(cmonth - 1)) then begin
		    cday = 1	&	cmonth = cmonth + 1
		    if (cmonth GT 12) then begin
			cmonth = 1	&	cyear = cyear + 1
		    endif
		endif
		date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
		tfile_name = 'po_pwi_sfr_' + date_string + '_v01.sv.cdf'
		file_name = PWI_path + tfile_name
		file_ok = 0
		for t = 0, ccmax do $
		    if (result(t) EQ tfile_name) then	file_ok = 1

		if (NOT file_ok) then begin
		    get_err_no = 1
		    get_err_msg = '% r_pwi_sfr_spec_sv_local: Unable to open ' $
				+ file_name + '.  File not found.'
		    if (end_sod - PWI_time(i - 1) LE 60.0) then begin
			print, 'r_pwi_sfr_spec_sv_local: ', get_err_msg
			goto, Finished0
		    endif else	goto, bad_data
		endif

		print, file_name
		CDF_ID = CDF_open (file_name)
		CDF_Result = CDF_inquire (CDF_ID)
		MaxRec = CDF_Result.MaxRec + 1
		start_offset = start_offset + Sec_per_day
		RecNum = 0L
	    endif
	endelse
    endwhile

    Finished: CDF_close, CDF_ID

    if (aspec_count GT 0) then begin
	fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, aoff_sum
	if (bspec_count GT 0) then $
	    fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum $
	else begin
	    PWI_spec_b(i, *) = 0.0
	    if (i GT 0) then	bPWI_time(i) = bPWI_time(i - 1) + sec_interval $
	    else	bPWI_time(i) = start_sod
	endelse

	if (do_Fce) then begin
	    if (Fce_count GT 0) then $
		fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count $
	    else begin
		if (i GT 0) then	Fce_time(i) = Fce_time(i - 1) + sec_interval $
		else	Fce_time(i) = start_sod
		FCE(i) = -1.0
	    endelse
	endif
;	i = i + 1	&	limit = limit + sec_interval
	i = i + 1	&	limit = limit + delta_t
	t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'****Time overflow in sfr_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
    endif else if (bspec_count GT 0) then begin
	fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum
	PWI_spec_a(i, *) = 0.0
	if (i GT 0) then	aPWI_time(i) = aPWI_time(i - 1) + sec_interval $
	else	aPWI_time(i) = start_sod
	if (do_Fce) then begin
	    if (Fce_count GT 0) then $
		fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count $
	    else begin
		if (i GT 0) then	Fce_time(i) = Fce_time(i - 1) + sec_interval $
		else	Fce_time(i) = start_sod
		FCE(i) = -1.0
	    endelse
	endif
;	i = i + 1	&	limit = limit + sec_interval
	i = i + 1	&	limit = limit + delta_t
	t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'*****Time overflow in sfr_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
    endif else if (do_Fce) then begin
	if (Fce_count GT 0) then begin
	    fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count
	    PWI_spec_a(i, *) = 0.0
	    if (i GT 0) then	aPWI_time(i) = aPWI_time(i - 1) + sec_interval $
	    else	aPWI_time(i) = start_sod
	    PWI_spec_b(i, *) = 0.0
	    if (i GT 0) then	bPWI_time(i) = bPWI_time(i - 1) + sec_interval $
	    else	bPWI_time(i) = start_sod
;	    i = i + 1	&	limit = limit + sec_interval
	    i = i + 1	&	limit = limit + delta_t
	    t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'******Time overflow in sfr_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endif
    endif

    if (NOT Done) then begin
	if (toffset LT end_sod) then begin
	    toffset = toffset + sec_interval
	    while (toffset LT end_sod) do begin
		aPWI_time(i) = toffset
		bPWI_time(i) = toffset
		PWI_spec_a(i, *) = 0.0
		PWI_spec_b(i, *) = 0.0
		if (do_Fce) then begin
		    FCE(i) = -1.0
		    Fce_time(i) = toffset
		endif
		toffset = toffset + sec_interval
		i = i + 1
		t = err_check (i, max_components, 'r_pwi_sfr_spec_sv_local', $
			'*******Time overflow in sfr_spec', PWI_time, end_sod)
		if (t LT 0) then	goto, bad_data $
		else if (t LT 1) then	goto, Finished0
	    endwhile
	endif
    endif

    Finished0:	nt = i	&	nf = 320L
    end_indx = nt - 1

    print, '% r_pwi_sfr_spec_sv_local:  spec dimensions ', nt, ' x ', nf

    time0 = { PB5:lonarr(3), t90:0.0D }

    if (AAnt EQ 0) then $
	tatitle = 'PWI SFR A/Eu' $
    else if (AAnt EQ 1) then $
	tatitle = 'PWI SFR A/Ez' $
    else if (AAnt EQ 2) then $
	tatitle = 'PWI SFR A/Eu/Ez' $
    else if (AAnt EQ 3) then $
	tatitle = 'PWI SFR A/Ez/Eu' $
    else 	tatitle = 'PWI SFR A'

    tbtitle = 'PWI SFR B/L'

    SFRAspctrm = { nt:0L, nf:nf, title:tatitle, ytitle:'Frequency <Hz>', $
		ztitle:'(V/m)!e2!n/Hz', frequency:fltarr(nf), $
		spec:fltarr(nt, nf), time:replicate(time0, nt) }

    SFRBspctrm = { nt:0L, nf:nf, title:tbtitle, ytitle:'Frequency <Hz>', $
		ztitle:'nT!e2!n/Hz', frequency:fltarr(nf), $
		spec:fltarr(nt, nf), time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time() is in SOD
    t90base = convert_timetoseconds(string (smonth, '/', sday, '/', syear, $
		' 00:00:00', format = '(i2.2, a1, i2.2, a1, i4.4, a9)'), /t90)

    if (do_Fce) then begin
	index = where (FCE(0:end_indx) GT 0.0, kount)
	if (kount GT 0) then begin
	    nt_f = kount
	    FCE_lp = { nt:0L, title:'Polar Fce', ytitle:'Frequency <Hz>', $
			fce:fltarr(nt_f), time:replicate(time0, nt_f) }
	    FCE_lp.nt = nt_f
	    FCE_lp.time(*).PB5(0) = syear
	    FCE_lp.time(*).PB5(1) = sdoy
	    FCE_lp.time(*).PB5(2) = Fce_time(index)
	    FCE_lp.time(*).T90 = Fce_time(index) + t90base.value
	    FCE_lp.fce = FCE(index)
; print, FCE(index)
	endif else begin
	    get_err_no = 4
	    get_err_msg = '**No Fce Data found'
	endelse
    endif

    max = nt - 1
    for j = 0L, max do begin
	SFRAspctrm.time(j).PB5(0) = syear
	SFRBspctrm.time(j).PB5(0) = syear
	SFRAspctrm.time(j).PB5(1) = sdoy
	SFRBspctrm.time(j).PB5(1) = sdoy
	SFRAspctrm.time(j).PB5(2) = aPWI_time(j)
	SFRBspctrm.time(j).PB5(2) = bPWI_time(j)
;change time formats to t90
	SFRAspctrm.time(j).t90 = aPWI_time(j) + t90base.value
	SFRBspctrm.time(j).t90 = bPWI_time(j) + t90base.value
	SFRAspctrm.spec(j, *) = PWI_spec_a(j, *)
	SFRBspctrm.spec(j, *) = PWI_spec_b(j, *)
    endfor

    SFRAspctrm.nt = nt	&	SFRBspctrm.nt = nt

; once you are herel, the data has been correctly read. At this point you
; may restrict the arrays passed back to the time chosen using xut1 and xut2,
; you need to convert the time information to T90 (use the routines in
; $PAPCO_BASE/PAPCO_Lib/convert.pro) - if you intend to use the PAPCO
; plotting templates for your data. If you use your own plot
; routines this step is not needed, and you also don't have to re-save the
; converted file.

    print, '% r_pwi_sfr_spec_sv_local: converted time to T90'

    if (get_err_no EQ 0) then	get_err_msg = 'pwi_sfr_spec read successfully'
    print, '% r_pwi_sfr_spec_sv_local: ', get_err_msg

    return

    bad_data:	print, get_err_msg
	print, '**No PWI data returned'
end

;****************************************************************************
;*  NAME:        r_pwi_sfr_spec_hr_local
;*
;*  DESCRIPTION: Routine to read SFR A data from a local CDF file.
;*		 Adheres to PAPCO conventions: Error common block, changes
;*		 time format to T90
;*
;*  INPUT:       none - gets data from common blocks
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
PRO r_pwi_sfr_spec_hr_local

    COMMON SFRAspctrmstuff, SFRAspctrm
    COMMON SFRBspctrmstuff, SFRBspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    total_sec = xut2 - xut1

    n = size(FCE_lp)
    if (n(0) GT 0) then begin
	mt = FCE_lp.nt - 1
	fac = 0.02 * total_sec
	if (FCE_lp.time(0).t90 LE xut1 + fac AND FCE_lp.time(mt).t90 $
			GE xut2 - fac) then	do_Fce = 0 $
	else	do_Fce = 1
    endif else	do_Fce = 1

    max_components = 3000	&	get_err_no = 0
    aspec_sum = fltarr (320)
    bspec_sum = fltarr (320)
    PWI_spec_a = fltarr (max_components, 320)
    PWI_spec_b = fltarr (max_components, 320)
    aPWI_time = dblarr (max_components)
    bPWI_time = dblarr (max_components)

    if (do_Fce) then begin
	FCE = fltarr(max_components)
	Fce_time = dblarr(max_components)
    endif

    ideal_components = 1000	&	fill_val = 1.0e-32
    SFR_sv_offset = 32.9728		&	Sec_per_day = 86400.0
    SFR2_offset = 16.4864	&	SFR3_offset = 8.2432
    SFR45_offset = 2.3552

    roffset = [ 0, 0 ]	&	sfr2_counts = [ 32, 2 ]
    sfr3_counts = [ 32, 4 ]	&	sfr45_counts = [ 64, 14 ]
    ainterval = [ 1, 1 ]

    max_SFR_sec = 64000

    spect_sec = 32.9728		&	SFR_interval = 33.0

    aPWI_offset = 0.0	&	aPWI_spec = fltarr (320)
    bPWI_offset = 0.0	&	bPWI_spec = fltarr (320)
    Fce_in = 0.0	&	fce_offset = 0.0

    SFR1A = fltarr(32)	&	SFR4A = fltarr(64, 14)
    SFR2A = fltarr(32, 2)	&	SFR5A = fltarr(64, 14)
    SFR3A = fltarr(32, 4)

    SFR1B = fltarr(32)	&	SFR4B = fltarr(64, 14)
    SFR2B = fltarr(32, 2)	&	SFR5B = fltarr(64, 14)
    SFR3B = fltarr(32, 4)

    PB5 = lonarr(3)	&	SFRA_Ant = 0L	&	SFRB_Ant = 0L

    start_offset = 0.0	&	prev_offset = 0.0

    PWI_path = getenv ('PWI_DATA')
    L = strlen(PWI_path)
    if (L EQ 0) then begin
	get_err_no = 1
	get_err_msg = 'Path to PWI data not found.  Env. variable Not set'
	goto, bad_data
    endif else if (strmid(PWI_path, L - 1, 1) NE '/') then $
		PWI_path = PWI_path + '/'

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    if (total_sec GT 2700) then	begin
	do_avg = 1	&	add_interval = 2.0 * SFR45_offset
	if (total_sec GE 10800) then begin
	    total_spec = long(total_sec / SFR45_offset + 0.5)
	    if (total_spec LT 1) then	total_spec = 1
	    num_to_avg = total_spec / ideal_components
	    if (num_to_avg LT 1) then begin
		num_to_avg = 1	&	sec_interval = SFR45_offset
	    endif else if (num_to_avg GT 2) then begin
		temp = fix(float(num_to_avg)/14.0 + 0.5)
		if (temp LT 14) then	temp = 14 $
		else if (temp LT 28) then	temp = 28 $
		else if (temp LT 42) then	temp = 42 $
		else	temp = 56

		num_to_avg = temp
		sec_interval = num_to_avg * SFR45_offset
		if (sec_interval LT SFR_sv_offset) then $
		    sec_interval = SFR_sv_offset

		spec_to_avg = num_to_avg / 14
	    endif else begin
		num_to_avg = 2	&	sec_interval = add_interval
	    endelse
	endif else begin
	    num_to_avg = 2	&	sec_interval = add_interval
	endelse

    endif else begin
	do_avg = 0	&	add_interval = SFR45_offset
	num_to_avg = 1
    endelse
    sec_interval = spect_sec

    delta_t = 1.6 * sec_interval	&	start_offset = 0.0

    mid_timestr = '1996-05-27T00:06:00'
    max_timestr = '1997-02-10T00:00:00'

    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
			sminute, ':', ssecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
			eminute, ':', esecond, $
	format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')

    if (beg_timestr LT mid_timestr) then begin

	if (end_timestr LT mid_timestr) then	Ant = 0 $
	else	Ant = 2

    endif else if (beg_timestr LT max_timestr) then begin

      if (end_timestr LT max_timestr) then	Ant = 1 $
      else	Ant = 3

    endif else	Ant = 0

    PB5_name = 'Time_PB5'
    SFR1A_name = 'SFR1A'	&	SFR2A_name = 'SFR2A'
    SFR3A_name = 'SFR3A'	&	SFR4A_name = 'SFR4A'
    SFR5A_name = 'SFR5A'

    SFR1B_name = 'SFR1B'	&	SFR2B_name = 'SFR2B'
    SFR3B_name = 'SFR3B'	&	SFR4B_name = 'SFR4B'
    SFR5B_name = 'SFR5B'

    AAnt_name = 'SFRA_Antenna'	&	BAnt_name = 'SFRB_Antenna'
    FCE_name = 'Fce'

    cyear = syear	&	cmonth = smonth	&	cday = sday
    if (syear mod 4 EQ 0) then	m2 = 29 $
    else	m2 = 28
    month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]
    date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
    result = findfile(PWI_PATH, count = cc)
    ccmax = cc - 1
    tfile_name = 'po_h1_pwi_' + date_string + '_v01.cdf'
    file_name = PWI_path + tfile_name
    file_ok = 0
    for t = 0, ccmax do $
	if (result(t) EQ tfile_name) then	file_ok = 1

    if (NOT file_ok) then begin
	get_err_no = 1
	get_err_msg = '*Unable to open ' + file_name + '.  File not found.'
	goto, bad_data
    endif

    print, file_name
    CDF_ID = CDF_open (file_name)

    CDF_Result = CDF_inquire (CDF_ID)
    MaxRec = CDF_Result.MaxRec
    MaxRec = MaxRec + 1

    VarNum_PB5 = CDF_varnum(CDF_ID, PB5_name)
    VarNum_AAnt = CDF_varnum(CDF_ID, AAnt_name)
    VarNum_BAnt = CDF_varnum(CDF_ID, BAnt_name)
    VarNum_FCE = CDF_varnum(CDF_ID, FCE_name)
    VarNum_SFR1A = CDF_varnum(CDF_ID, SFR1A_name, IsZvar)
    VarNum_SFR2A = CDF_varnum(CDF_ID, SFR2A_name, IsZvar)
    VarNum_SFR3A = CDF_varnum(CDF_ID, SFR3A_name, IsZvar)
    VarNum_SFR4A = CDF_varnum(CDF_ID, SFR4A_name, IsZvar)
    VarNum_SFR5A = CDF_varnum(CDF_ID, SFR5A_name, IsZvar)
    VarNum_SFR1B = CDF_varnum(CDF_ID, SFR1B_name, IsZvar)
    VarNum_SFR2B = CDF_varnum(CDF_ID, SFR2B_name, IsZvar)
    VarNum_SFR3B = CDF_varnum(CDF_ID, SFR3B_name, IsZvar)
    VarNum_SFR4B = CDF_varnum(CDF_ID, SFR4B_name, IsZvar)
    VarNum_SFR5B = CDF_varnum(CDF_ID, SFR5B_name, IsZvar)

    RecNum = 0L	&	i = 0L	&	Done = 0

    aPWI_time(*) = 0.0	&	aspec_sum(*) = 0.0
    aspec_count = 0L	&	aoff_sum = 0.0
    bPWI_time(*) = 0.0	&	bspec_sum(*) = 0.0
    bspec_count = 0L	&	boff_sum = 0.0
    Fce_sum = 0.0	&	foff_sum = 0.0
    Fce_count = 0L

    while (NOT Done) do begin
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start = RecNum
	toffset = start_offset + double(PB5(2))/1000.0
; the time that we want for an HR plot may occur in the middle of a record
; so we want to stop at the record that contains it.
	toffset1 = toffset + SFR_sv_offset
	if (toffset GE start_sod or toffset1 GT start_sod) then begin
	    Done = 1
	    CDF_varget, CDF_ID, VarNum_AAnt, SFRA_Ant, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, SFRB_Ant, Rec_Start = RecNum
	    if (do_Fce) then $
		CDF_varget, CDF_ID, VarNum_Fce, FCE_in, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR1A, SFR1A, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, SFR2A_name, SFR2A, $
		Rec_Start = RecNum, Rec_count = 1, Rec_interval = 1, $
		count = SFR2_counts, offset = roffset, interval = ainterval
	    CDF_varget, CDF_ID, VarNum_SFR3A, SFR3A, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR3_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR4A, SFR4A, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR5A, SFR5A, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR1B, SFR1B, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, SFR2B_name, SFR2B, $
		Rec_Start = RecNum, Rec_count = 1, Rec_interval = 1, $
		count = SFR2_counts, offset = roffset, interval = ainterval
	    CDF_varget, CDF_ID, VarNum_SFR3B, SFR3B, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR3_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR4B, SFR4B, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR5B, SFR5B, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	endif else	prev_offset = toffset
	RecNum = RecNum + 1
    endwhile

    limit = start_sod + delta_t

    Done = 0	&	do_print = 1	&	spec_done = 0

    while (NOT Done) do begin
	if (do_avg) then begin
	    if (num_to_avg GT 2) then begin
		if (aspec_count EQ spec_to_avg) then begin
		    fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, aoff_sum
		    spec_done = 1
		endif
		if (bspec_count EQ spec_to_avg) then begin
		    fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum
		    if (NOT spec_done) then begin
			if (aspec_count GT 0) then $
			    fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, $
							aoff_sum $
			else begin
			    if (i GT 0) then $
				aPWI_time(i) = aPWI_time(i - 1) + sec_interval $
			    else	aPWI_time(i) = start_sod
			    PWI_spec_a(i, *) = 0.0
			endelse
			spec_done = 1
		    endif
		endif else if (spec_done) then begin
		    if (bspec_count GT 0) then $
			fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, $
					bspec_sum, boff_sum $
		    else begin
			if (i GT 0) then bPWI_time(i) = bPWI_time(i - 1) + sec_interval $
			else	bPWI_time(i) = start_sod
			PWI_spec_b(i, *) = 0.0
		    endelse
		endif
		if (do_Fce) then begin
		    if (Fce_count EQ spec_to_avg) then begin
			fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count
			if (NOT spec_done) then begin
			    if (aspec_count GT 0) then $
				fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, $
					aspec_sum, aoff_sum $
			    else begin
				if (i GT 0) then $
				    aPWI_time(i) = aPWI_time(i - 1) + sec_interval $
				else	aPWI_time(i) = start_sod
				PWI_spec_a(i, *) = 0.0
			    endelse
			    if (bspec_count GT 0) then $
				fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, $
					bspec_sum, boff_sum $
			    else begin
				if (i GT 0) then $
				    bPWI_time(i) = bPWI_time(i - 1) + sec_interval $
				else	bPWI_time(i) = start_sod
				PWI_spec_b(i, *) = 0.0
			    endelse
			    spec_done = 1
			endif
		    endif else if (spec_done) then begin
			if (i GT 0) then Fce_time(i) = Fce_time(i - 1) + sec_interval $
			else	Fce_time(i) = start_sod
			FCE(i, *) = 0.0
		    endif
		endif
		if (spec_done) then begin
		    limit = aPWI_time(i) + delta_t
;		    if (limit GT toffset - 15.0) then $
;			limit = toffset + sec_interval
		    i = i + 1
		    t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
			'Time overflow in sfr_spec', PWI_time, end_sod)
		    if (t LT 0) then	goto, bad_data $
		    else if (t LT 1) then	goto, Finished0
		    spec_done = 0
		endif
	    endif
	endif
	if (toffset GE limit) then begin
	    if (num_to_avg GT 2) then begin
		if (aspec_count GT 0) then $
		    fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, aoff_sum $
		else begin
		    if (i GT 0) then aPWI_time(i) = aPWI_time(i - 1) + sec_interval $
		    else	aPWI_time(i) = start_sod
		    PWI_spec_a(i, *) = 0.0
		endelse
		if (bspec_count GT 0) then $
		    fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum $
		else begin
		    if (i GT 0) then bPWI_time(i) = bPWI_time(i - 1) + sec_interval $
		    else	bPWI_time(i) = start_sod
		    PWI_spec_b(i, *) = 0.0
		endelse
		if (do_Fce) then begin
		    if (Fce_count GT 0) then $
			fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count $
		    else begin
			if (i GT 0) then Fce_time(i) = Fce_time(i - 1) + sec_interval $
			else	Fce_time(i) = start_sod
			FCE(i, *) = 0.0
		    endelse
		endif
	    endif
	    limit = aPWI_time(i) + delta_t
;	    if (limit GT toffset - 15.0) then $
;		limit = toffset + sec_interval
	    i = i + 1
	    t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
			'*Time overflow in sfr_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	    while (toffset GT limit) do begin

	        if (do_avg) then begin

		    if (num_to_avg GT 2) then begin
			if (i GT 0) then begin
			    aPWI_time(i) = aPWI_time(i - 1) + sec_interval
			    bPWI_time(i) = bPWI_time(i - 1) + sec_interval
			    if (do_Fce) then $
				Fce_time(i) = Fce_time(i - 1) + sec_interval
			endif else begin
			    aPWI_time(i) = start_sod
			    bPWI_time(i) = start_sod
			    if (do_Fce) then	Fce_time(i) = start_sod
			endelse
			PWI_spec_a(i, *) = 0.0
			PWI_spec_b(i, *) = 0.0
			if (do_Fce) then	FCE(i) = -1.0
			limit = aPWI_time(i) + delta_t
			i = i + 1
			t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
				'**Time overflow in sfr_spec', PWI_time, end_sod)
			if (t LT 0) then	goto, bad_data $
			else if (t LT 1) then	goto, Finished0
		    endif else begin
			for j = 0, 13, 2 do begin
			    if (i GT 0) then begin
				aPWI_time(i) = aPWI_time(i - 1) + add_interval
				bPWI_time(i) = bPWI_time(i - 1) + add_interval
				if (do_Fce) then $
				    Fce_time(i) = Fce_time(i - 1) + add_interval
			    endif else begin
				aPWI_time(i) = start_sod
				bPWI_time(i) = start_sod
				if (do_Fce) then	Fce_time(i) = start_sod
			    endelse
			    PWI_spec_a(i, *) = 0.0
			    PWI_spec_b(i, *) = 0.0
			    if (do_Fce) then	FCE(i) = -1.0
			    limit = aPWI_time(i) + delta_t
			    i = i + 1
			    t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
				'***Time overflow in sfr_spec', PWI_time, end_sod)
			    if (t LT 0) then	goto, bad_data $
			    else if (t LT 1) then	goto, Finished0
			endfor
		    endelse
		endif else begin	; num_to_avg = 2
		    for j = 0, 13, 2 do begin
			if (i GT 0) then begin
			    aPWI_time(i) = aPWI_time(i - 1) + add_interval
			    bPWI_time(i) = bPWI_time(i - 1) + add_interval
			    if (do_Fce) then $
				Fce_time(i) = Fce_time(i - 1) + add_interval
			endif else begin
			    aPWI_time(i) = start_sod
			    bPWI_time(i) = start_sod
			    if (do_Fce) then	Fce_time(i) = start_sod
			endelse
			PWI_spec_a(i, *) = 0.0
			PWI_spec_b(i, *) = 0.0
			if (do_Fce) then	FCE(i) = -1.0
			limit = aPWI_time(i) + delta_t
			i = i + 1
			t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
				'****Time overflow in sfr_spec', PWI_time, end_sod)
			if (t LT 0) then	goto, bad_data $
			else if (t LT 1) then	goto, Finished0
		    endfor
		endelse
		if (limit GT end_sod) then	goto, Finished
	    endwhile
	    if (toffset GT end_sod) then goto, Finished
	endif

	if (do_avg) then begin

	    if (num_to_avg GT 2) then begin

; Make 1 SFR spectrum
		if (SFRA_Ant(0) LT 2) then begin
		    for j = 0, 63, 2 do begin
			k = j + 1	&	m = j/2
			aPWI_spec(j) = SFR1A(m)
			aPWI_spec(k) = SFR1A(m)
			aPWI_spec(64 + j) = (SFR2A(m, 0) + SFR2A(m, 1))/2.0
			aPWI_spec(64 + k) = aPWI_spec(64 + j)
			aPWI_spec(128 + j) = (SFR3A(m, 0) + SFR3a(m, 1) + $
				SFR3a(m, 2) + SFR3a(m, 3))/4.0
			aPWI_spec(128 + k) = aPWI_spec(128 + j)
		    endfor
		    for j = 0, 63 do begin
			k = j + 192	&	m = k + 64
			for s = 0, 13 do begin
			    aPWI_spec(k) = aPWI_spec(k) + SFR4A(j, s)
			    aPWI_spec(m) = aPWI_spec(m) + SFR5A(j, s)
			endfor
			aPWI_spec(k) = aPWI_spec(k) /14.0
			aPWI_spec(m) = aPWI_spec(m) /14.0
		    endfor
		    aoff_sum = aoff_sum + toffset
		    aspec_count = aspec_count + 1
		    aspec_sum = aspec_sum + aPWI_spec(*)
		endif

		if (SFRB_Ant(0) GT 2) then begin
		    for j = 0, 63, 2 do begin
			k = j + 1	&	m = j/2
			bPWI_spec(j) = SFR1B(m)
			bPWI_spec(k) = SFR1B(m)
			bPWI_spec(64 + j) = (SFR2B(m, 0) + SFR2B(m, 1))/2.0
			bPWI_spec(64 + k) = bPWI_spec(64 + j)
			bPWI_spec(128 + j) = (SFR3B(m, 0) + SFR3B(m, 1) + $
				SFR3B(m, 2) + SFR3B(m, 3))/4.0
			bPWI_spec(128 + k) = bPWI_spec(128 + j)
		    endfor
		    for j = 0, 63 do begin
			k = j + 192	&	m = k + 64
			for s = 0, 13 do begin
			    bPWI_spec(k) = bPWI_spec(k) + SFR4B(j, s)
			    bPWI_spec(m) = bPWI_spec(m) + SFR5B(j, s)
			endfor
			bPWI_spec(k) = bPWI_spec(k) /14.0
			bPWI_spec(m) = bPWI_spec(m) /14.0
		    endfor
		    boff_sum = boff_sum + toffset
		    bspec_count = bspec_count + 1
		    bspec_sum = bspec_sum + bPWI_spec(*)
		endif

		if (do_Fce) then begin
		    Fce_Sum = Fce_sum + FCE_in(0)
		    foff_sum = foff_sum + toffset
		    Fce_count = Fce_count + 1
		endif

	    endif else begin

; Make 7 SFR spectra
		if (SFRA_Ant(0) LT 2) then begin
		    indx = 0
		    for s = 0, 13, 2 do begin
			m2 = fix(s/7)	&	m3 = fix(s/3.5)
			for j = 0, 63, 2 do begin
			    k = j + 1	&	m = j/2
			    aPWI_spec(j) = SFR1A(m)
			    aPWI_spec(k) = SFR1A(m)
			    aPWI_spec(64 + j) = SFR2A(m, m2)
			    aPWI_spec(64 + k) = SFR2A(m, m2)
			    aPWI_spec(128 + j) = SFR3A(m, m3)
			    aPWI_spec(128 + k) = SFR3A(m, m3)
			endfor
			for j = 0, 63 do begin
			    k = j + 192	&	m = k + 64
			    aPWI_spec(k) = (SFR4A(j, s) + SFR4A(j, s + 1))/2.0
			    aPWI_spec(m) = (SFR5A(j, s) + SFR5A(j, s + 1))/2.0
			endfor

			k = i + indx
			toffset = start_offset + s*add_interval + double(PB5(2)) / 1000.0
			aPWI_time(k) = toffset
			PWI_spec_a(k, *) = aPWI_spec
			t = err_check (k, max_components, 'r_pwi_sfr_spec_hr_local', $
				'*****Time overflow in sfr_spec', PWI_time, end_sod)
			if (t LT 0) then	goto, bad_data $
			else if (t LT 1) then	goto, Finished0
			indx = indx + 1
		    endfor
		endif

		if (SFRB_Ant(0) GT 2) then begin
		    indx = 0
		    for s = 0, 13, 2 do begin
			m2 = fix(s/7)	&	m3 = fix(s/3.5)
			for j = 0, 63, 2 do begin
			    k = j + 1	&	m = j/2
			    bPWI_spec(j) = SFR1B(m)
			    bPWI_spec(k) = SFR1B(m)
			    bPWI_spec(64 + j) = SFR2B(m, m2)
			    bPWI_spec(64 + k) = SFR2B(m, m2)
			    bPWI_spec(128 + j) = SFR3B(m, m3)
			    bPWI_spec(128 + k) = SFR3B(m, m3)
			endfor
			for j = 0, 63 do begin
			    k = j + 192	&	m = k + 64
			    bPWI_spec(k) = (SFR4B(j, s) + SFR4B(j, s + 1))/2.0
			    bPWI_spec(m) = (SFR5B(j, s) + SFR5B(j, s + 1))/2.0
			endfor

			k = i + indx
			toffset = start_offset + s*add_interval + double(PB5(2)) / 1000.0
			bPWI_time(k) = toffset
			PWI_spec_b(k, *) = bPWI_spec
			t = err_check (k, max_components, 'r_pwi_sfr_spec_hr_local', $
				'******Time overflow in sfr_spec', PWI_time, end_sod)
			if (t LT 0) then	goto, bad_data $
			else if (t LT 1) then	goto, Finished0
			indx = indx + 1
		    endfor
		endif

		if (do_Fce) then begin
; put in 1 real FCE and 6 dummies (to be removed later)
		    FCE(i) = FCE_in(0)
		    toffset = double(PB5(2))/1000.0 + start_offset
		    Fce_time(i) = toffset
		    for s = 1, 6 do begin
			k = i + s
			t = err_check (k, max_components, 'r_pwi_sfr_spec_hr_local', $
				'*******Time overflow in sfr_spec', PWI_time, end_sod)
			if (t LT 0) then	goto, bad_data $
			else if (t LT 1) then	goto, Finished0
			FCE(k) = -1.0
			Fce_time(k) = toffset + s*add_interval
		    endfor
		endif

		i = i + 7
		limit = toffset + delta_t
	    endelse	; num_to_avg = 2
	endif else begin	; no Avg

; Make 14 SFR spectra
	    if (SFRA_Ant(0) LT 2) then begin
		for s = 0, 13 do begin
		    m2 = fix(s/7)	&	m3 = fix(s/3.5)
		    for j = 0, 63, 2 do begin	; use SFR1, 2, 3 twice each
			    k = j + 1	&	m = j/2
			aPWI_spec(j) = SFR1A(m)
			aPWI_spec(k) = SFR1A(m)
			aPWI_spec(64 + j) = SFR2A(m, m2)
			aPWI_spec(64 + k) = SFR2A(m, m2)
			aPWI_spec(128 + j) = SFR3A(m, m3)
			aPWI_spec(128 + k) = SFR3A(m, m3)
		    endfor

		    for j = 0, 63 do begin	; use SFR4 & 5 once each
			k = j + 192	&	m = k + 64
			aPWI_spec(k) = SFR4A(j, s)
			aPWI_spec(m) = SFR5A(j, s)
		    endfor

		    k = i + s
		    toffset = start_offset + s*add_interval + double(PB5(2)) / 1000.0
		    aPWI_time(k) = toffset
		    PWI_spec_a(k, *) = aPWI_spec
		    t = err_check (k + 1, max_components, 'r_pwi_sfr_spec_hr_local', $
				'********Time overflow in sfr_spec', PWI_time, end_sod)
		    if (t LT 0) then	goto, bad_data $
		    else if (t LT 1) then	goto, Finished0
		endfor
	    endif else begin	; fill with 0's
		for index = 0, 13 do begin
		    k = i + indx
		    if (k GT 0) then $
			aPWI_time(k) = aPWI_time(k - 1) + add_interval $
		    else	aPWI_time(k) = start_sec
		    PWI_spec_a(k, *) = 0.0
		    t = err_check (k + 1, max_components, 'r_pwi_sfr_spec_hr_local', $
				'*********Time overflow in sfr_spec', PWI_time, end_sod)
		    if (t LT 0) then	goto, bad_data $
		    else if (t LT 1) then	goto, Finished0
		endfor
	    endelse

	    if (SFRB_Ant(0) GT 2) then begin
		for s = 0, 13 do begin
		    m2 = fix(s/7)	&	m3 = fix(s/3.5)
		    for j = 0, 63, 2 do begin	; use SFR1, 2, 3 twice each
			k = j + 1	&	m = j/2
			bPWI_spec(j) = SFR1B(m)
			bPWI_spec(k) = SFR1B(m)
			bPWI_spec(64 + j) = SFR2B(m, m2)
			bPWI_spec(64 + k) = SFR2B(m, m2)
			bPWI_spec(128 + j) = SFR3B(m, m3)
			bPWI_spec(128 + k) = SFR3B(m, m3)
		    endfor

		    for j = 0, 63 do begin	; use SFR4 & 5 once each
			k = j + 192	&	m = k + 64
			bPWI_spec(k) = SFR4B(j, s)
			bPWI_spec(m) = SFR5B(j, s)
		    endfor
		    k = i + s
		    toffset = start_offset + s*add_interval + double(PB5(2)) / 1000.0
		    bPWI_time(k) = toffset
		    PWI_spec_b(k, *) = bPWI_spec
		    t = err_check (k + 1, max_components, 'r_pwi_sfr_spec_hr_local', $
				'**********Time overflow in sfr_spec', PWI_time, end_sod)
		    if (t LT 0) then	goto, bad_data $
		    else if (t LT 1) then	goto, Finished0
		endfor
	    endif else begin	; fill with 0's
		for index = 0, 13 do begin
		    k = i + indx
		    if (k GT 0) then $
			bPWI_time(k) = bPWI_time(k - 1) + add_interval $
		    else	bPWI_time(k) = start_sec
		    PWI_spec_b(k, *) = 0.0
		    t = err_check (k + 1, max_components, 'r_pwi_sfr_spec_hr_local', $
				'***********Time overflow in sfr_spec', PWI_time, end_sod)
		    if (t LT 0) then	goto, bad_data $
		    else if (t LT 1) then	goto, Finished0
		endfor
	    endelse

	    if (do_Fce) then begin
; put in 1 real FCE and 13 dummies (to be removed later)
		FCE(i) = FCE_in(0)
		toffset = double(PB5(2))/1000.0 + start_offset
		Fce_time(i) = toffset
		for s = 1, 13 do begin
		    k = i + s
		    t = err_check (k, max_components, 'r_pwi_sfr_spec_hr_local', $
				'***********Time overflow in sfr_spec', PWI_time, end_sod)
		    if (t LT 0) then	goto, bad_data $
		    else if (t LT 1) then	goto, Finished0
		    FCE(k) = -1.0
		    Fce_time(k) = toffset + s*add_interval
		endfor
	    endif

	    i = i + 14;
	    limit = toffset + delta_t
	endelse


	prev_offset = toffset
	CDF_varget, CDF_ID, VarNum_PB5, PB5, Rec_Start = RecNum
	toffset = start_offset + double(PB5(2))/1000.0
	if (toffset GT end_sod) then Done = 1 $
	else begin
	    prev_ant = SFRA_Ant(0)
	    CDF_varget, CDF_ID, VarNum_AAnt, SFRA_Ant, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_BAnt, SFRB_Ant, Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR1A, SFR1A, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR2A, SFR2A, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR2_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR3A, SFR3A, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR3_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR4A, SFR4A, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR5A, SFR5A, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR1B, SFR1B, /ZVARIABLE, $
				Rec_Start = RecNum
	    CDF_varget, CDF_ID, VarNum_SFR2B, SFR2B, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR2_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR3B, SFR3B, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR3_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR4B, SFR4B, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	    CDF_varget, CDF_ID, VarNum_SFR5B, SFR5B, /ZVARIABLE, $
		Rec_Start = RecNum, count = SFR45_counts, interval = ainterval, $
		offset = roffset
	    RecNum = RecNum + 1
	    if (RecNum GE MaxRec) then begin
		if (end_sod EQ start_offset + sec_per_day) then goto, Finished
		CDF_close, CDF_ID

		cday = cday + 1
		if (cday GT month_tab(cmonth - 1)) then begin
		    cday = 1	&	cmonth = cmonth + 1
		    if (cmonth GT 12) then begin
			cmonth = 1	&	cyear = cyear + 1
		    endif
		endif
		date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
		tfile_name = 'po_h1_pwi_' + date_string + '_v01.cdf'
		file_ok = 0
		for t = 0, ccmax do $
		    if (result(t) EQ tfile_name) then	file_ok = 1

		file_name = PWI_path + tfile_name

		if (NOT file_ok) then begin
		    get_err_no = 1
		    get_err_msg = '*Unable to open ' + file_name + '.  File not found.'
		    if (end_sod - aPWI_time(i - 1) LE 60.0) then begin
			print, '% r_pwi_sfr_spec_hr_local: ', get_err_msg
			goto, Finished0
		    endif else	goto, bad_data
		    goto, bad_data
		endif
		print, file_name
		CDF_ID = CDF_open (file_name)

		CDF_Result = CDF_inquire (CDF_ID)
		MaxRec = CDF_Result.MaxRec
		MaxRec = MaxRec + 1
		start_offset = start_offset + Sec_per_day
		RecNum = 0L
	    endif
	endelse
    endwhile

    Finished: CDF_close, CDF_ID

    if (aspec_count GT 0) then begin
	fill_spec, PWI_spec_a, aPWI_time, i, aspec_count, aspec_sum, aoff_sum
	toffset = aPWI_time(i) + delta_t
	add_to_i = 1
    endif else	add_to_i = 0

    if (do_Fce) then begin
	if (Fce_count GT 0) then begin
	    fill_Fce, Fce_time, FCE, i, Fce_sum, foff_sum, Fce_count
	    if (NOT add_to_i) then begin
		add_to_i = 1
		aPWI_time(i) = toffset
		PWI_spec_a(i, *) = 0.0
	    endif
	endif else if (add_to_i) then begin
	    FCE(i) = -1.0
	    Fce_time(i) = toffset
	endif
    endif
    
    if (bspec_count GT 0) then begin
	fill_spec, PWI_spec_b, bPWI_time, i, bspec_count, bspec_sum, boff_sum
	if (NOT add_to_i) then begin
	    aPWI_time(i) = toffset
	    PWI_spec_a(i, *) = 0.0
	    if (do_Fce) then begin
		FCE(i) = -1.0
		Fce_time(i) = toffset
	    endif
	    toffset = bPWI_time(i) + delta_t
	endif
	i = i + 1
	t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
			'************Time overflow in sfr_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
    endif else if (add_to_i) then begin
	bPWI_time(i) = toffset
	PWI_spec_b(i, *) = 0.0
	toffset = bPWI_time(i) + delta_t
	i = i + 1
	t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
			'*************Time overflow in sfr_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
    endif

    if (NOT Done) then begin
	if (end_sod - toffset GT delta_t) then begin
	    toffset = aPWI_time(i - 1) + delta_t
	    while (end_sod - toffset GT delta_t) do begin
		aPWI_time(i) = toffset
		bPWI_time(i) = toffset
		PWI_spec_a(i, *) = 0.0
		PWI_spec_b(i, *) = 0.0
		if (do_Fce) then begin
		    FCE(i) = -1.0
		    Fce_time(i) = toffset
		endif
		toffset = toffset + delta_t
		i = i + 1
		t = err_check (i, max_components, 'r_pwi_sfr_spec_hr_local', $
			'*************Time overflow in sfr_spec', PWI_time, end_sod)
		if (t LT 0) then	goto, bad_data $
		else if (t LT 1) then	goto, Finished0
	    endwhile
	endif
    endif

    Finished0:	nt = i	&	nf = 320L
    end_indx = nt - 1

    print, '% r_pwi_sfr_spec_hr_local:  spec dimensions ', nt, ' x ', nf

    time0 = { PB5:lonarr(3), t90:0.0D }

    if (Ant EQ 0) then $
	ttitle = 'PWI SFR A/Eu' $
    else if (Ant EQ 1) then $
	ttitle = 'PWI SFR A/Ez' $
    else if (Ant EQ 2) then $
	ttitle = 'PWI SFR A/Eu/Ez' $
    else if (Ant EQ 3) then $
	ttitle = 'PWI SFR A/Ez/Eu' $
    else 	ttitle = 'PWI SFR A'

    SFRAspctrm = { nt:0L, nf:nf, title:ttitle, ytitle:'Frequency <Hz>', $
		ztitle:'(V/m)!e2!n/Hz', frequency:fltarr(nf), $
		spec:fltarr(nt, nf), time:replicate(time0, nt) }

    SFRBspctrm = { nt:0L, nf:nf, title:ttitle, ytitle:'Frequency <Hz>', $
		ztitle:'nT!e2!n/Hz', frequency:fltarr(nf), $
		spec:fltarr(nt, nf), time:replicate(time0, nt) }

; use 00:00:00 here because aPWI_time() is in SOD
    t90base = convert_timetoseconds(string (smonth, '/', sday, '/', syear, $
		' 00:00:00', format = '(i2.2, a1, i2.2, a1, i4.4, a9)'), /t90)

    print, '% r_pwi_sfr_spec_hr_local: converted time to T90'

    if (do_Fce) then begin
	index = where (FCE(0:end_indx) GT 1.0, kount)
	if (kount GT 0) then begin
	    nt_f = kount
	    FCE_lp = { nt:0L, title:'Polar Fce', ytitle:'Frequency <Hz>', $
			fce:fltarr(nt_f), time:replicate(time0, nt_f) }
	    FCE_lp.nt = nt_f
	    FCE_lp.time(*).PB5(0) = syear
	    FCE_lp.time(*).PB5(1) = sdoy
	    FCE_lp.time(*).PB5(2) = Fce_time(index)
	    FCE_lp.time(*).T90 = Fce_time(index) + t90base.value
	    FCE_lp.fce = FCE(index)
	endif else begin
	    get_err_no = 4
	    get_err_msg = '**No Fce Data found'
	endelse
    endif

    max = nt - 1
    for j = 0L, max do begin
	SFRAspctrm.time(j).PB5(0) = syear
	SFRBspctrm.time(j).PB5(0) = syear
	SFRAspctrm.time(j).PB5(1) = sdoy
	SFRBspctrm.time(j).PB5(1) = sdoy
	SFRAspctrm.time(j).PB5(2) = aPWI_time(j)
	SFRBspctrm.time(j).PB5(2) = bPWI_time(j)
;change time formats to t90
	SFRAspctrm.time(j).t90 = aPWI_time(j) + t90base.value
	SFRBspctrm.time(j).t90 = bPWI_time(j) + t90base.value
	SFRAspctrm.spec(j, *) = PWI_spec_a(j, *)
	SFRBspctrm.spec(j, *) = PWI_spec_b(j, *)
    endfor

    SFRAspctrm.nt = nt	&	SFRBspctrm.nt = nt

; once you are herel, the data has been correctly read. At this point you
; may restrict the arrays passed back to the time chosen using xut1 and xut2,
; you need to convert the time information to T90 (use the routines in
; $PAPCO_BASE/PAPCO_Lib/convert.pro) - if you intend to use the PAPCO
; plotting templates for your data. If you use your own plot
; routines this step is not needed, and you also don't have to re-save the
; converted file.

    if (get_err_no EQ 0) then	get_err_msg = 'pwi_sfra_spec read successfully'
    print, '% r_pwi_sfr_spec_hr_local: ', get_err_msg

    return

    bad_data:	print, '% r_pwi_sfr_spec_hr_local: ', get_err_msg
	print, 'No PWI data returned'

end

;****************************************************************************
;*  NAME:        r_pwi_sfra_spec
;*
;*  DESCRIPTION: Routine to read SFR A data over the internet. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90.  Calls r_pwi_sfra_spec_sv or r_pwi_sfra_spec_hr
;*		 to do the actual reading.
;*
;*  INPUT:       do_local from draw_pwi_sfra_spec
;*
;*  OUTPUT:      none - writes data to common blocks
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_sfra_spec, do_local

    COMMON SFRAspctrmstuff, SFRAspctrm
    COMMON SFRBspctrmstuff, SFRBspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    get_err_no = 0	&	get_err_msg = ''

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    sec_per_day = 86400.0

   if (do_local) then begin
	file_name = ''

	PWI_path = getenv ('PWI_DATA')
	L = strlen(PWI_path)
	if (L EQ 0) then begin
		get_err_no = 1
		get_err_msg = 'Path to PWI data not found.  Env. variable Not set'
		goto, bad_data
	endif else if (strmid(PWI_path, L - 1, 1) NE '/') then $
		PWI_path = PWI_path + '/'
	cyear = syear	&	cmonth = smonth	&	cday = sday
	if (syear mod 4 EQ 0) then	m2 = 29 $
	else	m2 = 28
	month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]
	date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')
	result = findfile(PWI_PATH, count = cc)
	ccmax = cc - 1
	sv_ok = 1

	if (total_sec GE 10800) then begin
	    file_name = 'po_pwi_sfr_' + date_string + '_v01.sv.cdf'
	    found = 0
	    for t = 0, ccmax do $
		if (result(t) EQ file_name) then found = 1

	    if (NOT found) then begin
		sv_ok = 0
		goto, do_HR
	    endif
	    r_pwi_sfr_spec_sv_local
	endif else begin
do_HR:
	    file_name = 'po_h1_pwi_' + date_string + '_v01.cdf'
	    for t = 0, ccmax do $
		if (result(t) EQ file_name) then	found = 1

	    if (NOT found) then	goto, do_survey

	    r_pwi_sfr_spec_hr_local
	    return

do_survey:  if (sv_ok) then begin
		print, '*Unable to open ' + file_name + '.  File not found.'
		print, 'Calling survey reader'
		r_pwi_sfr_spec_sv_local
	    endif else begin
		get_err_msg = 'No data found for ' + date_string
		goto, bad_data
	    endelse
	endelse
   endif else if (total_sec GT 4500.0) then $
	r_pwi_sfra_spec_sv_net $
   else	r_pwi_sfra_spec_hr_net

   return

bad_data:	print, 'r_pwi_sfra_spec: ', get_err_msg
		print, '*NO PWI data returned'
end

