
;****************************************************************************
;*  NAME:        r_pwi_sfrb_spec_sv_net
;*
;*  DESCRIPTION: Routine to read sfr data generated by the survey reader
;*		 via the internet. Adheres to PAPCO conventions: Error common
;*		 block, changes time format to T90
;*
;*  INPUT:       none - gets data from common blocks
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_sfrb_spec_sv_net

   COMMON SFRBspctrmstuff, SFRBspctrm
   COMMON time, xut1, xut2
   COMMON get_error, get_err_no, get_err_msg

   max_components = 2000	&	get_err_no = 0
   PWI_freq = fltarr (320)
   PWI_spec = fltarr (max_components, 320)
   PWI_time = dblarr (max_components)

   ideal_components = 1000	&	fill_val = 1.0e-32
   SFR_interval = 33.0		&	spect_sec = 2.3552

   aPWI_offset = 0.0		&	prev_offset = 0.0

   aPWI_spec = fltarr (320)	&	start_offset = 0.0

   PWI_freq = 10^(findgen (320)/319.0*4.47966 + 1.42765)

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

; the PWI read routines all use Gnu wget to retrieve calibrated spectral
; densities via the internet, eliminating the need for files.

; the latest version of wget (as of 05/97) is available at
;       <URL:ftp://prep.ai.mit.edu/pub/gnu/wget-1.4.3.tar.gz>.
; if wget is installed in the system path, you should not need to do
; anything special to get this procedure to work.

    total_sec = xut2 - xut1

   convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
			scmonth, sday
   start_sod = ssecond + 60.0*(sminute + 60.0*shour)

   convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
			ecmonth, eday
   end_sod = start_sod + total_sec

    total_spec = long(total_sec / spect_sec + 0.5)
    if (total_spec LT 1) then	total_spec = 1
    num_to_avg = total_spec / ideal_components
    if (num_to_avg LT 1) then	num_to_avg = 1
    sec_interval = num_to_avg * spect_sec

    if (sec_interval LT SFR_interval) then	sec_interval = SFR_interval

    sec_incr = sec_interval / 2.0
    if (sec_incr LE SFR_interval) then	sec_incr = 0.0

    delta_t = 1.25 * sec_interval

    start_offset = start_sod	&	limit = delta_t
 
    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
                sminute, ':', ssecond, $
        format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')
    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
                eminute, ':', esecond, $
        format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')
 
    command = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=papco_sfrb&begin_time= ' + $
                beg_timestr + '&end_time= ' + end_timestr + '"'

    print, command
    spawn, command, unit=lun

    PWI_time(*) = 0.0

    DataOK = 1	&	i = 0L

    readu, lun, aPWI_offset, aPWI_spec
    on_ioerror, bad_data0

    while (DataOK) do begin
	prev_offset = aPWI_offset

	while (aPWI_offset GT limit) do begin
	    if (i GT 0) then $
		PWI_time(i) = PWI_time(i - 1) + sec_interval $
	    else	PWI_time(i) = start_offset
	    PWI_spec(i, *) = 0.0
	    i = i + 1	&	limit = limit + sec_interval
	    if (i GT max_components) then begin
		print, '% r_pwi_sfrb_spec_sv_net:  need more columns (i = ', i, ')'
		get_err_no = 2
		get_err_msg = '**Time overflow in pwi_sfrb_spec'
		if (end_sod - PWI_time(i - 1) LE 60.0) then begin
		    print, get_err_msg
		    goto, Finished0
		endif else	goto, bad_data
	    endif
	endwhile

	PWI_time(i) = start_offset + aPWI_offset
	PWI_spec(i, *) = aPWI_spec
	i = i + 1	&	limit = aPWI_offset + delta_t
	t = err_check (i, max_components, 'r_pwi_sfrb_spec_sv_net', $
		'Time overflow in pwi_sfrb_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0

	readu, lun, aPWI_offset, aPWI_spec
	on_ioerror, Finished
    endwhile

   Finished: close, lun	&	free_lun, lun
   if (i GT 1) then	tdiff = PWI_time(i - 1) - PWI_time(i - 2) $
   else	tdiff = 0.0

   if (prev_offset LT total_sec) then begin
	prev_offset = prev_offset + sec_interval
	while (prev_offset LT total_sec) do begin
	    PWI_time(i) = start_offset + prev_offset
	    PWI_spec(i, *) = 0.0
	    prev_offset = prev_offset + sec_interval
	    i = i + 1
	    t = err_check (i, max_components, 'r_pwi_sfrb_spec_sv_net', $
		'*Time overflow in pwi_sfrb_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
   endif

   Finished0:	nt = i	&	nf = n_elements (PWI_freq)

   print, '% r_pwi_sfrb_spec_sv_net:  spec dimensions ', nt, ' x ', nf
   time0 = { PB5:lonarr(3), t90:0.0D }

   SFRBspctrm = { nt:0L, nf:0L, title:'PWI SFR B/Loop', $
		ytitle:'Frequency <Hz>', ztitle:'nT!e2!n/Hz', $
		frequency:fltarr(nf), spec:fltarr(nt, nf), $
		time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time() is in SOD
   t90base = convert_timetoseconds(string (smonth, '/', sday, '/', syear, $
		' 00:00:00', format = '(i2.2, a1, i2.2, a1, i4.4, a9)'), /t90)

   k = nt - 1
   for j = 0L, k do begin
	SFRBspctrm.time(j).PB5(0) = syear
	SFRBspctrm.time(j).PB5(1) = sdoy
	SFRBspctrm.time(j).PB5(2) = PWI_time(j)
;change time formats to t90
	SFRBspctrm.time(j).t90 = PWI_time(j) + t90base.value
	SFRBspctrm.spec(j, *) = PWI_spec(j, *)
   endfor

   SFRBspctrm.nt = nt	&	SFRBspctrm.nf = nf

   print, '% r_pwi_sfrb_spec_sv_net: converted time to T90'

   SFRBspctrm.frequency = PWI_freq

   if (get_err_no EQ 0) then $
	get_err_msg = '**pwi_sfrb_spec read succesfully'

   print, '% r_pwi_sfrb_spec_sv_net: ', get_err_msg

   return

   bad_data0:	get_err_no = 1
		get_err_msg = '**NO Network data returned'

   bad_data:
	print, get_err_msg
	print, '**No PWI Data Returned'

end

;****************************************************************************
;*  NAME:        r_pwi_sfrb_spec_hr_net
;*
;*  DESCRIPTION: Routine to read high resolution sfr B data via the
;* 		 internet. Adheres to PAPCO conventions: Error common block,
;*		 changes time format to T90
;*
;*  INPUT:       none - gets data from common blocks
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_sfrb_spec_hr_net

    COMMON SFRBspctrmstuff, SFRBspctrm
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    max_components = 6000	&	get_err_no = 0
    PWI_freq = fltarr (320)
    PWI_spec = fltarr (max_components, 320)
    PWI_time = dblarr (max_components)

    ideal_components = 4075.0	&	fill_val = 1.0e-32

    aPWI_offset = 0.0	&	prev_offset = 0.0

    aPWI_spec = fltarr (320)

    spec_sum = fltarr (320)	&	start_offset = 0.0

    PWI_freq = 10^(findgen (320)/319.0*4.47966 + 1.42765)

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

; the PWI read routines all use Gnu wget to retrieve calibrated spectral
; densities via the internet, eliminating the need for files.

; the latest version of wget (as of 05/97) is available at
;       <URL:ftp://prep.ai.mit.edu/pub/gnu/wget-1.4.3.tar.gz>.
; if wget is installed in the system path, you should not need to do
; anything special to get this procedure to work.

    total_sec = xut2 - xut1

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
				scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
				ecmonth, eday
    end_sod = start_sod + total_sec

    if (total_sec GT 4500) then begin
	print, '	**Time range too large for HR reader'
	return
    endif

    sec_interval = 2.3552		&	delta_t = 2.0 * sec_interval

    sec_incr = sec_interval / 2.0
    if (sec_incr LE 2.36) then	sec_incr = 0.0

    start_offset = start_sod	&	limit = delta_t
 
    beg_timestr = string (syear, '-', smonth, '-', sday, 'T', shour, ':', $
                sminute, ':', ssecond, $
        format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')
    end_timestr = string (eyear, '-', emonth, '-', eday, 'T', ehour, ':', $
                eminute, ':', esecond, $
        format = '(i4.4, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2, a1, i2.2)')
 
    command = 'wget -O - "http://www-pw.physics.uiowa.edu/cgi-test/papco.cgi?' $
		+ 'submit=process&data_set=sfrb_L&begin_time= ' + $
                beg_timestr + '&end_time= ' + end_timestr + '"'

    print, command
    spawn, command, unit=lun

    PWI_time(*) = 0.0

    spec_count = 0L	&	spec_sum(*) = 0.0	&	off_sum = 0.0

    DataOK = 1	&	i = 0L

    readu, lun, aPWI_offset, aPWI_spec
    on_ioerror, bad_data0

    while (DataOK) do begin
	prev_offset = aPWI_offset
	while (aPWI_offset GT limit) do begin
	    if (i GT 0) then $
		PWI_time(i) = PWI_time(i - 1) + sec_interval $
	    else	PWI_time(i) = start_offset
	    PWI_spec(i, *) = 0.0
	    i = i + 1	&	limit = limit + sec_interval
	    t = err_check (i, max_components, 'r_pwi_sfrb_spec_hr_net', $
		'*Time overflow in pwi_sfrb_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
	PWI_spec(i, *) = aPWI_spec
	PWI_time(i) = start_offset + aPWI_offset
	i = i + 1	&	limit = aPWI_offset + delta_t
	t = err_check (i, max_components, 'r_pwi_sfrb_spec_hr_net', $
		'**Time overflow in pwi_sfrb_spec', PWI_time, end_sod)
	if (t LT 0) then	goto, bad_data $
	else if (t LT 1) then	goto, Finished0
	readu, lun, aPWI_offset, aPWI_spec
	on_ioerror, Finished
    endwhile

    Finished:

    if (prev_offset LT total_sec) then begin
	prev_offset = prev_offset + sec_interval
	while (prev_offset LT total_sec) do begin
	    if (i GT 0) then $
		    PWI_time(i) = PWI_time(i - 1) + sec_interval $
	    else	goto, bad_data0
	    PWI_spec(i, *) = 0.0
	    i = i + 1	&	prev_offset = prev_offset + sec_interval
	    t = err_check (i, max_components, 'r_pwi_sfrb_spec_hr_net', $
		'***Time overflow in pwi_sfrb_spec', PWI_time, end_sod)
	    if (t LT 0) then	goto, bad_data $
	    else if (t LT 1) then	goto, Finished0
	endwhile
    endif

    Finished0:	nt = i	&	nf = n_elements (PWI_freq)

    print, '% r_pwi_sfrb_spec_hr_net:  spec dimensions ', nt, ' x ', nf
    time0 = { PB5:lonarr(3), t90:0.0D }

    SFRBspctrm = { nt:0L, nf:0L, title:'PWI SFR B/Loop', $
		ytitle:'Frequency <Hz>', ztitle:'nT!e2!n/Hz', $
		frequency:fltarr(nf), spec:fltarr(nt, nf), $
		time:replicate(time0, nt) }

; use 00:00:00 here because PWI_time() is in SOD
    t90base = convert_timetoseconds(string (smonth, '/', sday, '/', syear, $
		' 00:00:00', format = '(i2.2, a1, i2.2, a1, i4.4, a9)'), /t90)

    for j = 0L, i - 1 do begin
	SFRBspctrm.time(j).PB5(0) = syear
	SFRBspctrm.time(j).PB5(1) = sdoy
	SFRBspctrm.time(j).PB5(2) = PWI_time(j)
;change time formats to t90
	SFRBspctrm.time(j).t90 = PWI_time(j) + t90base.value
    endfor

    SFRBspctrm.nt = nt	&	SFRBspctrm.nf = nf

    print, '% r_pwi_sfrb_spec_hr_net: converted time to T90'

    SFRBspctrm.frequency = PWI_freq

    k = nt - 1
    for j = 0L, k do $
	SFRBspctrm.spec(j, *) = PWI_spec(j, *)

    if (get_err_no EQ 0) then $
	get_err_msg = '**pwi_sfrb_spec read succesfully'

    print, '% r_pwi_sfrb_spec_hr_net: ', get_err_msg

    return

    bad_data0:	get_err_no = 1
		get_err_msg = '**NO Network data returned'

    bad_data:
	print, get_err_msg
	print, '**No PWI Data Returned'

end

;****************************************************************************
;*  NAME:        r_pwi_sfrb_spec
;*
;*  DESCRIPTION: Routine to read SFR A data over the internet. Adheres to
;*               PAPCO conventions: Error common block, changes time format
;*               to T90.  Calls r_pwi_sfrb_spec_sv or r_pwi_sfrb_spec_hr
;*		 to do the actual reading.
;*
;*  INPUT:       do_local from draw_pwi_sfrb_spec
;*
;*  OUTPUT:      none - places data in common blocks
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES: 
;****************************************************************************
pro r_pwi_sfrb_spec, do_local

    COMMON SFRAspctrmstuff, SFRAspctrm
    COMMON SFRBspctrmstuff, SFRBspctrm
    COMMON FCEstuff, FCE_lp
    COMMON time, xut1, xut2
    COMMON get_error, get_err_no, get_err_msg

    get_err_no = 0	&	get_err_msg = ''

    total_sec = xut2 - xut1
print, xut1, xut2, total_sec

    convert_t90_to_date, xut1, syear, sdoy, shour, sminute, ssecond, smonth, $
					scmonth, sday
    start_sod = ssecond + 60.0*(sminute + 60.0*shour)

    convert_t90_to_date, xut2, eyear, edoy, ehour, eminute, esecond, emonth, $
					ecmonth, eday
    end_sod = start_sod + total_sec

    sec_per_day = 86400.0

   if (do_local) then begin
	file_name = ''

	PWI_path = getenv ('PWI_DATA')
	L = strlen(PWI_path)
	if (L EQ 0) then begin
	    get_err_no = 1
	    get_err_msg = 'Path to PWI data not found.  Env. variable Not set'
	    goto, bad_data
	endif else if (strmid(PWI_path, L - 1, 1) NE '/') then $
	    PWI_path = PWI_path + '/'
	cyear = syear	&	cmonth = smonth	&	cday = sday
	if (syear mod 4 EQ 0) then	m2 = 29 $
	else	m2 = 28
	month_tab = [ 31, m2, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ]
	date_string = string (cyear, cmonth, cday, format = '(i4.4, i2.2, i2.2)')

	result = findfile (PWI_PATH, count = cc)
	ccmax = cc - 1
	sv_ok = 1

	if (total_sec GE 10800) then begin
	    file_name = 'po_pwi_sfr_' + date_string + '_v01.sv.cdf'

	    found = 0
	    for i = 0, ccmax do $
		if (result(i) EQ file_name) then	found = 1

	    if (found) then begin
		sv_ok = 1
		r_pwi_sfr_spec_sv_local
	    endif else begin
		sv_ok = 0
		goto, do_hires
	    endelse
	endif else begin
	    sv_ok = 1
	    do_hires:

	    file_name = 'po_h1_pwi_' + date_string + '_v01.cdf'

	    found = 0
	    for i = 0, ccmax do $
		if (result(i) EQ file_name) then	found = 1

	    if (NOT found) then goto, do_survey

	    r_pwi_sfr_spec_hr_local
	    return

do_survey:  if (sv_ok) then begin
		print, '*Unable to open ' + file_name + '.  File not found.'
		print, 'Calling survey reader'
		r_pwi_sfr_spec_sv_local
	    endif else begin
		get_err_msg = 'NO data found for ' + date_string
		goto, bad_data
	    endelse
	endelse
   endif else if (total_sec GT 4500.0) then $
	r_pwi_sfrb_spec_sv_net $
   else	r_pwi_sfrb_spec_hr_net

   return

bad_data:	print, get_err_msg
		print, '*NO PWI data returned'
end

