;******************************************************************************
;* file:  papcoadd_tide.pro
;*        papcoadd_tide.pro
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a tide. 
;*
;* Whenever PAPCO has to undertake some action for a
;* tide, the field plotInfo.panelkind is compared to the entries in the
;* (COMMON ADDINS) tideS array. If a match is found, the required
;* procedure name is constructed and called If not, an error occurs.
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_tide,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_tide, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO descriptor_tide, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_tide
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_tide, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           PAPCO_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_tide, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_tide,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this tide to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      tides.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_tide,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified in September 1997 for use in plotting TIDE data
;******************************************************************************
pro addplottype_tide,plottypes_base,buttonGeometry,pb_tide

; VALUE assigns a label for the button
; The tide portion of UVALUE is used to construct the calls to all
; other procedures in this add-in needed by PAPCO.

; simple button use to start tide plot

  pb_tide = WIDGET_BUTTON(plottypes_base,  $
                                 VALUE=strcenter('POLAR_TIDE',14), $
                                 UVALUE='pb_tide', $
                                 _extra=buttonGeometry)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_tide, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by CRRES orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_tide, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_tide, plottable

   plottable = 1   ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_tide, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this tide. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_tide, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_tide, plotInfo, descriptStr

;  if PlotInfo.typeVector(0) ne 0 then $
;     descriptStr=descriptStr+'Lineplot Channel : ' + $
;       strtrim(string(plotInfo.typevector(0)), 2) $
;  else $
;     descriptStr=descriptStr+'Spectrum plot'

      case PlotInfo.typeVector(0) of
         0: str1 = 'Density'
         1: str1 = 'T-Para'
         2: str1 = 'T-Perp'
         3: str1 = 'Vx'
         4: str1 = 'Vy'
         5: str1 = 'Vz'
         6: str1 = 'Energy'
         7: str1 = 'Spin Angle'
         8: str1 = 'Polar Angle'
         9: str1 = 'S/C Pot (EFI K0)'
      else: str1 = '?'
      endcase

      case PlotInfo.typeVector(1) of
         0: str2 = ' Total Ions'
         1: str2 = ' H+'
         2: str2 = ' O+'
         3: str2 = ' He+'
      else: str2 = ' ?'
      endcase

      descriptStr = str1 + str2

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_tide
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the PAPCO_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the tide.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_tide
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_tide,event

  panelKind='tide'
  panelEditor='tide_panel'
  defaults=PAPCO_getPlotInfoStruct()  ; returns a basic plotinfo structure
  defaults.typeVector=[0,0,0,0]       ; set some tags to desired defaults
  defaults.channel=0                  ; channel tag is used for the color table
  PAPCO_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_tide, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      PAPCO_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several tide plotss to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for tide.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_tide, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_tide, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=tide_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_tide, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the PAPCo core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of PAPCO functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure PAPCO_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 switches. Currently
;*                      Three are supported (this may be extended later)
;*                        1 = mouse zooms (utility slice, always possible)
;*                        2 = mouse slices
;*                        4 = mouse calls level zero
;*                        8 = mouse views slice (polar)  
;*                       16 = mouse views ephemeris
;*                       32 = mouse writes paneldata (utility slice)
;*                       64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      draw_tide, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_tide, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead

   COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

; as papco dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
  
  slice_assigned_value=papco_slice_value('USER Defined Slice A')
  
; set up which mouse functions to allow

   get_data_call="r_tide"   ; data read call used by slices
   slice_type = 1 + 2 + 16   ; Zoom + Slice + Hair Cursor

   ptype = currPlotInfo.typeVector(0) * 10 + currPlotInfo.typeVector(1) 

   IF NOT IsRead THEN BEGIN      ; check if data has already been read

      r_tide                     ; read data

      ; check the read for success.

      IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) $
      THEN BEGIN
        IsRead=1                ; set data as read, avoids re-read for zooming
        
        if(currPlotInfo.typeVector(0) lt 6 or currPlotInfo.typeVector(0) eq 9) $
          then begin
          p_tide_line, panelVector, ptype, _EXTRA=extra_PlotInfo
        endif else begin
          p_tide_spec, panelVector, ptype, _EXTRA=extra_PlotInfo
        endelse
      ENDIF
   ENDIF ELSE BEGIN
         
         if(currPlotInfo.typeVector(0) lt 6 or $
            currPlotInfo.typeVector(0) eq 9) then begin
           p_tide_line, panelVector, ptype, _EXTRA=extra_PlotInfo
         endif else begin
           p_tide_spec, panelVector, ptype, _EXTRA=extra_PlotInfo
         endelse
   ENDELSE

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot switch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END



