;******************************************************************************
;* FILE:
;*    $PAPCO_XX/PAPCO_templates/tide_panel.pro
;*
;*    This file belongs to the PAPCO-project. See $PAPCO_XX/papco.pro for
;*    further information.
;*
;* DESCRIPTION:
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by
;*    $PAPCO_XX/papco_panel.pro (a collection of standard PAPCO widgets)
;*
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ (under $PAPCO_XX/PAPCO_doc)
;*    papco_conventions.text - which conventions were used when developing ?
;*    papco_variables.text   - what structures and global variables
;                              are used y bwhich file ?
;*
;* COMMON-BLOCKS:
;*    PAPCO_PANEL_TEMPLATE_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_XX/papco_panel.pro for a
;*    description).
;*       {PAPCO_PANEL_TEMPLATE_DATA,
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*
;*    For a more complete description see $PAPCO_DOC/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  PAPCO_PANEL_TEMPLATE, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*       a=PAPCO_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information
;*	 a=tide_panel(a)
;*       which calls up the window.
;*       Interrogating a with help,a,/str shows the contents. This is a good
;*       way of testing your own panel editor.
;*
;* MODIFICATION HISTORY:
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:
;*      FUNCTION  tide_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*
;* DESCRIPTION:
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:
;*      aPlotInfo       a structure of the PAPCO_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type.
;*      All the functions in $PAPCO_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $
;*                      switch:[0,0,0], $
;*	                fileName:'', $       
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle,
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $
;*	                xut1:0.0, $
;*                      xut2:0.0}
;*
;*
;*      For an enhanced description read
;*                          $PAPCO_XX/PAPCO_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  tide_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON tide_panel_DATA, widgetData
   COMMON mjdt, mjdt_start, mjdt_end

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------

   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
       column=1, $
       GROUP_LEADER=group)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionality that PAPCO supports.
; They dynamically extend the panel editors widget structure, and all the
; events they generate are supported by PAPCO_panel_Event, transparent to the
; user.

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

; --- Choose plot type------------------------------------------------------

   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=2, /EXCLUSIVE)
   typeNames=['ion density           ', $
              'ion T-para            ', $
              'ion T-perp            ', $
              'ion Vx                ', $
              'ion Vy                ', $
              'ion Vz  (H+,O+,He+ only)      ', $
              'energy spectrogram      ', $
              'spin angle spectrogram  ', $
              'polar angle spectrogram ', $
              'Spacecraft Potential (EFI K0)        ']
;   typeNames=['ion density (Y-log)   ', $
;              'ion T-para (Y-log)    ', $
;              'ion T-perp (Y-log)    ', $
;              'ion Vx (Y-lin)        ', $
;              'ion Vy (Y-lin)        ', $
;              'ion Vz (Y-lin) (H+,O+,He+ only)      ', $
;              'energy spectrogram (Y-log,Z-log)     ', $
;              'spin angle spectrogram (Y-lin,Z-log) ', $
;              'polar angle spectrogram (Y-lin,Z-log)', $
;              'Spacecraft Potential (EFI K0)        ']
   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(0)), SET_BUTTON=1

; --- Choose ion ------------------------

   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=6, /EXCLUSIVE)
   typeNames=['Total Ions after 12/07/96 else   ', $
              'H+   ', $
              'O+   ', $
              'He+']

   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
     spb_type(i)=WIDGET_BUTTON(type_base, $
                               VALUE=typeNames(i), $
                               UVALUE='spb_ion'+strtrim(string(i), 2) )
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(1)), SET_BUTTON=1

; Jahn 10/11/99
;
; If data are from after they only contain total ion information. So *if* the
; data information is available, make sure only the valid buttons are
; shown. Note, however, that at first startup of PaPCo the date of the desired
; data is usually *not* passed. In that case leave all buttons availabel and
; make a check later.


; if mjdt_start and mjdt_end are not defined, we have to catch that and give
; them a value which will not be represented in the data

catch, tide_error_status

if tide_error_status ne 0 then begin
    ; set time for Jan 01, 1999, 00:00:00 UT; way before TIDE flew
    mjdt_start = {papco_time, mjd:0L, t:0.0, t90:0L}
    mjdt_end = {papco_time, mjd:0L, t:0.0, t90:0L}
endif

convert_t90_to_Date, mjdt_start.t90, year, day_oy, hour, minute, $
  second,month, cmonth, day_om 

; now we have two conditions:
;   1.  if mjdt_start LT 50424l, then we have start foild data (= 3D distr.);
;       otherwise we have stops only (i.e. mjdt_start ge 50424L
;   2.  if mjdt_start eq 0, then we have no date information yet; in this case
;       we will default to total ions, but will *not* blank out any buttons.
;

;
;   IF (mjdt_start LT 50424l) AND $
;      (aPlotInfo.typeVector(1) EQ 0) THEN aPlotInfo.typeVector(1)=1
; WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(1)), SENSITIVE = 0  
; end edit


   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(1)), SET_BUTTON=2

;--------------------  overplot s/c potential -----------------------

;    type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=6, /EXCLUSIVE)
;    typeNames=['Overplot s/c potential in Volts', $
;               'no s/c potential']

;    spb_type=lonarr(N_ELEMENTS(typeNames))

;    FOR i=0, N_ELEMENTS(typeNames)-1 DO $
;      spb_type(i)=WIDGET_BUTTON(type_base, $
;                                VALUE=typeNames(i), $
;                                UVALUE='s_c_pot_'+strtrim(string(i), 2) )
;    WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(2)), SET_BUTTON=1


;--------------------  end overplot s/c potential -------------------


;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;

; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $PAPCO_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
; In this way the widget is always reinitialized to its last state.

; The next four PAPCO panel widgets will be added only if you want to
; support that functionality, and if they make sense for your tide.

;--- add the PAPCO panel widget for choosing the color bar for color plots ----
   widgets=PAPCO_panel_AddColorBar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the  PAPCO panel widget for choosing the plot-style for line plots ---
   widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widget for choosing the y-axis-ranges ----------------
   widgets=PAPCO_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widgets for choosing the z-axis-ranges ---------------
   widgets=PAPCO_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the tide example module) and describe
;    your data and your choices in your widget. Put in here also the ID's of
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpfile:'tide_panel.help', $
                    typevector:aPlotInfo.typevector}
   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the
; data contained in aPlotInfo would be returned.
   xmanager, 'tide_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the
; tags provided in aPlotInfo, it makes life easier.

;  returnData={typevector:widgetData.typevector, $
;              switch:widgetData.switch}
   returnData={typevector:widgetData.typevector}
   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:
;*      PRO tide_panel_event, event
;*
;* DESCRIPTION:
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:
;*      event            an XManager event
;*
;* KEYWORDS:
;*       none
;*
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO tide_panel_event, event

   COMMON tide_panel_DATA, widgetData

   WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to
; 'PAPCO_panel_Event', which handles the events generated
; by the PAPCO-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

; change Su 3/18/99
   IF strleft(uval, 8) EQ 'spb_type' then begin
       typeValue=fix(strfrom(uval, 8))
       widgetData.typeVector(0)=typeValue
       CASE typeValue OF
         0: widgetData.ylinlog=1
         1: widgetData.ylinlog=1
         2: widgetData.ylinlog=1
         3: widgetData.ylinlog=0
         4: widgetData.ylinlog=0
         5: widgetData.ylinlog=0
         6: widgetData.ylinlog=1
         7: widgetData.ylinlog=0
         8: widgetData.ylinlog=0
         9: widgetData.ylinlog=0
        ENDCASE
       papco_panel_refresh,widgetData,/yscale
; end change
   endif else if strleft(uval, 7) EQ 'spb_ion' then begin
       widgetData.typeVector(1)=fix(strfrom(uval, 7))
   endif else BEGIN
       widgetData=PAPCO_panel_Event(event, widgetData)
   ENDELSE
END











