;******************************************************************************
;* file p_tide_line.pro
;*
;* this file plots TIDE moments (density, velocity, temperature) data as line
;* plots
;*
;******************************************************************************
;* PROCEDURE:     
;*      p_tide_line, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_tide_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;******************************************************************************

pro p_tide_line, panel, channel, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_tide_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, switch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:

   COMMON tide_panel_DATA, widgetData

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
   common get_error, get_err_no, get_err_msg

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_tide. 

   common tide_data, input_header, input_data

   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

; since there is a common time axis plots look better if they all use the same
; number of xticks. In PAPCO, this number is 6 by default. This is set by PAPCO
; using the IDL graphics system variable !X.TICKS.
; You can override this for your own plot by setting the variable
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by PAPCO to default values that guarantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options, different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.
; In this example the data array consist of flux values in ten channels 
; at different energies, typical of particle spectrometer data. Here I use the
; channel information passed in the plotinfo structure from the panel editor
; to select the channel to be drawn.

; set the begin end end times. PAPCo convention is to use running seconds
; starting with the no of seconds since the beginning of the start day.

   tim1 = input_data.seconds
   xut1 = mjdt_start.t                       
   xut2 = long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t

   plottype = channel / 10 
   ion = (channel mod 10)
   ncnt = n_elements(input_data)
   tmin = min(tim1)

   case ion of
     0: ion_lab = 'Ion'
     1: ion_lab = 'H+'
     3: ion_lab = 'He+'
     4: ion_lab = 'O+'
   endcase

   case plottype of

; density

     0:begin
         y1 = input_data.moments(0,ion)
         utitle = 'TIDE ' + ion_lab + ' Density'
         uytitle='cm3'
         temp = where((y1 gt 0.0),cnt)
         if(cnt gt 0) then begin
           yray = y1(temp)
           time = tim1(temp)
         endif else begin
           print,'************************************'
           print,'no data for ',utitle
           print,'************************************'
           yray = [0.001,0.01]
           time = [tmin,tmin]
         endelse
       end

; parallel temperature

     1:begin
         y1 = input_data.moments(4,ion)
         utitle = 'TIDE ' + ion_lab + ' Tpara'
         uytitle = 'eV'
         temp = where((y1 gt 0.0),cnt)
         if(cnt gt 0) then begin
           yray = y1(temp)
           time = tim1(temp)
         endif else begin
           print,' '
           print,'******  no data for ',utitle,' ******'
           print,' '
           yray = [0.001,0.01]
           time = [tmin,tmin]
         endelse
       end

; perpendiular temperature

     2:begin
         y1 = input_data.moments(5,ion)
         utitle = 'TIDE ' + ion_lab + ' Tperp'
         uytitle = 'eV'
         temp = where((y1 gt 0.0),cnt)
         if(cnt gt 0) then begin
           yray = y1(temp)
           time = tim1(temp)
         endif else begin
           print,' '
           print,'******  no data for ',utitle,' ******'
           print,' '
           yray = [0.001,0.01]
           time = [tmin,tmin]
         endelse
       end

; velocity - X

     3:begin
         y1 = input_data.moments(1,ion)
         utitle = 'TIDE ' + ion_lab + ' Vx'
         uytitle = 'km/sec'
         temp = where((y1 gt -1.0e10),cnt)
         if(cnt gt 0) then begin
           yray = y1(temp)
           time = tim1(temp)
         endif else begin
           print,' '
           print,'******  no data for ',utitle,' ******'
           print,' '
           yray = [0.001,0.01]
           time = [tmin,tmin]
         endelse
       end

; velocity - Y

     4:begin
         y1 = input_data.moments(2,ion)
         utitle = 'TIDE ' + ion_lab + ' Vy'
         uytitle = 'km/sec'
         temp = where((y1 gt -1.0e10),cnt)
         if(cnt gt 0) then begin
           yray = y1(temp)
           time = tim1(temp)
         endif else begin
           print,' '
           print,'******  no data for ',utitle,' ******'
           print,' '
           yray = [0.001,0.01]
           time = [tmin,tmin]
         endelse
       end

; velocity - Z

     5:begin
         y1 = input_data.moments(3,ion)
         utitle = 'TIDE ' + ion_lab + ' Vz'
         uytitle = 'km/sec'
         temp = where((y1 gt -1.0e10),cnt)
         if(cnt gt 0) then begin
           yray = y1(temp)
           time = tim1(temp)
         endif else begin
           print,' '
           print,'******  no data for ',utitle,' ******'
           print,' '
           yray = [0.001,0.01]
           time = [tmin,tmin]
         endelse
       end

; EFI Spacecraft potential

     9:begin
         y1 = input_data.scpot
         too_small = where(y1 lt 0.32)
         if (too_small[0]) gt -1 then begin
             y1[too_small] = 0.32
         endif
        
         utitle = 'EFI S/C Pot'
         uytitle = 'V'
         temp = where((y1 gt -1.0e10),cnt)
         if(cnt gt 0) then begin
           yray = y1(temp)
           time = tim1(temp)
         endif else begin
           print,' '
           print,'******  no data for ',utitle,' ******'
           print,' '
           yray = [0.001,0.01]
           time = [tmin,tmin]
         endelse
       end
   endcase

  print,utitle
  print,ion_lab

; restrict data to actual time range requested - this makes for faster zooming

   index=where((time ge xut1) and (time le xut2),count)
   if count ne 0 then begin
      yray=yray(index)
      time=time(index)
   endif

   yst = min(yray)
   yen = max(yray)

;  if panel is velocity, y axis is log, and negative values are present, 
;  reset y axis to linear

   if(plottype gt 2 and yscl(panel(0),3) eq 1) then begin
     index = where((yray le 0),count)
     if(count gt 0) then begin
       print,' '
       print,'**************************************************************'
       print,' negative or zero velocity values present for ', utitle
       print,' no data plotted, redo plot using a linear scale'
       print,'**************************************************************'
       print,' '
       yray = [0.001,0.01]
       time = [tmin,tmin]
       yst = min(yray)
       yen = max(yray)
       yscl(panel(0),1)=yst
       yscl(panel(0),2)=yen
     endif
   endif

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; the procedure YLIM3 is optional and sets the ylimit automatically for the
; data leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; (in $PAPCO/PAPCO_Lib/ylim.pro)

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.
   
   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yen=yscl(panel(0),2)
   endif else begin
      yscl(panel(0),1)=yst
      yscl(panel(0),2)=yen
   endelse

; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override PAPCO defaults.
; See $PAPCO_BASE/PAPCO/papco_draw for the definition of the PAPCO default
; graphic system variable settings.

   extra_plotPar_common={yrange:[yst,yen], xrange:[xut1,xut2], $
                         yticks:ytickno, ytitle:uytitle, $
                         ylog:yscl(panel(0),3)}

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; PAPCO is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 

  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse     

; add keyword structure set here with the one passed in

   extra_plotPar_common=create_struct(extra_plotPar_common, extra_par)

; extend this structure by those keywords specific to the plot being bottom
; panel or not. 
; You might have to add your own xtickformat for time axis labeling. If you 
; stick to the PAPCO convention of having your plot  time array having 
; consecutive seconds from midnight of the start day, then  you can use the 
; pre-defined xtickformat functions defined in ticks.pro in
; PAPCO_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behavior.
; down sets the axis labeling level

   down=0
   if panel(0) eq 0 then $
     extra_plotPar=create_struct({xtickformat:'MJDT_DATE_4'}, $
                                  extra_plotPar_common) $
   else $
     extra_plotPar=create_struct({xtickformat:'noticks'}, $
                                  extra_plotPar_common)

; To keep things general, plotting of the data is split up into plotting the 
; axis with no data, and then plotting the data with no axis. This is done to 
; enable the feature of overplots, which just plot data into an existing axis.
; plot the axis in color=1, and only plot axis if if you're NOT over-plotting!
; down sets the axis labeling level (adds carriage returns [!C] to labels).

   down=0
   color1_plotpar=extra_plotPar
   color1_plotpar.color=1

   if not keyword_set(overplot) then $
      plot,time,yray, _extra=color1_plotPar, xstyle = 1, ystyle = 1, /NODATA

; now do the actual data plot, data only, no axis

   plot,time,yray, _extra=extra_plotPar, xstyle=5, ystyle=5

; store the coordinate information into common block for slices

   plotted_x = !x
   plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; The routine x_side_label puts x-axis label to the right of the axis.
; If you stick to the PAPCO convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; PAPCO_Lib/axis_label.
; Or use any other function you define yourself.
; down sets the axis labeling level

;  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
;     extra_plotPar=create_struct({xtickformat:'MJDT_DATE_3'}, $
;                                 extra_plotPar_common) 
;     extra_plotPar.color=1     
;     down=1
;     plot,time,yray, _extra=extra_plotPar, xstyle = 1, ystyle = 1, /NODATA
;     x_side_label,panel,'time (UT)!CDate'
;  endif

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     down=1
     extra_plotPar.color=1 
     extra_plotPar.xtickformat='MJDT_DATE_3'    
     plot,time,time,_extra=extra_plotPar,/NODATA
     r_polar_eph ;read  the ephemeris file and check if succesfull!
     IF get_err_no eq 0 then begin
       extra_plotPar.color=1 
       extra_plotPar.xtickformat='polar_L_ticks' 
       down=2
       plot,time,time,_extra=extra_plotPar,/NODATA
       extra_plotPar.color=1  
       extra_plotPar.xtickformat='polar_mlt_ticks'
       down=3
       plot,time,time,_extra=extra_plotPar,/NODATA
       extra_plotPar.color=1
       extra_plotPar.xtickformat='polar_ilat_ticks'
       down=4
       plot,time,time,_extra=extra_plotPar,/NODATA
       x_side_label,panel,'time (UT)!CDate!CL!CMLT!CiLAT'
     ENDIF ELSE x_side_label,panel,'time (UT)!CDate'
  endif



; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION

   if not keyword_set(overplot) then begin
     right_side_label,panel,utitle,_extra=extra_Par
   endif else begin
     for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
;;     right_side_label,panel,utitle,_extra=extra_Par
   endelse

end 
