;****************************************************************************
;*  NAME:        r_tide
;*
;*  DESCRIPTION: Routine to read tide files. Adheres to
;*               PAPCO conventions: Error common block
;*                                  Uses MJDT time
;*                                  PAPCO read philosophy
;*
;*  INPUT:       none (uses common block time from PAPCO)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*                              running example module on it's own
;*               July 1999, J-M Jahn, SwRI, jmj@swri.org
;*                              Overhaul of TIDE data read routine. Make sure
;*                              data are properly read and stored for
;*                              multi-day plots.
;****************************************************************************
pro r_tide

   common tide_data, input_header, input_data
   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. Papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

   get_err_no=0
   get_err_msg='tide read succesfully'

; check for env. variable pointing to data directory
   data_dir='tide_data'
   c_path=getenv(data_dir)
   if strlen(c_path) eq 0 then begin
      get_err_no=1
      get_err_msg='Environmental variable '+data_dir+' not set'
      print,get_err_msg
      return
   endif

; check for existence of data directory
   res=findfile(c_path,count=cc)
   if cc eq 0 then begin
      get_err_no=2
      get_err_msg='Data directory '+c_path+' not found'
      print,get_err_msg
     return
   endif
   
   if !version.os_family eq 'unix'  then $  
      IF strmid(c_path,strlen(c_path)-1,1) NE '/' THEN $
         c_path=c_path+'/'

   print,' *************************************************'
   print,' @@   r_tide.pro                                @@'
   print,' *************************************************'

; this example assumes the data was written using the IDL SAVE function.
; replace tide_filename with the correct filename for your data
; this filename is usualy constructed using the information of the start
; and end time as passed from PAPCO in the common block mjdt. 
; Remember that this is time in MJDT format: A structure containing
; modified Julian date and seconds since the start of that day.
; if your data is a different format write a seperate read routine which is
; called here (e.g. r_tide_ascii or r_tide_f77bin etc.)

; PAPCO read philosophy reads the data in whatever format once and then
; makes an IDL binary copy for faster read next time around. The routine
; is expected to concatonate data from various files if needed, and to
; return a data array (e.g. input_data) for exactly the time requested.
; Normally this will mean that a routine to read one file of original data
; is needed (e.g. r_tide_ascii) and another to read the idl binary.
; Files are concatonated in this routine if needed.

; convert_t90_to_date, time, year, day_oy, hour, minute, second,$
;                 month, cmonth, day_om, /mjdt   

time = mjdt_start 
n_files_read = 0

while (mjdt_end.mjd ge time.mjd) do begin
    
;;  convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second,$
;;                  month, cmonth, day_om, /mjdt   
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
      month, cmonth, day_om, /mjdt   
    st_day = day_oy
    datestr=string(year-1900,month,day_om,hour,minute,$
                   format="('t',i2.2,i2.2,i2.2,i2.2,i2.2,'_')")
    
;;  convert_t90_to_date, mjdt_end, year, day_oy, hour, minute, second,$
;;                  month, cmonth, day_om, /mjdt   
    
;;  if(st_day ne day_oy and hour eq 0 and minute eq 0) then $
    datestr=string(year-1900,month,day_om,$
                   format="('t',i2.2,i2.2,i2.2)")
    fln = datestr + '0000_2400' ;;else $
;;    fln = datestr + string(hour,minute, format="(i2.2,i2.2)")
    
    print,'start:',mjdt_start
    print,'  end:',mjdt_end
    print,'fln: ', fln
    fln_time = fln
    
;    goto, temp_detour
  
    ;;; find the data file. We look for a gzipped papco file, and unzipped
    ;;; papco file, and an IDL saveset file, in that order. That way we
    ;;; determine the course of action accorfing to the last file found

    determine_file:

    FileType = 'none'

    filename = c_path+fln+'.papco.gz'
    junk = findfile(filename,count=n_files)
    if (n_files eq 1) then FileType = 'gz'
    filename = c_path+fln+'.papco'
    junk = findfile(filename,count=n_files)
    if (n_files eq 1) then FileType = 'papco'
    filename = c_path+fln+'.idl'
    junk = findfile(filename,count=n_files)
    if (n_files eq 1) then FileType = 'idl'

;;; if file with exact specified times not available, have the user pick the
;;; file instead
    
    if (FileType eq 'none') then begin
        ;; goto, temp_detour
       ;fnm = pickfile(path=c_path)
       ;fln_txt = fnm
       ;chn = strlen(fnm)
       ;nch = rstrpos(fnm,'/')
       ;c_path = strmid(fnm,0,nch+1)
       ;fln = strmid(fnm,nch+1,chn-nch)
       ;goto, determine_file  ;;; ugly, ugly, ugly
        get_err_msg = '% r_tide: TIDE PaPCo file ' + fln + ' not found.'
        if not keyword_set(silent) then print, get_err_msg
        get_err_no = 1
        version = 0
        return
  endif 

    if (FileType eq 'gz') then begin
        filename = c_path+fln+'.papco.gz'
        print, '% unzipping  ' + filename
        spawn, 'gzip -d '+ filename
        FileType = 'papco'   ;;; reset to papco so that file is read
    endif

    if (FileType eq 'papco') then begin
        filename = fln + '.papco'
        r_tide_file, c_path, filename, fln_time
        print,'% r_tide: converting time to PAPCO plot standard'
        filename =  c_path + filename
        print,'% r_tide: compressing papco file ' + filename
        FileType = 'idl'  ;;; force program to reread IDL file
        spawn, 'gzip -f '+ filename + ' &'
    endif    

    if (FileType eq 'idl') then begin
        print, '% reading ' + c_path + fln + '.idl'
        restore,c_path+fln+'.idl',/verbose
        print,'% r_tide: idl saveset read succesfully'
    endif

  
  if (n_files_read eq 0) then begin
      input_data_buf = temporary(input_data)
      input_header_buf = temporary(input_header)
  endif else begin
      ;;; first we need to fix the time. each data file will start at time
      ;;; zero, we need to change that so that the time (in seconds) is
      ;;; continually running through the data file. We assume here that each
      ;;; data file has 24 hours !!!
      input_data[*].seconds = input_data[*].seconds + $
        86400. * n_files_read
      input_data_buf = [input_data_buf,input_data]
;;;      input_header_buf = [input_header_buf,input_header]
  endelse

temp_detour:
  
  n_files_read = n_files_read + 1
  time.mjd = time.mjd + 1
  
endwhile

;;  we are done reading data; shuffle data back into the proper arrays ready
;;  for PaPCo to plot

input_data = input_data_buf
input_header = input_header_buf

return

end

;-------------------------------------------------------------------------------
; r_tide_file is a procedure, called by r_tide, that reads a user-specified 
; tide papco text file
;-------------------------------------------------------------------------------

  pro  r_tide_file, c_path, fln, fln_time

   common tide_data, input_header, input_data

   input_header = {tide_hdr, ion_label:strarr(6), htr_units:' ', $
     ion_num:0, spn_avg:0, htr_limit_type:0, $
     mom_limits:fltarr(2,3), htr_limits:fltarr(2,3)}

;   dummy_struct = {tide, year:0, doy:0, seconds:0l, scpot:0.0, $
;     moments:fltarr(6,6), en_htr:fltarr(31,6), spn_htr:fltarr(32,6), $
;     pol_htr:fltarr(7,6)}

;;; JMJ: change arrays to depth 4 - only 4 different values are supported (H+,
;;; O+, He+ and total ions); also moved the definition of the dummy structure
;;; to a point after which we first have read the date so that we can decide
;;; which kind of data we are actually reading. This seems awkward, but since
;;; we are currently not passing the data form the input form to the read
;;; program, this is the best we can do. 

;  fln_txt = c_path + fln + '.papco'
   fln_txt = c_path + fln
   print,'reading: ',fln_txt
   openr, unit, fln_txt, /get_lun

;  read header information

   dum_str = ' '
   i1 = 0l
   i2 = 0l
   a1 = 0.0
   a2 = 0.0
   max_rec = 0L

   readf,unit,dum_str
   for ii = 0,3 do begin
     readf,unit,dum_str
     input_header.ion_label(ii) = strtrim(dum_str,2)
   endfor

   readf,unit,format="(a15,i4)", dum_str, i1
   input_header.spn_avg = i1
   sec_avg = i1 * 6
   readf,unit,format="(a18,i7)", dum_str, max_rec
   readf,unit,format="(a11,a)", dum_str, dum_str
   input_header.htr_units = strtrim(dum_str,2)

   readf,unit,dum_str
   for ii = 0,2 do begin
     readf,unit,format="(a7,2i3,2f8.3)", dum_str, i1, i2, a1,a2
     input_header.mom_limits(0,ii) = a1
     input_header.mom_limits(1,ii) = a2
   endfor

   readf,unit,dum_str

   input_header.htr_limit_type = 1
   if(strpos(dum_str,'aver') gt 1) then input_header.htr_limit_type = 2
   if(strpos(dum_str,'max') gt 1) then input_header.htr_limit_type = 3
   for ii = 0,2 do begin
     readf,unit,format="(a7,2i3,2f8.3)", dum_str, i1, i2, a1,a2
     input_header.htr_limits(0,ii) = a1
     input_header.htr_limits(1,ii) = a2
   endfor

   readf,unit,dum_str
   readf,unit,dum_str
   readf,unit,dum_str
   readf,unit,dum_str
   readf,unit,dum_str

;  read remaining data

;;;   input_data=replicate(dummy_struct,max_rec)
;;;   help, input_data
   ii = -1
   msec_strt = 0l
   msec_stop = 0l
   year = 0
   doy = 0
   sec = 01
   secadd = 01
   scpot = 0.0
   fill = -1.0e31
   
   while not eof(unit) do begin
      readf,unit,format = "(i4,1x,i3,1x,i8,1x,i8,1x,f10.4)", year, doy, $
        msec_strt, msec_stop, scpot

;;; JMJ: only if data were taken up to 12/07/1996 will the whole record be
;;; read; data taken after that date will only read the total ion entry of the
;;; papco file since species identification is not available;  07 December
;;; 1996 = DOY 343

      if ((year eq 1996) and (doy le 343)) then $
        STOPS_only = 0 else STOPS_only = 1

;;; JMJ: if this is the first run through we will define a couple of arrays to
;;; make reading of the ASCII file faster. Remember: reading arrays is quicker
;;; than reading lines, especially regarding that we also eliminate the
;;; additional overhead of initializing loops!
      
      if ( ii eq -1) then begin
          if (STOPS_only) then $
            dummy_struct = {tide, year:0, doy:0, seconds:0l, scpot:0.0, $
                            moments:fltarr(6), en_htr:fltarr(31), $
                            spn_htr:fltarr(32), $
                            pol_htr:fltarr(7)} $
          else $
            dummy_struct = {tide, year:0, doy:0, seconds:0l, scpot:0.0, $
                            moments:fltarr(6,4), en_htr:fltarr(31,4), $
                            spn_htr:fltarr(32,4), $
                            pol_htr:fltarr(7,4)}
          
          input_data=replicate(dummy_struct,max_rec)
          temp_moments = fltarr(6,4)
          temp_en      = fltarr(31,4)
          temp_spn     = fltarr(32,4)
          temp_pol     = fltarr(7,4)
          
      endif

;     check for day crossing, save start time in seconds

      dummy_struct.year = year
      dummy_struct.doy = doy
      sec_strt = (msec_strt + 500l) / 1000l
      sec_stop = (msec_stop + 500l) / 1000l
      diff = sec_stop - sec_strt
      if(diff lt -10000l) then secadd = secadd + 84000l
      dummy_struct.seconds = sec_strt + secadd

      dummy_struct.scpot = scpot

;     read moments and spectrogram data

      readf,unit,temp_moments
      readf,unit,temp_en
      readf,unit,temp_spn
      readf,unit,temp_pol

      if (STOPS_only) then begin
          dummy_struct.moments = temp_moments[*,0]
          dummy_struct.en_htr  = temp_en[*,0]
          dummy_struct.spn_htr = temp_spn[*,0]
          dummy_struct.pol_htr = temp_pol[*,0]
      endif else begin
          dummy_struct.moments = temp_moments
          dummy_struct.en_htr  = temp_en
          dummy_struct.spn_htr = temp_spn
          dummy_struct.pol_htr = temp_pol
      endelse
           
      ii = ii + 1
      if  ((ii mod 1199) eq 0) then $
        print, ii, msec_strt/3600000., $
        FORMAT='("% r_tide: processing record", I7,",",3X,f4.1," UT")'
      input_data[ii] = dummy_struct

   endwhile

   close,unit
   free_lun,unit

; create text header file for use in labeling plot

;  fnm_txt = c_path + fln + '.txt'
;  print,' '
;  print,'header information has been saved in ', fnm_txt
;  openw, unit, fnm_txt, /get_lun

;    printf, unit, 'Thermal Ion Dynamics Experiment (POLAR/TIDE)
;    line = ' '

;  case input_header.htr_limit_type of
;     2: line = 'moments calculation ranges:       spectrogram average ranges:'
;     3: line = 'moments calculation ranges:       spectrogram maximum ranges:'
;  else: line = 'moments calculation ranges:       spectrogram summation ranges:'
;  endcase

;    printf, unit, ' '
;    printf, unit, line

;    rng1 = string(format="('     energy: ',f5.1,' - ',f5.1,' eV ')", $
;      input_header.mom_limits(0,0),input_header.mom_limits(1,0))
;    rng2 = string(format="('         energy: ',f5.1,' - ',f5.1,' eV ')", $
;      input_header.htr_limits(0,0),input_header.htr_limits(1,0))
;    printf,unit,rng1,rng2

;    rng1 = string(format="(' spin angle: ',f5.1,' - ',f5.1,' deg')", $
;      input_header.mom_limits(0,1),input_header.mom_limits(1,1))
;    rng2 = string(format="('     spin angle: ',f5.1,' - ',f5.1,' deg')", $
;      input_header.htr_limits(0,1),input_header.htr_limits(1,1))
;    printf,unit,rng1,rng2

;    rng1 = string(format="('polar angle: ',f5.1,' - ',f5.1,' deg')", $
;      input_header.mom_limits(0,2),input_header.mom_limits(1,2))
;    rng2 = string(format="('    polar angle: ',f5.1,' - ',f5.1,' deg')", $
;      input_header.htr_limits(0,2),input_header.htr_limits(1,2))
;    printf,unit,rng1,rng2

;  close,unit
;  free_lun,unit


; save read data with correct time. Rename data to *.idl

  save,filename=c_path+fln_time+'.idl',input_header,input_data,/verbose  

return
  
end





