;+
; NAME: tide_color_bar
;
; PURPOSE:
; This IDL-procedure plots a colorbar on the right side of the screen
; in a standard way. (i.e. multiple plots will have colorbars that
; look similar.
; Intended to be very general yet easy to use, by allowing the user
; to override the automatic location of the colorbar relative to the
; plot, and to print only the ztitle or colorbar image.  Also, plot
; keywords not recognized are passed onto the axis command, so for
; instance, tickformats can be specified (use ytickformat).
;
;
; CATEGORY:
;       Plotting (Color)
;
; CALLING SEQUENCE:
;
;       tide_COLOR_BAR, [ZMIN],[ZMAX],[ztitle], $
;            zrange=zrange, zlog=zlog, ztitle=ztitle $
;            ypos=ypos, xpos=xpos, nobar=nobar, _extra=e
;
;
; INPUTS:     in accordance with PLOT_CLR
;             ZMIN     float bottom value of color axis (optional)
;             ZMAX     float top value of color axis (optional)
;             ZTITLE   label for colorbar.  (optional)
;             zrange, ztitle have functionality of ZMIN,ZMAX,ZTITLE
;             /zlog    use log axis
;             ypos=[y1,y2]  specify location of colorbar in normal
;                 coordinates relative to current plot window.
;                  i.e. [ 0.5,1 ] is half heieht of current plot.
;             xpos=[y1,y2]  specify location of colorbar like with y

;
; PROCEDURE:  location of the colorbar is determined from
;                the current plot position, ypos, and xpos specs.
;             colorbar colors are drawn with polyfills.
;             axis is used to draw axis.
;             ztitle is printed.
;
; MODIFICATION HISTORY:
;     Written for PaPCo, October 7, 1997, Jeremy Faden
;     Renamed Hydra version, ESC, 12/97.
;-
PRO tide_COLOR_BAR, ZMIN,ZMAX,ztitle, $
                     zlog=zlog, zrange=zrange, ztitle=ztitle_key, $
                     ypos=ypos, xpos=xpos, nobar=nobar, color_range=colors, $
                     _extra=e



; PROCESS INPUT PARAMETERS
  drawbar= keyword_set(nobar) eq 0

  IF N_PARAMS(0) LT 3 THEN begin
      zmin= zrange(0)
      zmax= zrange(1)
      if keyword_set( ztitle_key ) then ztitle=ztitle_key else ztitle=''
  endif

  zlog= keyword_set(zlog)

  if n_elements( ypos ) ne 2 then ypos=[ 0, 1. ]
  if n_elements( xpos ) ne 2 then xpos=[ 1+1/11., 1+1/11.+1/30. ]

; find colorbar range
  if n_elements(colors) eq 0 then begin
     common papco_color, papco_color, r_papco, g_papco, b_papco
     cindex= where( papco_color(*,2) eq 1 )
     if (cindex(0) eq -1) then begin
        print, 'Color system failure, consult jbf@space-theory.physics.uiowa.edu'
        print, 'Using color set 0'
        cindex=0
     endif
     color_range= PAPCO_get_Colorindices( cindex )
  endif else begin
     color_range= colors
  endelse

  cbase= color_range(0)
  clen= color_range(1)-color_range(0)
  ctop= color_range(1)

; save current viewport and data-screen-relation:
   oldx=!x
   oldy=!y

   linlogbit= keyword_set( zlog )

   color_byte= [1,1] # bindgen( clen ) + cbase

   if !y.type eq 1 then ycrange=10.^!y.crange else ycrange=!y.crange
   if !x.type eq 1 then xcrange=10.^!x.crange else xcrange=!x.crange

   NormalXY0 = convert_coord( [xcrange(0),ycrange(0)], $
                              /data, /to_normal )
   NormalXY1 = convert_coord( [xcrange(1),ycrange(1)], $
                              /data, /to_normal )

   scalex=( NormalXY1(0)-NormalXY0(0) )
   scaley=( NormalXY1(1)-NormalXY0(1) )

   SIZE=SQRT(scaley/scalex)*1.5

   NormalXY1(0)= NormalXY0(0) + xpos(1)*scalex
   NormalXY0(0)= NormalXY0(0) + xpos(0)*scalex
   NormalXY1(1)= NormalXY0(1) + ypos(1)*scaley
   NormalXY0(1)= NormalXY0(1) + ypos(0)*scaley

   DeviceXY0 = round(convert_coord(NormalXY0, /NORMAL, /TO_DEVICE))
   DeviceXY1 = round(convert_coord(NormalXY1, /NORMAL, /TO_DEVICE))
   xsize = DeviceXY1(0)-DeviceXY0(0) ; width of the plot-rectangle
   ysize = DeviceXY1(1)-DeviceXY0(1) ; its height
   xll= DeviceXY0(0)
   yll= DeviceXY0(1)

   if ysize lt 0 then begin ; upside-down color bar flip over
      color_byte= rotate(color_byte,7)
      yll=yll+ysize
      ysize=-ysize
 ;     temp=zmax
 ;     zmax=zmin
 ;     zmin=temp
   endif

   if drawbar then begin
       if !d.name eq 'PS' then begin
           tv, color_byte, NormalXY0(0), NormalXY0(1), $
             xsize=( NormalXY1(0)-NormalXY0(0) ), $
             ysize=( NormalXY1(1)-NormalXY0(1) ), /normal
       endif else begin
           tv, congrid( color_byte, xsize, ysize ), xll, yll
       endelse
       ystyle=1
   endif else ystyle=5

   PLOT, [0,1], [0,1],/noerase, /nodata, $
      position= [ NormalXY0(0),NormalXY0(1),NormalXY1(0),NormalXY1(1)], $
      xstyle=5, ystyle=5, /normal

   if drawbar then $
      axis, yaxis=1, ylog=zlog, yrange=[zmin,zmax], _extra=e, ticklen=-0.6, $
      color=1, ystyle=ystyle
   xyouts, 4.00, .50, ztitle, align=0.5, charsize=size, orient=-90., color=1
;   !p.font=-1
;   !p.position(0)=!p.position(2)
;   !p.position(2)=!p.position(0)+scalex/8.0
;   !y.range=[0.,100.]
;   PLOT,[0.,100.],[100., 100.] , /noerase, /nodata,  ystyle = 5,  xstyle = 5
;   XYOUTS,75.,50.,ZTITLE,alig=0.5,chars=SIZE,orient=-90.,color=1

;  restore entry plot parms
   !x=oldx
   !y=oldy

   return
end
