;***************************************************************************
;***************************************************************************
;* 
;* FILE:
;*    $PAPCO_PATH/timas_spec_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by 
;*    $PAPCO_PATH/papco_panel.pro
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ
;*       $PAPCO_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $PAPCO_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for PAPCO to run, read the 
;*    file addpapco_init.PRO in $PAPCO_MODULES
;*
;* COMMON-BLOCKS:
;*    hydra_spec_panel_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_PATH/papco_panel.pro for a
;*    description). 
;*       {hydra_spec_panel_DATA, 
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  hydra_spec_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, hydra_spec_panel(PAPCO_getPlotInfoStruct())
;*       which calls up the window.
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     December 1995, modified by R.Friedel at Goddard
;*     
;***************************************************************************
;***************************************************************************

;***************************************************************************
;* 
;* FUNCTION:     
;*      FUNCTION  hydra_spec_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,	
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=PAPCO_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;*     Adapted from the Hydra version by J. Faden, ESC, 1/98.
;      Added invalid pitch angle option, ESC, 4/98.
;*
;***************************************************************************
;FUNCTION  hydra_spec_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;   COMMON hydra_spec_panel_DATA, widgetData
FUNCTION  timas_spec_panel, aPlotInfo, GROUP=group, ACTION=ACTION
   COMMON timas_spec_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ----------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
	 column=1, $
	 GROUP_LEADER=group)

;--- the first part of the window will consist in the widgets ------------------
;    that control the height of that plot
   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)


; --- Choose plot type------------------------------------------------------
   type_base=WIDGET_BASE(panel_base, /FRAME, ROW=1, /EXCLUSIVE) 
   typeNames=['H+ ', $
              'O+ ', $
              'He+', $
              'He++' ]
   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(0)), SET_BUTTON=1

; --- Choose pitch angle range ------------------------
   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1, /EXCLUSIVE) 
   typeNames=['all pitch angles (weighted by solid angle)', $
         'field aligned (pitch= 0-30 deg; weighted by solid angle) "Aligned"', $
         'perpendicular (pitch= 75-105 deg; weighted by solid angle) "Perp"', $
         'field opposed (pitch= 150-180 deg; weighted by solid angle) "Opposed"', $
         'all pitch angles (unweighted; ignores pitch angle) "Omni"']

   spb_type=lonarr(N_ELEMENTS(typeNames))
   
   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
     spb_type(i)=WIDGET_BUTTON(type_base, $
                               VALUE=typeNames(i), $
                               UVALUE='spb_pitch'+strtrim(string(i), 2) )
   
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(2)), SET_BUTTON=1

; --- Choose spectrogram resolution -------------------
   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1, /EXCLUSIVE) 
   typenames=[ 'High Resolution Only', $
               'Medium Resolution Included' ]
   
   spb_type=lonarr(N_ELEMENTS(typeNames))
   
   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
     spb_type(i)=WIDGET_BUTTON(type_base, $
                               VALUE=typeNames(i), $
                               UVALUE='spb_tres'+strtrim(string(i), 2))
                               
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(1)), SET_BUTTON=1                               
   
; --- Choose quality ----------------------------------
   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1, /EXCLUSIVE) 
   typenames=[ 'High Quality Only', $
               'Adequate Quality Included' ]
   
   spb_type=lonarr(N_ELEMENTS(typeNames))
   
   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
     spb_type(i)=WIDGET_BUTTON(type_base, $
                               VALUE=typeNames(i), $
                               UVALUE='spb_qual'+strtrim(string(i), 2))
   
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(3)), SET_BUTTON=1

; --- Configure button -----------------
   type_base=WIDGET_BASE(panel_base, /FRAME, /row )
;   spb= widget_button( type_base, $
;                       value='Configure', $
;                       uvalue='spb_configure' )

;   spb= widget_button( type_base, $
;                       value='Options', $
;                       uvalue='spb_options' )
                       
;   spb= widget_button( type_base, $
;                       value='Info', $
;                       uvalue='spb_info' )
   
   spb= widget_button( type_base, $
                       value='Reload Data', $
                       uvalue='spb_reload' )

   spb= widget_button( type_base, $
                       value='Examine Data', $
                       uvalue='spb_command' )
                       


; ---add z range scaling widgets
   widgets=PAPCO_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- add the widgets for choosing the color bar for color plots
;                                ;----------------
;   widgets=PAPCO_panel_AddColorBar_Widgets(panel_base, aPlotInfo)
;   additionalWData=create_struct(widgets, additionalWData)

;  add swittch variable since PAPCO_paneladd_PlotStyleButtons() is not used.
   additionalWData=create_struct({swittch:intarr(3)}, additionalWData)

;--- finally, the Actionbuttons (help, config, cancel, add xxx and continue,
;    add xxx and close) are added to the editor window.
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /config)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'timas_spec_panel.help', $
                    typevector:aPlotInfo.typeVector, $
                    spb_type:spb_type}

   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'timas_spec_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={typeVector:widgetData.typeVector, $
     }

   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO timas_spec_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;
;  ROUTINES CALLED:
;;       timas_spec_configure
;       timas_spec_options
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;*     Adapted from the Hydra version by J. Faden, ESC, 1/98.
;*
;***************************************************************************
PRO timas_spec_panel_event, event
   COMMON timas_spec_panel_DATA, widgetData
   WIDGET_CONTROL,GET_UVALUE=uval, event.id

;   common hydra_spec_data, $
;     data_context, $            ; selects current data set
;     spec_type_names, $
;     spec_header, $
;     spec_data, $
;     spec_aligned_data, $
;     spec_perp_data, $
;     spec_opposed_data, $
;     spec_skew_data
     
   common timas_l1_data, $
     data_context, $            ; selects current data set
     spec_type_names, $
     header, $
     data   

   ; Ion type
   IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
       widgetData.typeVector(0)=fix(strfrom(uval, 8))
       
   ; Pitch angle range
   ENDIF ELSE IF strleft(uval, 9) EQ 'spb_pitch' then begin
       widgetData.typeVector(2)=fix(strfrom(uval, 9))
       ;data_context=widgetData.typeVector(2)
       
   ; Time resolution
   ENDIF ELSE IF strleft(uval, 8) EQ 'spb_tres' then begin
       widgetData.typeVector(1)=fix(strfrom(uval, 8))
       
   ; Quality
   ENDIF ELSE IF strleft(uval, 8) eq 'spb_qual' THEN BEGIN
       widgetData.typeVector(3)=fix(strfrom(uval, 8))
       
;   ENDIF ELSE if uval eq 'spb_configure' then begin
;       timas_spec_configure
;   ENDIF ELSE if uval eq 'spb_options' then begin
;       timas_spec_options

   ; Reload data
   ENDIF ELSE IF uval eq 'spb_reload' then begin
       if n_elements(header) gt 0 then $
         ;spec_header(widgetData.typeVector(2)).reload_flag=1
         header(data_context).reload_flag=1
         
   ; Examine data   
   ENDIF ELSE IF uval eq 'spb_command' then begin
        ;timas_command, group
        !p.position=fltarr(4)
        window
        PRINT,'% timas_spec_panel: Execution of papco is suspended.'
        PRINT,'     To continue, enter .con '
        PRINT,'     Now you can examine Timas data with IDL commands.'
        PRINT,'     A plot window has been created for your convenience.'
        PRINT,'     Variables:
        PRINT,''
        PRINT,'      data.tim   Timas data structure'
        PRINT,'                  (Ion species order: H+, O+, He+, He++)'
        PRINT,'      data.pap   Timas start and stop time structure'
        PRINT,'                  (Seconds of Day)'        
        PRINT,'      header     Timas header structure'
        PRINT,''
        stop
        
   ENDIF ELSE BEGIN
       widgetData=PAPCO_panel_Event(event, widgetData)
   ENDELSE
   
END

   




















