;+----------------------------------------------------------------------
; NAME:
;	R_TIMAS_SPEC
;
; PURPOSE:
;	Read TIMAS data for spectogram plots.
;
; INPUTS:
;       None
;
; CALLED ROUTINES:
;       convert_t90_to_date
;       r_timas_spec_l1
;       timas_define_structure
;
; COMMENTS:
;       Time range for the data read request is passed in common time.
;       xut1 and xut2 are the begin and end times in T90 format (seconds
;       since the beginning of 1990 as a long integer).
;
;       Data is returned in  common timas_l1_data  in variables  spec_header
;       and  spec_data .  spec_data  contains two structures.  spec_data.tim
;       is of type TIMAS_L1 and  spec_data.pap  is of type TIMAS_PAPCO.
;       spec_header  is defined in routine  timas_define_structure .
;
; MODIFICATION HISTORY:
;	ESC, 11/97, based on r_hydra_spec by J. Faden.
;       ESC, 4/99, added line to process four-digit year from PAPCO V. 9
;                  convert_t90_to_date routine.
;-----------------------------------------------------------------------

pro r_timas_spec
;pro r_hydra_spec

   common time, xut1, xut2     ; T90
   
; common get_error is used by PAPCO to print any errors generated in the
; read process.  Routine  draw_timas_spec  (in file  papcoadd_polar_timas.pro )
; calls routine  PAPCO_Draw_CheckResult  (in file  papco_draw.pro ).
  
   common get_error, get_err_no, get_err_msg
   
;   common hydra_spec_data, $
;     data_context, $            ; selects current data set
;     spec_type_names, $
;     spec_header, $
;     spec_data, $
;     spec_aligned_data, $
;     spec_perp_data, $
;     spec_opposed_data, $
;     spec_skew_data, $
;     hr_spec_data, $
;     hr_spec_aligned_data, $
;     hr_spec_perp_data, $
;     hr_spec_opposed_data, $
;     hr_spec_skew_data
     
   common timas_l1_data, $
     data_context, $            ; selects current data set
     spec_type_names, $
     spec_header, $
     spec_data

   module_name = 'r_timas_spec'

   get_err_no = 0
   get_err_msg = module_name + ' data-file read successfully'
   
;*****
; Section below is needed in case this routine is being used stand-alone
; to define  spec_header  and  spec_type_names  and  data_context .
; (Copied from  draw_timas_spec  in  papcoadd_polar_timas.pro .)

   ; Define the header structure
   if n_elements( spec_header ) eq 0 then begin
       timas_define_structure, /spec_header
   endif       

;   case tres of 
;       0: data_context= type 
;       1: data_context= type+5
;       else: begin
;           get_error_msg='draw_timas_spec: internal error'           
;           return
;       end
;   endcase

   ; We are using only one "data_context", ESC, 12/97.
  data_context = 0
;*****   

   data_header= spec_header(data_context)

;   survey= data_header.time_resolution ge 13.8

; setup search path for data
;   if survey then data_dir_env='HYDRA_SURVEY_DATA_PATH' $
;   else data_dir_env='HYDRA_LEVEL1_DATA_PATH'
   data_dir_env = 'TIMAS_LEVEL1_DATA_PATH'
   data_dir = getenv(data_dir_env)
   if data_dir eq '' then begin
       get_err_no = 1
       ; Uses !c to cause a CR (Carriage Return) in an IDL widget, ESC, 11/20.
       get_err_msg = module_name+' environmental variable !c('+data_dir_env+$
         ')!c for data path not set'
       print,get_err_msg
       return
   endif

; read data files for time period
   num_files_read = 0
   time = xut1    ; T90
   while time-1 lt xut2 do begin    ; Loop once for each day
       ; day_start is the beginning of the day (00:00:00) in T90 time (sec).
       day_start = time / 86400 * 86400    ; T90
       start_time = time - day_start       ; SOD
       ;  end_time is xut2 or midnight, whichever is sooner...
;       end_time = min( [ ( (time+1)/86400 +1) * 86400, xut2 ] )
       end_time = min( [ day_start + 86400, xut2 ] )    ; T90 
       end_time = end_time - day_start     ; SOD
       convert_t90_to_date, time, year,doy,h,m,s,month,cm,day_om
       ; Change year to two digits if it's four
       if (year gt 99) then begin
         year = year - year/100*100
       endif
       ;  Example:  date_str = '970831'      
       date_str = string(year,month,day_om,format="(i2.2,i2.2,i2.2)")       
       if end_time gt start_time then begin
;           case data_header.time_resolution of
;               13.8: r_hydra_spec_survey, $
;                 date_str, start_time, end_time, $
;                 data_header, data_values
;               1.15: r_hydra_spec_l1, date_str, start_time, end_time,$
;                 data_header, data_values
;           endcase

           r_timas_spec_l1, date_str, start_time, end_time, $
               data_header, timas, papco

           if get_err_no eq 0 then begin
               ;  Convert time from SOD to SFS (Seconds Of Day to Seconds From Start)
               ;data_values(*).time = data_values(*).time + ( day_start - xut1 )
               papco.data_start = papco.data_start + ( day_start - xut1 )
               papco.data_stop = papco.data_stop + ( day_start - xut1 )
               if (num_files_read eq 0) then begin
                   data_buf_timas = temporary(timas)
                   data_buf_papco = temporary(papco)
               endif else begin
                   data_buf_timas = [temporary(data_buf_timas), temporary(timas)]
                   data_buf_papco = [temporary(data_buf_papco), temporary(papco)]
               endelse
               num_files_read = num_files_read+1
           endif       
       endif
;       time = time + 86400          ; prepare to read next day
       ; Correction made after consulting J. Faden about error, 12/4/97.
       time = day_start + 86400          ; prepare to read next day
   endwhile 

;   if n_elements( data_buf ) eq 0 then begin   
;       get_err_no = 2
;       print, get_err_msg
;       return
;   endif
   if n_elements( data_buf_timas ) eq 0 then begin   
       get_err_no = 2
       print, get_err_msg
       return
   endif


   data_buf = { $
        tim: temporary(data_buf_timas), $     ; Timas TIMAS_L1 structure
        pap: temporary(data_buf_papco) $      ; Timas TIMAS_PAPCO structure for PAPCO
        }

   
; At high-time resolutions, electron and ion sweeps are interleaved.
; We attempt to flag species of each data block (for the slicer).
;   n=n_elements( data_buf(0).spectrum )

;  look for spectra that are not all fill data (1e-20).
;   rspec= indgen(n/2)           ; electrons
;   tot= total( data_buf(*).spectrum(rspec),1 )
;   r= where( tot eq 0 or tot ge 1e-10 ) 
;   if r(0) ne -1 then data_buf(r).species= data_buf(r).species or 1
;   rspec= n-indgen(n/2)         ; ions
;   tot= total( data_buf(*).spectrum(rspec),1 )
;   r= where( tot eq 0 or tot ge 1e-10 ) 
;   if r(0) ne -1 then data_buf(r).species= data_buf(r).species or 2

;   exe_str= spec_type_names(data_context)+'_data'+'=data_buf'
;    Define  spec_data  or any other data type.
   exe_str= spec_type_names(data_context)+'_data'+'=temporary(data_buf)'
   r=execute( exe_str )
;   spec_header(data_context)= data_header
   spec_header(data_context) = temporary(data_header)
   get_err_msg = ' '
   get_err_no = 0

   print, 'Read ' + module_name + ': done'

   return
end


