;+----------------------------------------------------------------------
; NAME:
;	R_TIMAS_SPEC_L1
;
; PURPOSE:
;	Read TIMAS Level 1 data for spectrogram plots.
;
; INPUTS:
;       DATE_STR_IN:  Day to find in the format 'YYYYMMDD'
;	START_TIME:  Start time for data (seconds of day)  
;       END_TIME:    End time for data (seconds of day)
;
; OUTPUT:
;       DATA_HEADER:  Structure of type timas_spec_header defined in
;                     timas_define_structure  
;       TIMAS:  Structure of type TIMAS_h0, defined in h0_summary_inc
;       PAPCO:  Structure of type TIMAS_PAPCO, defined below
;
; CALLED ROUTINES:
;       timas_findfile
;
; INCLUDES:
;       h0_summary_inc
;
; PROCEDURE:
;       Finds data file for requested date with routine  timas_findfile .
;       Uses  h0_summary_inc  include file to get TIMAS_h0 data structure
;       and header parameter values.
;       Selects records by stop time, since these are uniform for
;       all four species (start times may vary among species).  Finds
;       maximum of the four stop times because the stop time
;       is zero if data is missing for that species.  (If data is
;       missing for all four species, the record is not present.)
;       Finds records that satisfy requested time range.  Reads in all
;       data for these records.
;
; MODIFICATION HISTORY:
;	ESC, 11/97, based on r_hydra_spec_survey by J. Faden.
;-----------------------------------------------------------------------

pro r_timas_spec_l1, date_str, start_time, end_time, $
                         data_header, timas, papco
;;pro r_hydra_spec_survey, date_str, start_time, end_time, $
;;                         data_header, data_values
; Read Spectrogram Survey
; Reads in one day of survey data.
;
; HISTORY:
;    v1.0  June 30, 97    Jeremy Faden    First rev coded and released.
;    v1.3  July 1, 97     Jeremy Faden    Fixed time tag bug -- whoops.
;    v1.4  July 23, 97    JBF             Time tags changed to double,
;                                         float is too dangerous
;                                         hydra_findfile function used
;

; common get_error is used by PAPCO to print any errors generated in the
; read process.  Routine  draw_timas_spec  (in file  papcoadd_polar_timas.pro )
; calls routine  PAPCO_Draw_CheckResult  (in file  papco_draw.pro ).
  
  common get_error, get_err_no, get_err_msg

  myname = 'r_timas_spec_l1'
;  data_header.reader_version = 1.0
;  1.0 uses hydra_findfile

;;  retry:
;;  f= timas_findfile( date_str, /survey, version=version )
  f = timas_findfile(date_str, Version=version)
  if f eq '' then begin
    return
;;      hydra_retrieve_data, $
;;        'hydra_spec_survey', '19'+date_str, status
;;      if (status eq 'okay') then goto, retry else return
  endif

  ; For now use pickfile.
;  data_dir= getenv('TIMAS_LEVEL1_DATA_PATH')
;  f = pickfile(Filter='*.cdf', Path=data_dir, /Read)
  cdf = cdf_open(f)
;;  data_header.data_version= version
  get_err_no = 0

;;  cdf_varget, cdf, 'b_sec', b_sec
;;  data_header.time_resolution=b_sec
  
;;  cdf_varget, cdf, 'MODE', mode, rec_count=6272
;;  rvalid= where( mode gt 0 )

;;  if rvalid(0) eq -1 then begin
;;      get_err_msg=my_name+': survey file contains no data.'
;;      print, get_err_msg
;;      get_err_no=1
;;      return
;;  endif

;;  cdf_varget, cdf, 'TIME', time, rec_count=6272
;;  rec_start= ( where( time(rvalid) ge start_time ) )(0)
;;  if rec_start(0) eq -1 then begin
;;      get_err_msg= myname+': No data found before start_time '
;;      print, get_err_msg
;;      return
;;  endif

;;  rec_start= rvalid( rec_start )

;;  r= where( time lt end_time $
;;            and time gt start_time, count )

;;  if count le 1 then begin
;;      print, myname+': no data found'
;;      return
;;  endif 
  
;;  rec_count= r(count-1) - r(0) + 1

  ;;data_values1= { hydra_spec_data, $
  ;;                time:double(0), $
  ;;                mode:byte(0), $
  ;;                spectrum:fltarr(111), $
  ;;                species:byte(0), $ ; 1=ele 2=ions 3=both
  ;;                sc_pot:float(0), $
  ;;                sc_pot_qual:byte(0) $
  ;;              }  
                   
@h0_summary_inc  ; Define TIMAS_h0 structure and set data header values

;  Definition of Hydra header (for reference)
;;       header= { hydra_spec_header, $
;;                 start_time:long(0), $ ; T90
;;                 end_time:long(0), $ ; T90
;;                 pitch_range:fltarr(2), $
;;                 skew_cone_width:float(0), $
;;                 time_resolution:float(0), $
;;                 spec_type_id:0, $
;;                 spec_type_name:' ', $                 
;;                 data_version:float(0), $
;;                 reader_version:float(0), $
;;                 rgains_version:float(0), $
;;                 igains_version:float(0), $
;;                 glint_mask_version:float(0), $
;;                 nen:fix(0), $
;;                 mode_energy:fltarr(111,50), $
;;                 reload_flag:0 $
;;               }
;;       spec_header= replicate( header, 10 )          


   ; "data header" structure is defined in
   ;    .../POLAR_timas/plot_read/timas_define_structure.pro
   ; which is called by  draw_timas_spec  in file
   ;    .../POLAR_timas/papco_interface/papcoadd_polar_timas.pro
   
;   data_header= { timas_spec_header, $
;                 start_time:long(0), $ ; T90
;                 end_time:long(0), $ ; T90
;                 energy_bins: 0, $              ; No. of energy bins
;                 energy_center: fltarr(28), $   ; (ev)
;                 energy_bounds: fltarr(29), $   ; (ev) 
;                 pa_bins: 0, $                  ; No. of pitch angle bins
;                 pa_center: fltarr(12), $       ; (deg)
;                 pa_bounds: fltarr(13), $       ; (deg)
;                 ion_mass: intarr(4), $         ; Integer (amu)
;                 ion_charge: intarr(4), $       ; Integer, positive charge
;                 ion_name: strarr(4), $         ; Ion name w/IDL escape seq.
;                 reload_flag:0 $
;               }
   ;  Fill data header structure
   ;
   ; NOTE:  The following three lines set start_time, end_time, and
   ;        and reload_flag in papcoadd_polar_timas.pro.  These lines
   ;        are only executed when new data is read in, not when the
   ;        user requests a subrange of data already present using the
   ;        time sliders.
   ; spec_header(data_context).start_time=xut1
   ; spec_header(data_context).end_time=xut2
   ; spec_header(data_context).reload_flag=1
   ;
   data_header.energy_bins = energy_bins
   data_header.energy_center = energy_center
   data_header.energy_bounds = energy_bounds
   data_header.pa_bins = pa_bins
   data_header.pa_center = pa_center
   data_header.pa_bounds = pa_bounds
   data_header.ion_mass = ion_mass
   data_header.ion_charge = ion_charge
   data_header.ion_name = ion_name

; h0_summary_inc.pro header and definition is from /timas/code/level_1,
;   1/29/98, ESC.

;+ h0_summary_inc.pro                     H L Collin June 23, 1997
;
; TIMAS Level-One Summary Data Product Structure
;
; Values which are not available will be given a unique fill value.
; Nominal frequency of variables will be 2 spins (12 seconds),
; If a variable is available only every four spins then it has data and 
; fill alternately and its start and stop epoch values will reflect its 
; 24 second accumulation interval. 
; Data collection will be preferentially from 14 energy MRDF's, then
; 7 energy MRDF's, and if neither are available, form LRDF's.  Thus 
; energy steps and/or spin or detector sectors may be summed together 
; These conditions are indicated by "energy_status" and "pa_status".
; Data quality will be implied by the magnitude of "sigma" or in extreme
; cases by the presence of fill.  The quality will also be indicated by 
; the "quality" flag.
;
; Flag Definitions:
;
;   Quality:
;     0    OK.
;     1    Some data missing.
;     2    Slight MCP saturation.
;     3    Moderate MCP saturation.
;     4    Severe MCP saturation.
;     5    No magnetometer data available.
;     6    Warning flags set.
;     99   No valid data.
;
;   PA Status:
;     0    All 22 degree data.
;     1    Mostly 22 degree data.
;     2    Mostly 45 degree data.
;     3    All 45 degree data.
;     4    Smeared 22 degree data. Not spin locked.
;     5    Smeared 45 degree data. Not spin locked.
;     99   Invalid pitch angles.
;
;   Energy Status:
;     0    All single spin 14 energy step data.
;     1    Mostly 14 energy step data. Some 7 energy step. All one or two spin.
;     2    Mostly 7 energy step data. Some 14 energy step. All one or two spin.
;     3    All one or two spin 7 energy step data.
;     4    Mostly 14 energy step data. Some 7 energy step multispins.
;     5    Mostly one or two spin 7 energy step data. Some multispins.
;     6    Mostly multispin 7 energy step data.
;     7    All multispin 7 energy step data.
;     99   Invalid energies.
;     ( Probably some of these conditions (1,2,4,5,6) will be very rare )
;
; An MRDF can be identified by ( pa_status eq 0 and en_status eq 0 )
;
; 
; DEFINITION OF STRUCTURE "TIMAS_h0":
; modified October 14, by WKP to give 12 pitch angle bins...
; Comments in () refer to corresponding variables written to the CDF file.
;
; There are; 28 energy steps, 12 pitch-angle bins and 4 mass species.
;
; h0 = { TIMAS_h0,                      
;        energy_range  : byte,            ; Full, Reduced or Low (0,1,2)
;                                           (Energy_range_ID  CDF_UINT1)
;
;        even_odd      : byte,            ; 0 = even spin first, 1 odd first
;                                           ()
;
;        fec           : float,           ; Total FEC for this interval. 
;                                           ()
;
;        bcr           : float,           ; Total BCR for this interval. 
;                                           ()
;
;        start_epoch   : dblarr(4),       ; Epoch at start of interval 
;                                           (Epoch_start(4)   CDF_EPOCH)
;
;        stop_epoch    : dblarr(4),       ; Epoch at end of interval 
;                                           (Epoch_stop(4)    CDF_EPOCH)
;
;                                           (H_center_time)
;
;        spins         : bytarr(4),       ; Number of spins in this interval 
;                                           (Spins)
;
;        energy_status : bytarr(4),       ; Energy step combination state 
;                                           (Energy_status)
;
;        pa_status     : bytarr(4),       ; Spin sector combination state 
;                                           (PA_status)
;
;        quality       : bytarr(4),       ; Data quality / Error condition 
;                                           (Quality)
;
;        flux          : fltarr(28,12,4), ; Flux values or fill
;                                           (Flux_H, Flux_O, 
;                                            Flux_He_1, Flux_He_2)
;        sigma         : bytarr(28,12,4)  ; Standard error as % of flux 
;                                           (Sigma_H, Sigma_O, 
;                                            Sigma_He_1, Sigma_He_2)
;      }
; 
;-

; DEFINITION OF STRUCTURE "TIMAS_h0":

;h0 = { TIMAS_h0,                                                         $
;       energy_range: -1B, even_odd: -1B, fec: -1.0, bcr: -1.0,           $
;       start_epoch: dblarr(4), stop_epoch: dblarr(4),                    $ 
;       spins: bytarr(4), energy_status: bytarr(4), pa_status: bytarr(4), $ 
;       quality: bytarr(4), flux: fltarr(28,12,4), sigma: bytarr(28,12,4) }


; DEFINITION OF ENERGY & PITCH-ANGLE BINS:

;energy_bins = 28L

;energy_center = [    25.0,    45.0,    68.0,    92.5, $
;                    125.8,   171.0,   232.6,   316.2, $
;                    430.1,   584.8,   795.3,  1081.5, $
;                   1470.7,  2000.0,  2438.0,  2972.0, $
;                   3622.9,  4416.4,  5383.6,  6562.7, $
;                   8000.0,  9752.1, 11888.0, 14491.6, $
;                  17665.4, 21534.4, 26250.7, 30702.0 ]

;energy_bounds = [ 16.0, 34.6, 54.4,    78.2,   106.5, $
;                    145.1,   197.6,   269.2,   366.7, $
;                    499.5,   680.1,   926.1,  1260.8, $
;                   1716.3,  2203.7,  2690.0,  3283.6, $
;                   4002.1,  4877.8,  5945.1,  7246.1, $
;                   8831.9, 10774.9, 13125.6, 16003.5, $
;                  19497.5, 23768.4, 28970.2, 33298.0 ]

;pa_bins = 12L

;pa_center = [   7.5,  22.5,  37.5,  52.5,  67.5,  82.5, $
;               97.5, 112.5, 127.5, 142.5, 157.5, 172.5 ]

;pa_bounds = [  0.0, 15.0, 30.0, 45.0, 60.0, 75.0, 90.0, $
;               105.0, 120.0, 135.0, 150.0, 165.0, 180.0 ]

; DEFINITION OF ION SPECIES:

;                 H+          O+           He+          He++

;ion_mass   = [     1,         16,           4,           4       ]

;ion_charge = [     1,          1,           1,           2       ]

;ion_name   = [ '!5H!U+!N', '!5O!U+!N', '!5He!U+!N', '!5He!U++!N' ]


       
;File cdftostruct_table.txt



;CDF Variable Name      Short Name      TIMAS_h0 Struct Tag Name


;Epoch_start            s_epoch         start_epoch

;Epoch_stop             st_epoch        stop_epoch

;Energy_Range_ID        erange          energy_range

;Even_odd               even_odd        even_odd

;Fec                    fec             fec

;Bcr                    bcr             bcr

;Spins                  spins           spins

;Energy_status          estatus         energy_status

;PA_status              pastatus        pa_status

;Quality                qual            quality

;Flux_H                 fh              flux(*,*,0)

;Flux_O                 fo              flux(*,*,1)

;Flux_He_1              fhe1            flux(*,*,2)

;Flux_He_2              fhe2            flux(*,*,3)

;Sigma_H                sh              sigma(*,*,0)

;Sigma_O                so              sigma(*,*,1)

;Sigma_He_1             she1            sigma(*,*,2)

;Sigma_He_2             she2            sigma(*,*,3)


  cdf_control, cdf, Variable='Epoch_stop', Get_Var_Info=einfo 
  cdf_varget, cdf, 'Epoch_stop', st_epoch, Rec_Count=einfo.maxrec+1
  
  ntime = n_elements(st_epoch)/4
  
  ;  Convert CDF_EPOCH time to SOD (Seconds Of Day) time.
  cdf_epoch, st_epoch(0), yr, mo, dy, hr, mn, sc, mil, /Breakdown_Epoch
  ;  Find epoch at start of day
  cdf_epoch, epoch_start_day, yr, mo, dy, /Compute_Epoch
  
  ; **** This section is old, retained in case we want MJDT time
  ;doy = fix(julday(mo, dy, yr) - julday(1, 1, yr)) + 1  ; Day of year
  ;seconds = hr*3600L + mn*60 + sc + mil/1000.d   ; Seconds Of Day  <double>
  ;  Next function makes MJDT in structure: time = {MJDT, MJD:0L, T:0.0}
  ;time = convert_TZERO_to_T90(seconds, doy, yr, /MJDT)
  ;  Make an array of MJDT time structures.
  ;  Each will have the starting date and time.
  ;time = replicate(time, ntime)
  ;  Add seconds to T tag.  Average start and stop times.  Convert ms to sec
  ;    but keep full precision available in time.T (float). 
  ;time.T = time.T + (s_epoch - s_epoch(0) + st_epoch - s_epoch(0))/2000.D
  ; ****
  
  ;  Eliminate zeroes and consolidate stop time to a one-dimensional
  ;    time array.  Use > operator to eliminate zeroes (find maximum).
  st_epoch_max = dblarr(ntime)
  ;for i=0, ntime-1 do $
  ;  st_epoch_max(i) = max(st_epoch(*,i))
  st_epoch_max = st_epoch(0,*) > st_epoch(1,*) > st_epoch(2,*) > st_epoch(3,*)
                   
  
  time = (st_epoch_max - epoch_start_day)/1000.   ; SOD  <double>
  ;*****
  
  r = where( time lt end_time $
            and time gt start_time, count )
; Next commented-out version was put in by J. Faden as a temporary fix
; for our bad data files (had many extra repeat records at end).
;  r = where( time lt end_time-12 $
;            and time gt start_time, count )

  if count le 1 then begin
      print, myname + ': no data found for requested time interval'
      get_err_no = 1
      ;stop
      return
  endif 
  
  
  rec_start = r(0)
  ; Some records may have time stamps out-of-range, so rec_count for
  ; read purposes must include these, i.e. rec_count ge count.
  rec_count = r(count-1) - r(0) + 1

  ; Definition of structure "TIMAS_PAPCO"
  papco1 = {TIMAS_PAPCO, $
           data_start: dblarr(4), $    ; SOD  
           data_stop:  dblarr(4)  $    ; SOD
           }
           ; Note:  Units are SOD (Seconds Of Day) in THIS routine
           ;        but are changed to SFS (Seconds From Start)
           ;        in calling routine, r_timas_spec.
           
  ;  Select records by stop time, since these are uniform for
  ;  all four species (start times may vary among species).  Find
  ;  maximum of the four stop times because the stop time
  ;  is zero if data is missing for that species.  (If data is
  ;  missing for all four species, the record is not present.)
  
  ;  Make arrays of data structures, one element for each CDF record
  timas = replicate(h0, count)
  papco = replicate(papco1, count)
  ;  Store stop times
  papco(*).data_stop = (st_epoch(*,r) - epoch_start_day)/1000.; SOD  <double> (4,nr)
  timas(*).stop_epoch = st_epoch(*,r)    ; CDF Epoch <double> (4,nr)
  ; Arrays read in from now on will have their 0 index at the old r(0)
  ; because Rec_Start=rec_start which is r(0).
  r=r-r(0)
  
  ;  Get start times  
  cdf_varget, cdf, 'Epoch_start', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  ;  Store start times
  timas(*).start_epoch = tmp(*,r)  ; CDF Epoch <double> (4,nr)
  papco(*).data_start = (tmp(*,r) - epoch_start_day)/1000. ; SOD  <double> (4,nr)
 
  cdf_varget, cdf, 'Quality', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).quality = tmp(*,r)
  cdf_varget, cdf, 'Energy_Range_ID', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).energy_range = (reform(tmp))(r)    ; Remove leading dimension
  cdf_varget, cdf, 'Spins', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).spins =  tmp(*,r)
  cdf_varget, cdf, 'Energy_status', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).energy_status =  tmp(*,r)
  cdf_varget, cdf, 'PA_status', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).pa_status =  tmp(*,r)
  cdf_varget, cdf, 'Flux_H', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).flux(*,*,0) =  tmp(*,*,r)
  cdf_varget, cdf, 'Flux_O', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).flux(*,*,1) =  tmp(*,*,r)
  cdf_varget, cdf, 'Flux_He_1', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).flux(*,*,2) =  tmp(*,*,r)
  cdf_varget, cdf, 'Flux_He_2', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).flux(*,*,3) =  tmp(*,*,r)
  cdf_varget, cdf, 'Sigma_H', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).sigma(*,*,0) = tmp(*,*,r)
  cdf_varget, cdf, 'Sigma_O', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).sigma(*,*,1) = tmp(*,*,r)
  cdf_varget, cdf, 'Sigma_He_1', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).sigma(*,*,2) = tmp(*,*,r)
  cdf_varget, cdf, 'Sigma_He_2', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).sigma(*,*,3) = tmp(*,*,r)
  cdf_varget, cdf, 'Even_odd', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).even_odd = (reform(tmp))(r)    ; Remove leading dimension
  cdf_varget, cdf, 'Fec', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).fec = (reform(tmp))(r)    ; Remove leading dimension 
  cdf_varget, cdf, 'Bcr', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).bcr = (reform(tmp))(r)    ; Remove leading dimension
  cdf_varget, cdf, 'PA_status', tmp, Rec_Start=rec_start, $
      Rec_Count=rec_count
  timas(*).pa_status = tmp(*,r)     
  
  cdf_close, cdf

  print, '% '+myname+': Finished reading '+strtrim(rec_count,2)+ $
    ' records of which '+strtrim(count,2)+' were stored.  
  

;;  cdf_varget, cdf, 'MODE_ENERGY', rec_count=50, mode_energy 
  
;;  modes= where( total( abs( mode_energy ), 1 ) gt 0 )
;;  data_header.mode_energy(*,modes)= mode_energy(*,modes)

;;  case data_header.spec_type_id of
;;      0: zvar_name='SPEC'
;;      1: zvar_name='SPEC_ALIGNED'
;;      2: zvar_name='SPEC_PERP'
;;      3: zvar_name='SPEC_OPPOSED'
;;      4: zvar_name='SPEC_SKEW'
;;      else: begin
;;          get_err_msg= my_name+': internal error.'
;;          print, get_err_msg
;;          get_err_no=1
;;          return
;;      end
;;  endcase

;;  cdf_varget, cdf, zvar_name, rec_start=rec_start, rec_count=rec_count, $
;;    spec_data
  
;;  if zvar_name eq 'SPEC_SKEW' then begin
;;      spec_data_temp= float( spec_data ) / 10
;;      rnodata= where( temporary(spec_data) eq -32768 )
;;      if rnodata(0) ne -1 then spec_data_temp(rnodata)= 1e-20
;      for ien=0,110-1 do begin
;          spec_data_temp( ien, * )= smooth( spec_data_temp( ien,* ), 9 )
;      endfor      
;      for itime=0,n_elements(spec_data_temp(0,*))-1 do begin
;          spec_data_temp( *, itime )= smooth( spec_data_temp( *,itime), 3 )
;      endfor
;;  endif else begin
;;      spec_data_temp= 10 ^ ( float( temporary(spec_data) ) / 1000 )
;;  endelse
;;  spec_data= temporary( spec_data_temp )

;;  data_values(*).spectrum= spec_data(*,r)

;;  cdf_varget, cdf, 'SC_POT', rec_start=rec_start, rec_count=rec_count, $
;;    sc_pot
;;  data_values(*).sc_pot= sc_pot(r)

;;  data_values(*).sc_pot_qual= 1 ; Goddard

;;  cdf_varget, cdf, 'MODE', rec_start=rec_start, rec_count=rec_count, $
;;    mode
;;  data_values(*).mode= mode(r)

;;  cdf_varget, cdf, 'GLINT_MASK_VERSION', glint_mask_version
;;  data_header.glint_mask_version= glint_mask_version

;;  cdf_varget, cdf, 'RGAINS_VERSION', rgains_version
;;  data_header.rgains_version= rgains_version

;;  cdf_varget, cdf, 'IGAINS_VERSION', igains_version
;;  data_header.igains_version= igains_version

  return
end


