;+----------------------------------------------------------------------
; NAME:
;	TIMAS_FINDFILE
;
; PURPOSE:
;	Return the filename of a Timas level 1 file for the date
;       requested.
;
; INPUTS:
;       DATE_STR_IN:  Day to find in the format 'YYYYMMDD'
;
; KEYWORDS:
;
;    Input:
;       SILENT:  Don't print messages swittch
;                  Default:  Not silent
;       PATH:  Path of each directory to search 
;                (directories separated by ":")
;                Default:  Env. variable 'TIMAS_LEVEL1_DATA_PATH'
;
;    Output:
;       VERSION:  Data file version of returned file (from file name)
;       DIRECTORY:  Directory of returned file (full path of directory)
;                     (Includes / at end of path)
;
; Return:  Filename (with path)
;
; PROCEDURE:
;
; EXAMPLE:
;       Return = timas_findfile('19960630')
;       finds the Timas CDF file 'po_h0_tim_19960630_v01.cdf' with full path.       
;
; MODIFICATION HISTORY:
;	ESC, 11/96, based on  hydra_findfile  by J. Faden.
;-----------------------------------------------------------------------

function timas_findfile, date_str_in, $
                         Version=version, Silent=silent, $
                         Path=path, Directory=dir
 
;function hydra_findfile, date_str_in, $
;                         level1=level1, survey=survey, ddcal=ddcal, $
;                         version=version, silent=silent, $
;                         path=path, directory=dir

; given date and data set name, return the fully-qualified name of the most
; current version of the data set.
;  IO:
;    date     (in)  Day to find, 'YYYYMMDD'
;    /level1  (in)  Find level one file
;    /level1cal (in) Find level one calibration file
;    /survey  (in)  Find survey file
;    /silent  (in)  Don't print messages.
;    path     (in)  use this searchpath instead of environment variables
;    version  (out) Version of the file found.
;   <returns> (out) Fully qualified name, or '' if none found.
;  Side Effects:
;    modifies the get_error common block with appropriate message.

  common get_error, get_err_no, get_err_msg

  level1 = 1b  ; For now we only have the level1 file type
  myname = 'timas_findfile'

;  getversion= getenv('hydra_findfile_getversion')

  if strlen(date_str_in) eq 8 then date_str= date_str_in else $
    date_str = '19'+date_str_in

  if keyword_set( level1 ) then begin
      if not keyword_set(path) then path = getenv( 'TIMAS_LEVEL1_DATA_PATH' ) 
      if path eq '' then begin
          print, myname+$
            ': environment variable TIMAS_LEVEL1_DATA_PATH not set'
          version = 0.0
          return, ''
      endif
      l1path = str_sep( path, ':' )
   
;     Search input path for highest version
      bestver = 0  &  bestfile = ''  &  bestdir = ''

      for i=0,n_elements(l1path)-1 do begin
          
          indir=strtrim(l1path(i),2)
          ;  Add a slash to the directory path if it is missing
          IF strmid(indir, strlen(indir)-1, 1) NE '/' THEN $
            indir = indir+'/'
       
          ;ff=findfile( indir+date_str+'_hyd_dd_v?.??.cdf', count=count )
          ff = findfile(indir + 'po_h0_tim_' + date_str + '_v??.cdf', $
                 Count=count)

          if (count ne 0) then begin
              n = strlen(ff(count-1))
;              l1version=float(strmid(ff(count-1),n-8,4))
              ; Be sure the version no. is greater than 0
              l1version = fix(strmid(ff(count-1),n-6,2))
              if l1version gt bestver then begin
                  bestver = l1version
                  bestfile = ff(count-1)
                  bestdir = indir
              endif
          endif
      endfor
      
      if bestver eq 0.0 then begin
          get_err_msg = myname+': Timas Level 1 file not found.'
          if not keyword_set(silent) then print, get_err_msg
          get_err_no = 1
          version = 0
          return, ''
      endif else begin
          if not keyword_set(silent) then print, myname + ': found ' + $
              bestfile
;          n=strlen(bestfile)
;          version=float(strmid(bestfile,n-8,4))
          version = bestver
          dir = bestdir
          get_err_no = 0          ; tell papco all is well
          return, bestfile
      endelse
  endif

  return, ''
end
