;+----------------------------------------------------------------------
; NAME:
;       TIMAS_MAKE_SPEC_MATRIX
;
; PURPOSE:
;	Construct a matrix of ion fluxes vs. time and energy for a
;       spectrogram or a vector of ion fluxes for a "slice" energy
;       spectrum.
;
; INPUT:
;       TYPE:  0=H+
;              1=O+
;              2=He+
;              3=He++
;
; OUTPUT:
;       RVIS: Subscripts of data meeting time, quality, and resolution
;             requirements
;       NVIS:  Dimension of RVIS
;       T90:  T90(*,0)=Start Time for each data point
;             T90(*,1)=Stop Time for each data point
;             If a SLICE_TIME is given, then
;             T90(0)=Start Time for nearest data point
;             T90(1)=Stop Time for nearest data point
;       PITCH_RANGE:  [min, max] pitch angle range
;       SPEC_MATRIX:  flux(28,*), 28 energy steps, NVIS time steps
;                     If a SLICE_TIME is given, then
;                     flux(28), 28 energy steps, is the energy spectrum
;                     at that time.
;
; KEYWORDS:
;     INPUT:
;       SLICE_TIME:  Time for "slice" from mouse click, SFS
;                    If this keyword is set, a single energy spectrum
;                    at the requested time is returned in SPEC_MATRIX.
;       NODATA:  Data value that signifies missing data, default=-1.e+31
;     OUTPUT:
;       IVIS:  Subscript for RVIS for the time requested
;       START_PANEL:  Time in s from beginning of PAPCO panel to start of
;                     data for spectrogram.  Allows gray to be inserted between
;                     start_panel time and start of data meeting quality and
;                     resolution criteria.  If there are bad time stamps on
;                     data at beginning of data, start_panel=-1.  (This allows
;                     gray fill even with bad time stamps.)
;       STOP_PANEL:  Time in s from end of data for spectrogram to end of PAPCO
;                    panel.  Allows gray to be inserted between end of data
;                    meeting quality and resolution criteria, and stop_panel.
;                    If there are bad time stamps on data at end of data,
;                    stop_panel=-1.  (This allows gray fill even with bad time
;                    stamps.)
;
; CALLED ROUTINES:
;       None
;
; COMMENTS:
;       Time range for the data, as originally read in, is passed in
;       common time, variables xut1 and xut2.
;       xut1 and xut2 are the begin and end times in T90 format (seconds
;       since the beginning of 1990 as a long integer).
;
;       Data is in  common timas_l1_data  in variables  spec_header
;       and  spec_data .  spec_data  contains two structures.  spec_data.tim
;       is of type TIMAS_L1 and  spec_data.pap  is of type TIMAS_PAPCO.
;       spec_header  is defined in routine  timas_define_structure .
;
;       When an energy spectrum is requested by setting keyword SLICE_TIME,
;       the fluxes and also the time selected and returned in T90 are for
;       the data meeting the filter criteria that is closest in time to
;       SLICE_TIME.
;
; MODIFICATION HISTORY:
;	ESC, 2/98, based on p_hydra_spec by J. Faden.
;       ESC, 4/98, modified to return start_panel and stop_panel and use nodata
;         for the missing data value.
;-----------------------------------------------------------------------

pro timas_make_spec_matrix, type, rvis, nvis, t90, pitch_range, spec_matrix, $
                      Slice_Time=slice_time, Ivis=ivis, Start_Panel=startpanel, $
                      Stop_Panel=stoppanel, NoData=nodata

   common time, xut1, xut2
   common timas_l1_data, $
     data_context, $            ; selects current data set
     spec_type_names, $
     spec_header, $
     data

   if (n_elements(nodata) eq 0) then $
      nodata = -1.e+31     
; The definition below is made in  r_timas_spec_l1 .
; Each element of the array data.pap is a structure of this type.
;   ; Definition of structure "TIMAS_PAPCO"
;   papco1 = {TIMAS_PAPCO, $
;           data_start: dblarr(4), $    ; SOD
;           data_stop:  dblarr(4)  $    ; SOD
;           }
;           ; Note:  Units are SOD (Seconds Of Day) in this routine
;           ;        but are changed to SFS (Seconds From Start)
;           ;        in calling routine, r_timas_spec.

   ;t90= data(*).time + ( spec_header(data_context).start_time - xut1 )
   
   ; Note:  data_start and data_stop are in SFS (Seconds From Start) where
   ;        the start time is start_time, the value of xut1 when the data was
   ;        read in.  The current value of xut1 (and xut2) may be different
   ;        because it may have been changed with the "start time" slider control.
   ;        The next step converts SFS to SFS from the new start time.
   ;        (The variable name t90 is a misnomer.)
   t90= dblarr(n_elements(data.pap),2)
   t90(*,0)= data.pap(*).data_start(type(0)) + $
                 ( spec_header(data_context).start_time - xut1 )
   t90(*,1)= data.pap(*).data_stop(type(0)) + $
                 ( spec_header(data_context).start_time - xut1 )
   dxs= t90(*,1)-t90(*,0)
;   dt= t90(1)-t90(0)
   dxut= xut2-xut1

   ; Set quality and resolution thresholds.  Only values less than
   ;   threshold are selected
   case type(3) of
     0: qual_thresh= 2
     1: qual_thresh= 4     
   endcase
   case type(1) of
     0: begin
          pa_stat_thresh= 1
          en_stat_thresh= 1
        end
     1: begin
          pa_stat_thresh= 5
          en_stat_thresh= 99
        end
   endcase
   ; Include bad pitch angles if requested
   if type(2) eq 4 then $
     pa_stat_thresh=100
   
   ; The check for correct time span ( ge 0 and le dxut ) must be made
   ; because the time interval given by xut1 and xut2 may now represent a subset
   ; of the time interval for the data as it was originally read in.
   ; Require integration time to be greater than 5 s and lt 50 s to eliminate
   ; spurious start and stop times.
;   rvis= where( t90 ge 0 and t90 le dxut and data(*).mode gt 0, nvis )
   ; First find starting and ending time for data records to be used to get
   ; gray at beginning and end in case not good date there.
   rvis=where( t90(*,0) ge 0 and t90(*,1) le dxut and dxs gt 5. and dxs lt 50., $
     nvis)
   if nvis lt 2 then begin
       print, '% timas_make_spec_matrix: no data in interval'
       return
   endif else begin
     startpanel= t90(rvis(0),0)
     stoppanel= t90(rvis(n_elements(rvis)-1),1)
     rvis= where( t90(*,0) ge 0 and t90(*,1) le dxut and dxs gt 5. and dxs lt 50. $
     ; Quality and resolution
     and data.tim.quality(type(0)) lt qual_thresh $
     and data.tim.pa_status(type(0)) lt pa_stat_thresh $
     and data.tim.energy_status(type(0)) lt en_stat_thresh, nvis )   
   endelse
   if nvis lt 2 then begin
       print, '% timas_make_spec_matrix: ' + $
          'No data in interval meets selection criteria.'  
       return
   endif   
   ; If the start time for this panel is more than 1.05 times the time delta,
   ; set the start time to -1.  (In this case there were some bad times.)
   if (startpanel gt 1.05*(t90(rvis(0),1)-t90(rvis(0),0))) then $
     startpanel= -1
   ; If the end time (xut2-xut1) - stoptime for this panel is more than 1.05 times
   ; the time delta, set the stop time to -1.  (In this case there were some bad
   ; times.)
   rvis_last= n_elements(rvis) - 1
   last_stop= t90(rvis(rvis_last),1)
   if (double(xut2-xut1)-stoppanel gt 1.05*(last_stop-t90(rvis(rvis_last),0))) then $
     stoppanel= -1   
   ;***
   ;stop
   ;***
   ;if nvis lt 2 then begin
   ;    print, '% timas_make_spec_matrix: no data in interval'
   ;    return
   ;endif

;   spec_matrix= data(rvis).spectrum(*)

   ; Flux units are (cm^2-s-sr-ev/q)^-1
   ; To combine bins, weight with solid angle and divide total by
   ;   total solid angle.  The twelve p.a. bins are each 15 deg wide.
   twopi= 2*!pi
   ;pa_bin_edge= indgen(13) * 15   ; 0 to 180 in increments of 15 deg
   pa_bin_edge= spec_header(data_context).pa_bounds
   omega= fltarr(12)
   for i=0,11 do begin   ; Find solid angle for each p.a. bin
     ; 2*pi*(-cos(b)+cos(a))
     if type(2) eq 4 then begin
       omega(i)= twopi/6     ; 4*pi/12
     endif else begin
       omega(i)= twopi*(-cos(pa_bin_edge(i+1)*!dtor) + cos(pa_bin_edge(i)*!dtor))
     endelse
   endfor
   
   case type(2) of
     0: begin
          pa_sub= indgen(12)
          pitch_range= [0,180]         
        end
     1: begin
          pa_sub= [0,1]
          pitch_range= [0,30]
        end
     2: begin
          pa_sub= [5,6]
          pitch_range= [75,105]
        end
     3: begin
          pa_sub= [10,11]
          pitch_range= [150,180]
        end
     4: begin
          pa_sub= indgen(12)
          pitch_range= [0,180]
        end
   endcase
   omegat= total(omega(pa_sub))
   if (n_elements(slice_time) ne 0) then begin
     ; Find bin closest in time; average start and stop times
     mintime= min(abs(total(t90(rvis,*),2)/2. - slice_time), ivis)
     i= ivis
     spec_matrix= fltarr(spec_header.energy_bins)
     ;                                         | pitch angle bin
     spec_matrix(*)= (data.tim(rvis(i)).flux(*,pa_sub,type(0))#omega(pa_sub)) $
              /omegat    ;                            | species
     t90= t90(rvis(ivis),*)
     ;***
     ;stop
     ;***                                                       
   endif else begin
     ; Initialize to nodata
     spec_matrix= fltarr(spec_header.energy_bins,nvis) + nodata
     if (pa_stat_thresh lt 100) then begin
       for i=0,nvis-1 do begin    ; Once for each time
         ; Use only energies for which there is data
         en_sub = where(data.tim(rvis(i)).flux(*,pa_sub(0),type(0)) ne nodata, count)
         ; Don't plot if ANY of the pitch angle bins is Nodata
         if (count ne 0) then begin
           for j=0,n_elements(en_sub)-1 do begin
             if ((where(data.tim(rvis(i)).flux(en_sub(j),pa_sub,type(0)) eq nodata))(0) eq -1) $
                 then $
               spec_matrix(en_sub(j),i)= $   ;       | pitch angle bin
                   (data.tim(rvis(i)).flux(en_sub(j),pa_sub,type(0))#omega(pa_sub)) $
                           /omegat    ;                     | species     
           endfor
         endif
       endfor
     endif else begin   ; If pitch angle is potentially bad
       for i=0,nvis-1 do begin    ; Once for each time
         if (data.tim(rvis(i)).pa_status(type(0)) lt 99) then begin
           ; If pitch angle is good
           ; Use only energies for which there is data
           en_sub = where(data.tim(rvis(i)).flux(*,pa_sub(0),type(0)) ne nodata, count)
           ;if (count ne 0) then $
           ;  spec_matrix(en_sub,i)= $   ;       | pitch angle bin
           ;      (data.tim(rvis(i)).flux(en_sub,pa_sub,type(0))#omega(pa_sub)) $
           ;           /omegat    ;                     | species
           ; Don't plot if ANY of the pitch angle bins is Nodata
           if (count ne 0) then begin
             for j=0,n_elements(en_sub)-1 do begin
               pa_subt = where(data.tim(rvis(i)).flux(en_sub(j),*,type(0)) eq nodata, count)
               if (count eq 0) $
                   then $
                 spec_matrix(en_sub(j),i)= $   ;          | pitch angle bin
                   total(data.tim(rvis(i)).flux(en_sub(j),*,type(0))) $
                        /n_elements(pa_sub)  ;              | species   
             endfor
           endif                 
         endif else begin
           ; If pitch angle is bad
           for j=0,spec_header.energy_bins-1 do begin    ; Once for each energy
             ; Use only pitch angles for which there is data; check for Nodata value.
             pa_subt = where(data.tim(rvis(i)).flux(j,*,type(0)) ne nodata, count)
             ; Average over pitch angle bins
             if (count ne 0) then $
               spec_matrix(j,i)= $   ;            | pitch angle bin
                   total(data.tim(rvis(i)).flux(j,pa_subt,type(0))) $
                        /n_elements(pa_subt)  ;           | species
           endfor
         endelse
       endfor
     endelse
     t90= t90(rvis,*)
   endelse 

   
   end     