
function timas_gcd1, a_in,b_in, errtot=errtot
; Renamed Hydra version, ESC, 2/98.

   if n_elements( errtot ) eq 0 then errtot=0

   a= a_in > b_in
   d= a_in < b_in
   
   r= a mod d

   r_imp= ( r < (d-r) )

   while r gt errtot do begin
       d= r
       r= a mod d
   endwhile
   
   return, d
end


function timas_gcd, A, errtot=errtot, guess=guess 
; timas_gcd: return the greatest common divisor of a set of numbers A.
; I/O: A, array of int, long or float.
;      errtot=errtot  same as errpct, but err is in same units as a,
;                     and is independent of d, specifically:
;                   r < errtot
;        
;      guess=guess  a number that is likely to be the gcd, improve calculation.
;
; Renamed Hydra version, ESC, 2/98.

  if n_elements( guess ) eq 0 then guess= A(0) 
  if n_elements( errtot ) eq 0 then errtot= 0

  gcd= guess
  
  for i=0,n_elements(A)-1 do gcd= timas_gcd1( gcd, A(i), errtot=errtot )
  
  return, gcd
end


function timas_hpc_average, X, navg, nodata=nodata, naverage, noavg=noavg, $
                        allow_smooth=allow_smooth

; averages array over first dimension (having n elements) down to
; navg elements.

;    x1 x2 x3 x4 x5 x6 x7 x8        <-- original 1D (or 2D array), nX elements
;    \      / \   / \      /        <-- (nX/navg) or (nX/navg+1)
;     \    /   \ /   \    /              averaged together
;       a1     a2      a3           <-- averaged array, navg elements
;
; INPUT:  X      is the array to average
;         navg   is the number of average bins
;         nodata specifies nodata value
;         noavg  don't divide by number of points, just sum
;
;         allow_smooth  (nX/navg+1) values go into each average, so
;                       some values in source array to go into
;                       two averages.  This is somewhat faster.
;
; OUTPUT: naverage   passes out the number of values in average (optional)
;         (returns)  the average array, or if /noavg, the sums, sum(Xi),
;                      to be divided by naverage
;
; ASSUMPTIONS and NOTES:
;  For 2d arrays, ANY value equal to 1e-20 is also treated as nodata.
;  It is also assumed that 1e-20 << X(i)*nsum .
;  Occasionally this routine triggers floating underflow.
;
; HISTORY:
;  Spring 1997   Written, Jeremy Faden, jbf@hydra.physics.uiowa.edu
;  Renamed Hydra version, ESC, 2/98.

   if n_elements(nodata) eq 0 then nodata= 1e-20
   allow_smooth= keyword_set(allow_smooth)

   sizeX= size(X)
   n= sizeX( 1 )                ; average over the first dimension
   nsum= float(n)/navg
   
   if (nsum eq 1.0) then begin  ; no averaging
       naverage= make_array( size=size(X), /int, value=1 )
       rnv= where( X eq nodata )
       if rnv(0) ne -1 then naverage(rnv)=0
       return, X
   endif 

   if (nsum lt 1.0 ) then begin ; use congrid to expand       
       print, '% timas_hpc_average: I don''t expand'
       return, X
   endif
   
   if ( sizeX(0) lt 1 or sizeX(0) gt 2) then begin
       print, '% timas_hpc_average: only 1- or 2-D arrays please...'
       return, -1
   endif else if sizeX(0) eq 1 then begin ; 1D average
       typeX= sizeX(2)
       saverage= make_array(navg,type=typeX,value=0)
       naverage= make_array(navg,/int)
   endif else begin             ; 2D average
       typeX= sizeX(3)
       ncol= sizeX(2)
       saverage= make_array(navg,ncol,type=typeX)
       naverage= make_array(navg,ncol,/int)
   endelse     
   
   if (allow_smooth) then begin
       print, '% timas_hpc_average: smoothing allowed.'
       nsum1=ceil(nsum) 
   endif else nsum1=fix(nsum)-1

   ind= long(indgen(navg)*nsum)
   ind0= [ind,n]                ; initial indeces

   if sizeX(0) eq 1 then begin
       for i=0, nsum1 do begin
           rv= where(X(ind) ne nodata)
           if (rv(0) ne -1) then begin
               saverage(rv)=saverage(rv)+X(ind(rv))
               naverage(rv)=naverage(rv)+1
           endif
           ind=ind+1
       endfor

       if (not allow_smooth) then begin
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               rv= where(X(ind) ne nodata)
               if (rv(0) ne -1) then begin
                   saverage(r(rv))=saverage(r(rv))+X(ind(rv))
                   naverage(r(rv))=naverage(r(rv))+1
               endif
           endif
       endif
   endif else begin
       if nodata ne 1e-20 then begin
           rnodata=where(X eq nodata) 
           if rnodata(0) ne -1 then X(rnodata)=1e-20
       endif
           
       for i=0, nsum1-1 do begin
           rv= where(X(ind,*) ne 1e-20)
           if (rv(0) ne -1) then begin
               saverage= saverage+X(ind,*)
               naverage(rv)=naverage(rv)+1
               rz=where(saverage eq 1e-20)
               if rz(0) ne -1 then saverage(rz)=0.0
           endif
           ind=ind+1    
       endfor

       if (not allow_smooth) then begin ; average in the remaining guys
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               rv= where(X(ind,*) ne 1e-20)
               if (rv(0) ne -1) then begin
                   saverage(r,*)= saverage(r,*)+X(ind,*)
                   naverage(r(rv),*)=naverage(r(rv),*)+1
                   rz=where(saverage eq 1e-20) 
                   if rz(0) ne -1 then saverage(rz)=0.0
               endif
           endif
       endif

       if nodata ne 1e-20 then if rnodata(0) ne -1 then X(rnodata)=nodata

   endelse

   r0= where(naverage eq 0)
   if (r0(0) ne -1) then saverage(r0)= nodata
   
   if keyword_set( noavg ) then return, saverage
   
   rv= where(naverage gt 0)
   if (rv(0) ne -1) then saverage(rv)=saverage(rv)/naverage(rv)
   
   return, saverage
end




pro timas_plotcolor, zmat_in, $
                     xarr_in, $
                     yarr_in, $
                     ylog=ylog, $
                     dx=dx, $
                     dy=dy, $
                     zlog=zlog, $
                     zrange=zrange, $
                     _extra=e, $
                     special_colors=special, $
                     nodata=nodata, $
                     sample=sample, $
                     Start_panel=startpanel, $
                     Stop_Panel=stoppanel
                   
; INPUTS:
;
; zmat is the data, dblarr( n, m )
;
; xarr are the x values for each z column.
;    if xarr=fltarr(n,2) then duples are start and end of column.
;    if xarr=fltarr(n)   then each is start of column with no gaps.
;
; yarr are the y values for each z row
;    same format as xarr. fltarr(m,2) or fltarr(m)
;    Units are ev, ESC, 3/30/98.
;
; /ylog is handled, but /xlog will give incorrect results.
;
; dx, dy  explicitly specify the bin size for x and y, otherwise
;    the binsize is the greatest common divisor, within pixel
;    precision of the x and y values.  (dx is in seconds, ESC, 2/23/98.)
;
; nodata specifies that a point should be plotted in grey and omitted
;    from averaging.
;
; special_colors = fltarr(2,n) can be used to insert special colors,
;                              e.g. saturation.  It is an array of pairs,
;                              each pair is (value,color).
;
; colortbl=[a,b]  specifies the range of colors to use for the colorbar. 
;    if not specified, then the following algorythm is used:
;        if n_elements(papco_color) gt 0 then use PAPCO color scheme.
;        otherwise, use 0,!d.n_colors
;
;       START_PANEL:  Time in s from beginning of PAPCO panel to start of
;                     data for spectrogram.  Allows gray to be inserted between
;                     start_panel time and start of data meeting quality and
;                     resolution criteria.
;       STOP_PANEL:  Time in s from end of data for spectrogram to end of PAPCO
;                    panel.  Allows gray to be inserted between end of data
;                    meeting quality and resolution criteria, and stop_panel.
;
; Notes:
;    No averaging is done on columns.

; MODIFICATIONS:
;    Renamed Hydra version, ESC, 2/98.
;    Removed wherex subscripting bug, improved PS output, added start_panel
;      and stop_panel keywords to allow gray at beginning and end of panel,
;      ESC, 4/98.

common time, xut1, xut2
COMMON papco_color, papco_color, r_papco, g_papco, b_papco 
COMMON papco_color_names, black, red, green, yellow, blue, magenta, cyan, $
  white, burgundy, olive, dark_green, teal, $
  royal_blue, violet, dark_grey, grey

if n_params() eq 1 then begin ; x and y not specified.
    m= n_elements(zmat_in(*,0))
    n= n_elements(zmat_in(0,*))
    xarr_in= findgen(m)
    xarr_in= reform( [ xarr_in, xarr_in+1 ], m, 2)
    yarr_in= xarr_in
endif

; copy inputs to local variables
zmat= zmat_in
xarr= xarr_in
yarr= yarr_in

if not keyword_set(nodata) then nodata=5.43e21

if not keyword_set(zrange) then begin
   zmin=min(zmat,max=zmax)
   zrange=[zmin,zmax]
endif else begin
   zmin= double(zrange(0))
   zmax= double(zrange(1))
endelse

zlog= keyword_set(zlog)

sample= keyword_set(sample)     ; sample data, don't average

; find colorbar range
if n_elements(colortbl) eq 0 then begin
    if n_elements( papco_color ) gt 0 then begin
        cindex= where( papco_color(*,2) eq 1 ) ; search the active vector
        if (cindex(0) eq -1) then begin
            print, 'Color system failure, '
            print, 'consult jbf@space-theory.physics.uiowa.edu'
            print, 'Using color set 0'
            cindex=0
        endif
        color_range= PAPCO_get_Colorindices( cindex )
    endif else begin
        color_range= [0,!d.n_colors-1]
        grey=0
    endelse
endif else begin
    color_range= colortbl
endelse
colorbase= color_range(0)
ncolors= color_range(1)-color_range(0)
colortop= color_range(1)
    
ylog= keyword_set( ylog )

; reformat xarr and yarr to start, end time format. (if necessary.)
s= size(xarr)
if s(0) eq 1 then begin
    xarr_new= make_array( s(1)-1, 2 )
    xarr_new(*,0)= xarr(0:(s(1)-2))
    xarr_new(*,1)= xarr(1:*)
    xarr=xarr_new
endif
s= size(yarr)
if s(0) eq 1 then begin
    yarr_new= make_array( s(1)-1, 2 )
    yarr_new(*,0)= yarr(0:(s(1)-2))
    yarr_new(*,1)= yarr(1:*)
    yarr=yarr_new
endif

; handle /ylog keyword.
if keyword_set( ylog ) then begin
    yarr=alog10(yarr)
endif

q=n_elements(xarr_in)-1 & r=n_elements(yarr_in)-1
plot, xarr_in([0,q]), yarr_in([0,r]), $
  /nodata, _extra=e, ylog=ylog
xcrange= !x.crange
ycrange= !y.crange

; calculate pixel resolution 
if ylog then ycrange10=10^ycrange else ycrange10=ycrange
lldev= convert_coord( xcrange(0), ycrange10(0), /data, /to_device )
urdev= convert_coord( xcrange(1), ycrange10(1), /data, /to_device )
resx= ( xcrange(1) - xcrange(0) ) / ( urdev(0)-lldev(0) )
resy= ( ycrange(1) - ycrange(0) ) / ( urdev(1)-lldev(1) )

; calculate bin size dx and dy
if not keyword_set(dx) then begin
    dxs= xarr(*,1)-xarr(*,0)
    dx= timas_gcd( dxs, errtot=resx )
endif

if not keyword_set(dy) then begin
    dys= yarr(*,1)-yarr(*,0)
    dy= timas_gcd( dys, errtot=resy )
endif


; find which xbins appear on the plot
rplotx= where( xarr(*,0) ge xcrange(0) and xarr(*,1) le xcrange(1), nplotx )
if nplotx lt n_elements( xarr(*,0) ) then begin
    if nplotx gt 0 then begin
        xarr=xarr(rplotx,*)
        zmat=zmat(rplotx,*)
    endif else begin
        xarr=xarr(0,*)          ; we'll see that nothing is printed later
        zmat=zmat(0,*)
    endelse
endif

rploty= where( yarr(*,0) ge ycrange(0) and yarr(*,1) le ycrange(1), nploty )
if nploty lt n_elements( yarr(*,0) ) then begin
    if nploty gt 0 then begin
        yarr=yarr(rploty,*)
        zmat=zmat(*,rploty)
    endif else begin
        yarr=yarr(0,*)          ; we'll see that nothing is printed later
        zmat=zmat(*,0)
    endelse
endif

; calculate the number of x and y bins for image
;   and form zimage -> zmat map
; This section revised to eliminate wherex subscript out-of-range error,
;  ESC, 3/31/98.
; Revised to eliminate narrow gray stripes in PS output, ESC, 4/5/98.
; Revised to include gray at beginning and end of panel if bad date there,
;   ESC. 4/6/98.
if (n_elements(startpanel) eq 0) then begin
  startpanel = xarr(0)
endif else begin
  ; Minimize misalignments between panels
  if (startpanel eq -1) then $
    startpanel = xarr(0) mod dx 
endelse
if (n_elements(stoppanel) eq 0) then begin
  stoppanel = xarr(n_elements(xarr)-1)
endif else begin
  ; Minimize misalignments between panels
  if (stoppanel eq -1) then begin
    stoppanel = double(xut2-xut1)
    stoppanel = stoppanel - ((stoppanel-xarr(n_elements(xarr)-1)) mod dx)
  endif
endelse
;nx= long( ( xarr(n_elements(xarr)-1) - xarr(0) ) / dx + 0.5 ) 
nx= long( ( stoppanel - startpanel ) / dx + 0.5 ) 
;wherex1= long((xarr(*,0)-xarr(0))/dx+0.5)
;wherex2= long((xarr(*,1)-xarr(0))/dx+0.5)-1
;dxp = ( xarr(n_elements(xarr)-1) - xarr(0) ) / nx  ; Time per bin
dxp = ( stoppanel - startpanel ) / nx  ; Time per bin
; -.5 eliminates bias toward high bin number and gives 1/2 bin accuracy
; in end point placement
;wherex1= long((xarr(*,0)-xarr(0))/dxp + .5) < (nx - 1)
wherex1= long((xarr(*,0)-startpanel)/dxp + .5) < (nx - 1)
; Except where diff gt .5 s, make stop time equal next start time 
; to eliminate gray stripes
last_el = n_elements(xarr)/2 - 1
stop_adj=dblarr(last_el + 1)
; Stop time for last element won't change
stop_adj(last_el) = xarr(last_el,1)
; Start by setting stop times to next start time since the gap is usually
; less than .02 s
stop_adj(0:last_el-1) = xarr(1:last_el,0)
; For any diffs greater than .5 s use actual stop time.
stop_in = where(xarr(1:last_el,0) - xarr(0:last_el-1,1) gt .5, count) 
if (count gt 0) then $
  stop_adj(stop_in) = xarr(stop_in,1)
;wherex2= long((xarr(*,1)-xarr(0))/dxp + .5) - 1
;wherex2= long((stop_adj-xarr(0))/dxp + .5) - 1
wherex2= long((stop_adj-startpanel)/dxp + .5) - 1
wherex2= wherex2 > wherex1

ny= long( ( yarr(n_elements(yarr)-1) - yarr(0) ) / dy + 0.5 )
;wherey1 = long((yarr(*,0)-yarr(0))/dy+0.5)
;wherey2 = long((yarr(*,1)-yarr(0))/dy+0.5)-1
dyp = ( yarr(n_elements(yarr)-1) - yarr(0) ) / ny  ; log10(energy) per bin
wherey1 = long((yarr(*,0)-yarr(0))/dyp + .5)
wherey2 = long((yarr(*,1)-yarr(0))/dyp + .5) - 1
;***
;stop
;***
wherex= intarr(nx)-1
wherey= intarr(ny)-1
for i=0,n_elements(wherex1)-1 do $
  wherex(wherex1(i):wherex2(i))= i
for i=0,n_elements(wherey1)-1 do $
  wherey(wherey1(i):wherey2(i))= i

zimage= make_array( nx, ny, /float, value=nodata )
rx= where( wherex ge 0 )
ry= where( wherey ge 0 )
for j=0,n_elements(ry)-1 do begin
    zimage(rx,ry(j))= zmat(wherex(rx),wherey(ry(j)))
endfor

; find the location of tv image of zimage on data coords.
;x1= xarr(0)
;x2= xarr( n_elements(xarr)-1 )
x1= startpanel
x2= stoppanel

if ylog then begin
    y1=10^yarr(0) 
    y2=10^yarr( n_elements(yarr)-1 )
endif else begin
    y1=yarr(0)
    y2=yarr( n_elements(yarr)-1 )
endelse

lld= convert_coord( x1, y1, /data, /to_device )
urd= convert_coord( x2, y2, /data, /to_device )

; find device size
xsized= long(urd(0)-lld(0))
ysized= long(urd(1)-lld(1))

if !d.name eq 'PS' then begin
    if !d.x_px_cm eq 1000 then begin
        print, '% timas_plotcolor: reducing PS resolution to 300 dpi'
        xsized = xsized * 300 / 2540
        ysized = ysized * 300 / 2540 
    endif
endif

; average down the array if there aren't enough pixels to show each 
; measurement.
if (xsized lt nx) then begin
    if (sample) then begin
        print, '% timas_plotcolor: sampling data'
        rrr= long( findgen( xsized ) / xsized * nx )
        zimage= zimage( rrr, * )
    endif else begin
        print, '% timas_plotcolor: time-averaging '+strtrim(nx,2)+$
          ' time intervals down to '+strtrim(xsized,2)+'.'    
        zimage= timas_hpc_average( zimage, xsized, /allow, nodata=nodata )
    endelse
    nx= xsized
endif

; transform data to color
if (zlog) then begin
    zsize= alog10( zmax/zmin )
    zcolor= make_array( nx, ny, /byte, value=colorbase )
    rok= where( zimage ge zmin )
    if rok(0) ne -1 then zcolor(rok)= alog10( zimage(rok)/zmin ) / $
      ( zsize / ncolors ) + colorbase < colortop
endif else begin
    zsize= zmax-zmin
    zcolor= make_array( nx, ny, /byte, value=colorbase )
    rcol= where( zimage ge zmin )
    zcolor(rcol)= ( zimage(rcol) - zmin ) / $
      ( zsize / ncolors ) + colorbase < colortop
    r= check_math()
    if r ne 0 then message, 'check_math()='+strtrim(r,2), /cont
endelse

; fill color nodata points grey
rnd=where( zimage eq nodata )
if rnd(0) ne -1 then zcolor( rnd ) = grey
; fill special color points with special colors
if n_elements(special) gt 0 then begin
    for i=0,n_elements(special(0,*))-1 do begin
        value= special(0,i)
        color= byte(special(1,i))
        rval= where( zimage eq value )
        if rval(0) ne -1 then zcolor(rval)=color
    endfor
endif

if (nplotx gt 0) then begin
    if !d.name eq 'PS' then begin
        lln= convert_coord( lld, /device, /to_normal )
        urn= convert_coord( urd, /device, /to_normal )
        xsizen= urn(0)-lln(0)
        ysizen= urn(1)-lln(1)
        tv, zcolor, lln(0), lln(1), xsize=xsizen, ysize=ysizen, /normal
    endif else begin
        tv, congrid(zcolor, xsized, ysized), lld(0), lld(1)+1 ; +1 is a kludge
    endelse
endif

plot, xarr_in(*), yarr_in(*), $
  /noerase, /nodata, _extra=e, ylog=ylog

return

end
