;******************************************************************************
;* FILE: pum_blank_panel.pro
;*
;* DESCRIPTION: 
;*    This is a demonstration on how to implement a papco-panel-editor.
;*    It makes use of all the features provided by $papco_XX/papco_panel.pro
;*    (a collection of standard papco widgets). It constructs an editor for a
;*    very simple papco-plot, that has all the default-functionality.
;*
;* COMMON-BLOCKS:
;*    pum_blank_panel_data   contains the window-information
;*
;*
;* MODIFICATION HISTORY:       
;*     August 2003, written by R. Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  pum_blank_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  pum_blank_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON pum_blank_panel_data, widgetData

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
title = aplotinfo.panelKind + ' - ' + papco_module_version('pum_blank')
panel_base=papco_WIDGET_BASE(TITLE = title, COLUMN = 1, GROUP_LEADER=group)

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
   
pum_lbl = WIDGET_LABEL(panel_base,  VALUE = 'Set Height of blank panel')

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the pum_blank example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'pum_blank_panel.help'}
widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData
papco_xmanager, 'pum_blank_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
additionalRData=papco_panel_ReturnData(widgetData)

return, additionalRData
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO pum_blank_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO pum_blank_panel_event, event
  
COMMON pum_blank_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

widgetData=papco_panel_Event(event, widgetData)
    
END
