;******************************************************************************
;* FILE:     
;*    map_panel.pro
;* 
;*    This file belongs to the papco-project. See $papco_XX/papco/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a papco-
;*    panel-editor. It makes use of all the features provided by 
;*    $papco_XX/papco_panel.pro (a collection of standard papco widgets)
;*    
;*    It constructs an editor for a very simple papco-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ (under $papco_XX/papco_doc)
;*    papco_conventions.text - which conventions were used when developing ?
;*    papco_variables.text   - what structures and global variables
;                              are used by which file ?
;*
;* COMMON-BLOCKS:
;*    papco_PANEL_TEMPLATE_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    papco_PANEL_EDITOR-structure (see $papco_XX/papco_panel.pro for a
;*    description). 
;*       {papco_PANEL_TEMPLATE_DATA, 
;*	     ; imagine the papco_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $papco_DOC/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  papco_PANEL_TEMPLATE, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*       a=papco_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=map_panel(a)
;*       which calls up the window.
;*       Interrogating a with help,a,/str shows the contents. This is a good
;*       way of testing your own panel editor. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;*     may 2004, modified by J. Niehof at BU
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  map_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      switch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $papco_XX/papco_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;*
;* MODIFICATION HISTORY:       
;*      June 1997, written by S. BOURDARIE
;*      September 1999, modified by S. BOURDARIE 
;******************************************************************************
FUNCTION  map_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON map_panel_DATA, widgetData
   COMMON map_files, data_names

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
       column=1, $
       GROUP_LEADER=group)

; All papco_panel_Add* routines are papco panel widgets that can be included
; in the panel editor or not. They provide functionality that papco supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by papco_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
; get list of map files available
   error_path = 0
; check for env. variable pointing to data directory
   data_dir='MAP_DATA'
   path=papco_getenv(data_dir)
   if strlen(path) eq 0 then begin
     get_err_no=1
     get_err_msg='Environmental variable '+data_dir+' not set'
     print,get_err_msg
     error_path = 1
   endif

; --- expand paths (JTN 27 May 2004) handle home directory, $PAPCO_DLNK------
   path=expand_path(path)
 
; --- check for existence of data directory -----------------------------------
   cc=papco_finddir(path)
   if cc eq 0 then begin
     get_err_no=2
     get_err_msg='Data directory '+path+' not found'
     print,get_err_msg
     error_path = 1
   endif

; --- Select data input from map files --------------------------------------
   if error_path eq 1 then begin
      data_names = 'no file found'
      channel_names = strarr(1)
      channel_names(0) = 'empty'
      goto, err1
   endif

; --- Get list of model files available only if widgetData does not exist yet -
   result=size(widgetData)
   exist=result(n_elements(result)-2)

   if exist eq 0 then begin
     message,'Looking for map files ...',/cont
     result=findfile(path+'/*/*.map')
     res   =findfile(path+'/*/*/*.map')
     result = [result,res]
     res   =findfile(path+'/*.map')
     result = [result,res]
     index = where(strpos(result,'.map') ne -1,count)
     exp_path=papco_expand_path(path)
     if count ne 0 then begin
       da_names = result(index)
       data_names=strarr(n_elements(da_names))
       print,data_names
       for i=0,n_elements(da_names)-1 do begin
         pos=strlen(exp_path)
         pos2=strpos(da_names(i),'.map')
         data_names(i)=strmid(da_names(i),pos,pos2-pos)
       print,data_names(i)
       endfor
     endif else data_names = 'no file found'

   endif 
err1 : burp = ' '

; --- create list widgets -----------------------------------------------------
   model_base=WIDGET_BASE(panel_base, /frame, column=1)

   type_base=WIDGET_LABEL(model_base,  VALUE='Select Data:')

   file_base=WIDGET_BASE(model_base, row=1)
   list_base=WIDGET_LIST(file_base, YSIZE=5, XSIZE = 60, /frame, $
                         VALUE=data_names, UVALUE='list_base')

   WIDGET_CONTROL, list_base, SET_LIST_SELECT=aPlotInfo.typeVector(1)
   aPlotInfo.filename=path+'/'+data_names(aPlotInfo.typeVector(1))+'.map'

   pb_getlist=WIDGET_BUTTON(file_base, UVALUE='pb_getlist', $
                            VALUE='REFRESH LIST')


; --- Plot Switches -----------------------------------------------------------
   switch_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1)

   smooth_base=WIDGET_BASE(switch_base, /FRAME, ROW=1, /EXCLUSIVE)

   smoothNames=['no smooth', $
              'smooth vertically', $
              'smooth horizontally', $
              'smooth both']
   spb_type=lonarr(N_ELEMENTS(smoothNames))

   FOR i=0, N_ELEMENTS(smoothNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(smooth_base, $
			VALUE=smoothNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_type(aPlotInfo.ioptions(17)), SET_BUTTON=1

   fill_base=WIDGET_BASE(switch_base, /FRAME, ROW=1, /EXCLUSIVE)

   fillNames=['no fill', $
              'fill missing data']
   spb_fill=lonarr(N_ELEMENTS(fillNames))

   FOR i=0, N_ELEMENTS(fillNames)-1 DO $
      spb_fill(i)=WIDGET_BUTTON(fill_base, $
			VALUE=fillNames(i), $
			UVALUE='spb_fill'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_fill(aPlotInfo.ioptions(18)), SET_BUTTON=1
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $papco_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the widget is always reinitialized to its last state.

; The next four papco panel widgets will be added only if you want to
; support that functionality, and if they make sense for your map.

;--- add the papco panel widget for choosing the color bar for color plots ----
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the map example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpfile:'map_panel.help', $
                    typevector:aPlotInfo.typevector, $
                    ioptions:aPlotInfo.ioptions, $
                    filename:aPlotInfo.filename, $
                    list_base:list_base, $
                    panel_base:panel_base}
   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'map_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
   returnData={typevector:widgetData.typevector, $
               ioptions:widgetData.ioptions}
   additionalRData=papco_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO map_panel_refresh, LIST=LIST
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO salammbo_panel_event
;*
;* MODIFICATION HISTORY:       
;*      June 1997, written by S. BOURDARIE
;*      September 1999, modified by S. BOURDARIE 
;******************************************************************************
PRO map_panel_refresh, LIST=LIST

   COMMON map_panel_DATA, widgetData
   COMMON map_files, data_names

   data_dir='MAP_DATA'
   path=papco_getenv(data_dir)
   error_path = 0
   if strlen(path) eq 0 then begin
     get_err_no=1
     get_err_msg='line_panel: Environmental variable '+data_dir+' not set'
     print,get_err_msg
     error_path = 1
   endif

; --- expand paths (JTN 27 May 2004) handle home directory, $PAPCO_DLNK------
   path=expand_path(path)
   
; --- check for existence of data directory -----------------------------------
   cc=papco_finddir(path)
   if cc eq 0 then begin
     get_err_no=2
     get_err_msg='Data directory '+path+' not found'
     print,get_err_msg
     error_path = 1
   endif

   if keyword_set(LIST) then begin
     WIDGET_CONTROL, widgetData.list_base, SENSITIVE=0
     if error_path eq 1 then  begin
        data_names = 'no file found'
        goto, lab0
     endif
     message,'Looking for map files ...',/cont
     result=findfile(path+'*/*.map')
     res   =findfile(path+'*/*/*.map')
     result = [result,res]
     res   =findfile(path+'*.map')
     result = [result,res]
     index = where(strpos(result,'.map') ne -1,count)
     if count ne 0 then begin da_names = result(index)
       exp_path=papco_expand_path(path)
       data_names=strarr(n_elements(da_names))
       for i=0,n_elements(da_names)-1 do begin
         pos=strlen(exp_path)
         pos2=strpos(da_names(i),'.map')
         data_names(i)=strmid(da_names(i),pos,pos2-pos)
       endfor
     endif else data_names = 'no file found'

lab0: burp = ' '

     WIDGET_CONTROL, widgetData.list_base, SET_value=data_names
     WIDGET_CONTROL, widgetData.list_base, SENSITIVE=1
     WIDGET_CONTROL, widgetData.list_base, SET_LIST_SELECT=0
     widgetData.typeVector(1) = 0
     widgetData.filename=path+'/'+data_names(widgetData.typeVector(1))+'.map'
   endif

END
;******************************************************************************
;* PROCEDURE:     
;*      PRO map_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*      June 1997, written by S. BOURDARIE
;*      September 1999, modified by S. BOURDARIE 
;******************************************************************************
PRO map_panel_event, event

   COMMON map_panel_DATA, widgetData
   COMMON map_files, data_names

   WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

   if strleft(uval, 8) EQ 'spb_type' THEN BEGIN
      smooth_choice=fix(strfrom(uval, 8)) 
      uval=strleft(uval, 8)
   endif

   if strleft(uval, 8) EQ 'spb_fill' THEN BEGIN
      fill_choice=fix(strfrom(uval, 8)) 
      uval=strleft(uval, 8)
   endif

   case uval of
      'list_base': BEGIN
          data_dir='MAP_DATA'
          path=papco_getenv(data_dir)
          result=WIDGET_INFO(event.id, /LIST_SELECT)
          widgetData.typevector(1)=result
          widgetData.filename=path+data_names(result)+'.map'
          message,widgetData.filename,/cont
       END  
       
      'pb_getlist': BEGIN
          map_panel_refresh, /LIST
       END

      'spb_type'       : BEGIN
                widgetData.ioptions(17)=smooth_choice
                  END

      'spb_fill'       : BEGIN
                widgetData.ioptions(18)=fill_choice
                  END
  
     'pb_done' : BEGIN
        data_dir='MAP_DATA'
        path=papco_getenv(data_dir)
        uval=''
        if WidgetData.filename eq path+'/no file found.map' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a file name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no file selected')
        endif else uval='pb_done'
      END

     'pb_done_and_cont' : BEGIN
        data_dir='MAP_DATA'
        path=papco_getenv(data_dir)
        uval=''
        if WidgetData.filename eq path+'/no file found.map' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a file name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no file selected')
        endif else uval='pb_done_and_cont'
      END

      else: widgetData=papco_panel_Event(event, widgetData)
   endcase

   if (uval eq 'pb_done') or (uval eq 'pb_done_and_cont') then $
            WidgetData=PAPCO_Panel_event(event,WidgetData)

END
