;******************************************************************************
;* file p_map_spec.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_map_spec, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_PLOTINFO.switch (three element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_map, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written June 1997, S. BOURDARIE
;*     modified September 1999, S. BOURDARIE
;******************************************************************************
pro p_map_spec, panel, type, plotinfo, $
                 OVERPLOT=OVERPLOT, $
                 PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 SUBTABLE=SUBTABLE, _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_map_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags whichare meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, switch. Replace type in the call with whatever variables
; you need, or add more

; Comments here should include an explanation of the plot options used
; in type, switch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
  common yscale, yscl		       ;man/auto yscaling
  common zscale, zscl		       ;man/auto zscaling
  common shift_label, down             ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                       ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_map. 
  common map_data, input_header, input_data

  papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
  panelset,panel		;sets the panel position viewport
  
  ioptions=plotinfo.ioptions
  
; since there is a common time axis plots look better if they all use the same
; number of xticks. In papco, this number is 6 by default. This is set by papco
; using the IDL graphics system variable !X.TICKS.
; You can overide this for your own plot by setting the varbale
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by papco to default values that gurantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; for the color plot routine, you need to set up arrays containing the upper  
; and lower limits of each channel. Here is an example for CRRES MEB energy 
; channels. There must be as many entries as there are channels in the data
; array! 
  yminarr=input_header.yarr(0:n_elements(input_header.yarr)-2)
  ymaxarr=input_header.yarr(1:n_elements(input_header.yarr)-1)
  n_channels=n_elements(yminarr)
  yarr=fltarr(n_channels,2)   
  yarr(*,0)= yminarr 
  yarr(*,1)=ymaxarr
  
; for color plots, the y-axis normally refers to a channel. You have to 
; re-scale the data array to the range required before passing it to the
; color plot routine, also the ymin and ymax arrays.

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

; in this example we have a single test array, so no case statement is needed.

; set the begin end end times. papco convention is to use running secconds
; starting with the no of seconds since the begiing of the start day.
  time=input_data.time
  xut1=float(mjdt_start.mjd) + mjdt_start.t / 86400.0      
  xut2=float(mjdt_end.mjd) + mjdt_end.t / 86400.0

  zmat=input_data.ymat
  zmat=transpose(zmat)
  uztit=input_header.unit
  uytitle=input_header.ytitle
  utitle=input_header.comment1 +'!C' + input_header.comment2

  yst=yminarr(0)  &  yen=ymaxarr(n_channels-1)

; restrict data to actual time range requested - this makes for faster zooming
;   index=where((time ge xut1) and (time le xut2),count)
;   if count ne 0 then begin
;      ymat=ymat(index,*)
;      time=time(index)
;   endif

; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

  if (ioptions(18) eq 1 or ioptions(18) eq 3) then fill_lav, zmat

;set flag for vertical smoothing if needed
  if (ioptions(17) eq 1) or (ioptions(17) eq 3) then smooth_vert, zmat, 3
;do horizontal smoothing if needed
  if (ioptions(17) eq 2) or (ioptions(17) eq 3) then smooth_horiz, zmat, 5

; the procedure autorange is optional and sets the zlimit automatically for
; the data, attempting to use the full dynamic range of the color table.
; you may need to supply your own procedure for your own data. 
; (in $papco/PLOT_prog/Common)

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
  if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1)  &  zmax=zscl(panel(0),2)
  endif else begin
    autorange,panel(0),time,zmat,zmax,zmin,1
    zscl(panel(0),1)=zmin  &  zscl(panel(0),2)=zmax
  endelse

  if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yen=yscl(panel(0),2)
      goto, donotzoom
      ;zoom max
      for ind=1,n_elements(ymaxarr)-1 do if ymaxarr(ind) ge yen $
                then goto,step1
      step1: if ind ge n_elements(ymaxarr) then begin
                yminarr=[yminarr,ymaxarr(n_elements(ymaxarr)-1)]
                ymaxarr=[ymaxarr,yen]
                inter=fltarr(n_elements(input_data),n_elements(yminarr))
                inter(*,0:n_elements(yminarr)-2)=zmat
                inter(*,n_elements(yminarr)-1)=0.
                zmat = inter
             endif else begin
                yminarr=yminarr(0:ind)
                ymaxarr=ymaxarr(0:ind)
                ymaxarr(ind)=yen
                zmat = zmat(*,0:ind)
             endelse
      ;zoom min
      for ind=0,n_elements(yminarr)-1 do if yminarr(ind) ge yst $
                then goto,step2
      step2: if ind eq 0 then begin
                yminarr=[yst,yminarr]
                ymaxarr=[yminarr(1),ymaxarr]
                inter=fltarr(n_elements(input_data),n_elements(yminarr))
                inter(*,0)=0.
                inter(*,1:n_elements(yminarr)-1)=zmat
                zmat = inter
             endif else begin
                Nmax = n_elements(yminarr)-1
                yminarr=yminarr(ind-1:Nmax)
                ymaxarr=ymaxarr(ind-1:Nmax)
                yminarr(0)=yst
                zmat = zmat(*,ind-1:Nmax)
             endelse
             
  donotzoom:     
             
  endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
  endelse
  
; uae auto y-labeling routine to get "good" y-lables. This sets the global IDL
; graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS - so if you want to 
; use this feature, avoid setting these or their keywords in your plot call.
  papco_y_label, yst, yen, log=yscl(panel(0),3)    
  
; set up extra plot keywords: yrange, xrange, are always needed. xtickformat
; sets the format of the x-axis ticks, default is no ticks. You can use any of
; the other keywords allowed with IDL's plot routine. But beware that some
; keywords have default values set by PAPCO.
; (Some of which you can set in the papco defaults widget):
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding values by setting keywords is possible and allowed but discouraged
; since then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
  extra_plotPar={yrange:[yst,yen], xrange:[xut1,xut2], zrange:[zmin,zmax], $
                 ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                 ztitle:uztit, ystyle:1, $
                 xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set here with the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
  
  ;check the routine papco_plot_colorspec in papco_XX\papco_lib\color_plot
  ;for all the possible options of papco_plot_colorspec.
  ;Hint: you might want to use the resx, resy keywords, which set the
  ;resolution to plot add. Without these, the routine will try to use
  ;device resolution, which for postscript may result in HUGE files!
  ;resx, resy are in data coordinates. If the axis is logarithmic, 
  ;resx, resy need to be too.
  resx=20  &  resy=0.05    
  resx=0  &  resy=0   
  papco_plot_colorspec,zmat,time,yarr, nodata=nodata, resx=resx, resy=resy, $
    xstyle=5, ystyle=5, _extra=extra_plotPar
  papco_color_bar, _extra=extra_plotPar
  
; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90  
  
; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1.
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    ;extra_plotPar=create_struct({xtickformat:'mjd_dec_1'}, $
    ;                            extra_plotPar_common) 
    ;extra_plotPar.color=1
    ;down=1
    ;plot,time,time, _extra=extra_plotPar, /NODATA
    ;x_side_label,panel,'time (UT)!CDate'
  endif
  
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
  right_side_label,panel,utitle,/rot90

end 







