;******************************************************************************
;*  NAME:        r_line
;*
;*  DESCRIPTION: Routine to read line files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1997
;*
;*  AUTOR:       S. BOURDARIE
;*
;*  CHANGES:     September 1999. Rewritten to be more general and fully
;*               adhering to papco philosophy. 
;*               June 2004. Stores time as a double instead of float
;*                          Float has 9 minute resolution; double microsecond
;*                          (Jon Niehof at BU)
;******************************************************************************
pro r_line, fln, VERBOSE=VERBOSE

   common line_data, input_header, input_data
   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

   get_err_no=0
   get_err_msg='line read succesfully'
   
; check for env. variable pointing to data directory
   data_dir='LINE_DATA'
   path=papco_getenv(data_dir)
   if strlen(path) eq 0 then begin
     get_err_no=1
     get_err_msg='Environmental variable '+data_dir+' not set'
     print,get_err_msg
     return
   endif
   
; check for existence of data directory
   cc=papco_finddir(path)
   if cc eq 0 then begin
     get_err_no=2
     get_err_msg='Data directory '+path+' not found'
     print,get_err_msg
     return
   endif
   
; check VERBOSE keyword and set
   if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   


; all times are in one file here! So read that file directly.
print,fln
   r_line_onefile, fln,/verbose
   
; restrict data to actual time range requested
   ut1=float(mjdt_start.mjd) + float(mjdt_start.t) / 86400.0d
   ut2=float(mjdt_end.mjd) + float(mjdt_end.t) / 86400.0d
   index=where((input_data.time ge ut1) and (input_data.time le ut2),count)

   if count ne 0 then begin
      input_data=input_data(index)
   endif else begin
      get_err_no=1
      get_err_msg='% r_line: no data found for time range'
   endelse      


end

;******************************************************************************
;*  NAME:        r_line_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1997
;*
;*  AUTOR:       S. BOURDARIE
;******************************************************************************
pro r_line_onefile, fln, VERBOSE=VERBOSE
  
   common line_data, input_header, input_data
   common get_error, get_err_no, get_err_msg
   
   get_err_no=0
   get_err_msg='line onefile read succesfully'
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
   on_ioerror, read_original
   
   restore,fln+'.idl',/verbose
   dealloc_lun,/quiet    ;frees logical units used by save/restore
   print,'% r_line_oneday: idl saveset read succesfully'
   return

   read_original:
   ; if the .idl save file could not be found, then try to read the original
print,'read original'
   r_line_onefile_original, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_line_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1997
;*
;*  AUTOR:       S. BOURDARIE
;*
;*  MODIFICATION HYSTORY : September 1999, by S. BOURDARIE
;******************************************************************************
pro r_line_onefile_original, fln, VERBOSE=VERBOSE
  
   common line_data, input_header, input_data
   common get_error, get_err_no, get_err_msg  
   
   get_err_no=0
   get_err_msg='line_onefile_orignal read succesfully'
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BU USER!!!!

;   on_ioerror,out
   openr,iunit,fln,/get_lun
;
;  read the header of the original file
;
   titre  = ''
   xtitre = ''
   ytitre = ''
   com1   = ''
   com2   = ''
   com3   = ''
   com4   = ''
   com5   = ''
   Nval   = LONARR(2)

   READF,iunit,titre
   READF,iunit,com1
   READF,iunit,com2
   READF,iunit,com3
   READF,iunit,com4
   READF,iunit,com5
   READF,iunit,xtitre
   READF,iunit,ytitre
   READF,iunit,Nval
   Etiquette = ''
   readf,iunit,Etiquette
   En = STRARR(Nval(1))
   READF,iunit,En

; define the header and data structures. Edit to suit your data!
   max_no_rec=Nval(0)              ;the maximum number time records in your data
;
   time_mjdt={MJDT, MJD:0l, T:0.0}
   input_header={yarr:strarr(Nval(1)), $
                             comment1:'', $
                             comment2:'', $
                             ytitle:'',$
                             unit:''}
   dummy={time:0.0D, ymat:fltarr(Nval(1)-1)}
   input_data=replicate(dummy, max_no_rec)
   
; fill header and data structures
   input_header.yarr=En
   input_header.comment1 = com1
   input_header.comment2 = com2
   input_header.ytitle = ' '

   res=str_sep(com5,'-')
   result = ''
   for i=0,n_elements(res)-2 do result = result + res(i) + '!e-'
   result = result + res(n_elements(res)-1)
   res=str_sep(result,' ') 
   result = ''
   for i=0,n_elements(res)-1 do result = result + res(i) + '!n' 
   input_header.unit='['+result+']'
        
   val = dblarr(Nval(1)) 
   for i=0L,max_no_rec-1L do begin
     READF,iunit,val
;tranform val(0) - time dec day - y,doy,hour,min,s
     aux  = val(0)
     year = floor(aux)
     Njours = 365.D0
     if (year eq 4*(year/4) and year ne 100*(year/100)) then Njours = 366.D0
     aux = (aux-double(year))*Njours
     doy = 1+floor(aux)
     aux = (aux-double(doy-1))*24.D0
     hour = floor(aux)
     aux = (aux-double(hour))*60.D0
     min = floor(aux)
     aux = (aux-double(min))*60.D0
     s = floor(aux)
     aux = (aux-double(s))
     time_mjdt=convert_date_to_t90(year=year, doy=doy ,$
                             hour=hour , minute = min , second = s,  /MJDT)
     input_data(i).time=double(time_mjdt.mjd) + $
                        (double(time_mjdt.t)+aux) / 86400.0D
;
;
;
     input_data(i).ymat=float(Val(1:Nval(1)-1))
   endfor
   close,iunit & free_lun,iunit
  
; end of user written section
;------------------------------------------------------------------------------
   
   
; save read data with correct time. Rename data to *.idl
;  save,filename=fln+'.idl',input_header,input_data,/verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='r_line_onefile:!C '+!ERR_STRING
   
end
