;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  salammbo_panel, plotinfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      plotinfo       a structure of the papco_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(plotinfo, ACTION=, GROUP=)
;*
;* MODIFICATION HISTORY:
;******************************************************************************
FUNCTION  salammbo_panel, plotinfo, GROUP=group, ACTION=ACTION

COMMON salammbo_panel_data, widgetData
COMMON salammbo

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)

IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(salammbo_control)
control = *plotinfo.USR_PTR1
;control now contains the user defined control structure for this module.

;--- create the base-window ---------------------------------------------------
title = PlotInfo.panelKind + ' - ' + papco_module_version('salammbo')
panel_base=papco_WIDGET_BASE(TITLE=title, ROW=1, GROUP_LEADER=group, /MODAL)
panel_left=widget_base(panel_base,COLUMN=1)
panel_right=widget_base(panel_base,COLUMN=1)

; All papco_panel_Add* routines are papco panel widgets that can be included
; in the panel editor or not. They provide functionality that papco supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by papco_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_left, plotinfo)
additionalWData=create_struct(widgets, additionalWData)   

;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

; --- choice for file select, or date select
run_names = papco_make_choice_names(control.run_info)
run_base = CW_BGROUP(panel_left, run_names, ROW = 1, /EXCLUSIVE, /FRAME, $
                     UVALUE='spb_run', LABEL_TOP = 'Salammbo run select:', $
                     SET_VALUE=control.run) 

dummy = WIDGET_BASE(panel_left, COLUMN = 2)

;--- select input data to code from satellites
names = input_sats & n_ch=n_elements(names) & set=bytarr(n_ch) 
for i=0,n_ch-1 DO if (control.input and 2^i) eq 2^i then set(i)=1 $
                  else set(i)=0
input_base = CW_BGROUP(dummy, names, COLUMN = 4, /NONEXCLUSIVE, /FRAME, $
                    LABEL_TOP='Select input satellite(s):', $
                    UVALUE='spb_input', SET_VALUE=set)

;--- select output satellite to "fly" through code
output_base = CW_BGROUP(dummy, output_sats, COLUMN = 4, /EXCLUSIVE, /FRAME, $
                    LABEL_TOP='Select output satellite:', $
                    UVALUE='spb_output', SET_VALUE=control.output)

;--- add Mag Field model choices ----------------------------------------------
widgets=papco_panel_Add_MagMod_Widgets(panel_left, control, /MODEL_ONLY)
additionalWData=create_struct(widgets, additionalWData)

;--- select plot products
prod_base = CW_BGROUP(panel_left, /ROW, $
         ['Original Data', 'Model output data', 'Error - (Original/Model)'], $
         LABEL_TOP='Plot Product', /EXCLUSIVE, /FRAME, $
         UVALUE='spb_product', SET_VALUE=control.product)

dummy = WIDGET_BASE(panel_left, ROW = 1)

;--- select type of plot 
graph = CW_BGROUP(dummy, ['Line', 'Spec', 'L-Spec'], COLUMN = 1, /EXCLUSIVE, $
                  LABEL_TOP='Plot Format', /FRAME, $
                  UVALUE='spb_graph', SET_VALUE=control.graph)

equ = CW_BGROUP(dummy, ['Mag. Equ.', 'Orbit'], COLUMN = 1, /EXCLUSIVE, $
                  LABEL_TOP='L-Spec Output', /FRAME, $
                  UVALUE='spb_equ', SET_VALUE=control.equ)

ech = CW_BGROUP(dummy, ['Original', 'Select'], COLUMN = 1, /EXCLUSIVE, $
                  LABEL_TOP='E-Ch. Output', /FRAME, $
                  UVALUE='spb_ech', SET_VALUE=control.ech)

etype = CW_BGROUP(dummy,['Int. (>E)', 'Diff (E1-E2)', 'Diff (E)'], COLUMN = 1,$
                  LABEL_TOP='Channel Format', /FRAME, /EXCLUSIVE, $,
                  UVALUE='spb_etype', SET_VALUE=control.etype)

init = CW_BGROUP(dummy,['default', 'prev. run'], COLUMN = 1, /EXCLUSIVE, $
                  LABEL_TOP='Init. State', /FRAME, $
                  UVALUE='spb_init', SET_VALUE=control.init)

fill_names = papco_make_choice_names(control.fill_info)
fill = CW_BGROUP(dummy,['default', 'prev. run'], COLUMN = 1, /EXCLUSIVE, $
                  LABEL_TOP='Fill. State', /FRAME, $
                  UVALUE='spb_fill', SET_VALUE=control.fill)


;--- make toggle bases to either enther channel # or energies.
channel_outer = WIDGET_BASE(panel_left, COLUMN=1, /ALIGN_CENTER)
ch_base = WIDGET_BASE(channel_outer, /ALIGN_CENTER)

chnum_base = WIDGET_BASE(ch_base, /FRAME, COLUMN=1, /align_center) 
fld_chnum = CW_FIELD(chnum_base, /COLUMN,  XSIZE=56, $
     TITLE = 'Enter list energy channel # to plot (0,1,2,...)' , $
     UVALUE='fld_coherent', VALUE = control.chnum)

chmev_base = WIDGET_BASE(ch_base, /FRAME, COLUMN=1, /align_center) 
fld_chmev = CW_FIELD(chmev_base, /COLUMN,  XSIZE=56, $
     TITLE = 'Enter list energy MeV to plot (MeV1, MeV2,...)' , $
     UVALUE='fld_coherent', VALUE = control.chmev)


; --- Plot Switches -----------------------------------------------------------
smoothNames=['none', 'vertically', 'horizontally', 'both']
smooth = CW_BGROUP(panel_left,smoothNames, ROW=1, /EXCLUSIVE, $
                  LABEL_LEFT='Smooth Plot:', /FRAME, $
                  UVALUE='spb_smooth', SET_VALUE=control.smooth)

;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- add the papco panel widget for choosing the color bar for color plots ----
widgets=papco_panel_Addcolorbar_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(panel_right, plotinfo, 0,  /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

scale_base=widget_base(panel_right, ROW=1)

;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_PANEL_AddYScale_Widgets(scale_base, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_PANEL_AddZScale_Widgets(scale_base, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_right, Action, plotinfo, $
                                     /CONFIG, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. 
thisWidgetsData={helpfile:'salammbo_panel.help', $
                 chnum_base:chnum_base, $
                 chmev_base:chmev_base, $
                 input_base:input_base, $
                 fld_chnum:fld_chnum, $
                 fld_chmev:fld_chmev, $
                 panel_base:panel_base}

; add control struct, papco_panel structs
widgetData=papco_PANEL_CreateWidgetData(control, thisWidgetsData)
widgetData=papco_PANEL_CreateWidgetData(widgetData, additionalWData)

;--- realize the window -------------------------------------------------------
salammbo_refresh, /ALL 
papco_Panel_Realize, panel_base, widgetData
papco_xmanager, 'salammbo_panel', panel_base


;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified }

additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO salammbo_refresh
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO salammbo_panel_event
;*
;* MODIFICATION HISTORY:       
;*      June 1997, written by S. BOURDARIE
;*      September 1999, modified by S. BOURDARIE 
;******************************************************************************
PRO salammbo_refresh, ALL = ALL, CHANNEL = CHANNEL

COMMON salammbo_panel_data, widgetData

IF keyword_set(ALL) THEN BEGIN
    CHANNEL = 1
ENDIF 

IF keyword_set(CHANNEL) THEN BEGIN
    wid_ids = [widgetData.chnum_base, widgetData.chmev_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid)
    CASE widgetData.ech OF
        0: map = [1, 0]         ;orig channels, use index
        1: map = [0, 1]         ;select channels, use MeV 
    ENDCASE 
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)
ENDIF 

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO salammbo_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*      June 1997, written by S. BOURDARIE
;*      September 1999, modified by S. BOURDARIE 
;******************************************************************************
PRO salammbo_panel_event, event

COMMON salammbo_panel_data, widgetData

WIDGET_CONTROL, GET_UVALUE=uval, event.id

CASE uval OF

    'spb_run': widgetData.run = event.value
    
    'spb_input': BEGIN
        WIDGET_CONTROL, widgetData.input_base, GET_VALUE=set
        widgetData.input = 0
        FOR i = 0, n_elements(set)-1 DO IF set(i) THEN $
            widgetData.input = widgetData.input+2^i
    END     
    
    'spb_output': widgetData.output = event.value
    
    'spb_product': widgetData.product = event.value

    'spb_graph': widgetData.graph = event.value
    
    'spb_equ': widgetData.equ = event.value
    
    'spb_ech': BEGIN
        widgetData.ech = event.value
        salammbo_refresh, /CHANNEL
    END 
 
    'spb_etype': widgetData.etype = event.value

    'spb_init': widgetData.init = event.value

    'spb_fill': widgetData.fill = event.value
    
    'spb_smooth': widgetData.smooth = event.value

    ELSE: BEGIN
       WIDGET_CONTROL, widgetData.fld_chnum, GET_VALUE=x
       widgetData.chnum = x
       WIDGET_CONTROL, widgetData.fld_chmev, GET_VALUE=x
       widgetData.chmev = x
       widgetData=papco_panel_Event(event, widgetData)
   ENDELSE 
   
ENDCASE 

END
