;******************************************************************************
;* file p_salammbo_spec.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_salammbo_spec, panel, plotinfo
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      plotinfo panel editor structure for plot
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and salammbo_line style.   
;* 
;* CALLING SEQUENCE:
;*       p_salammbo_spec, panel, plotinfo, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written June 1997, S. BOURDARIE
;*     modified September 1999, S. BOURDARIE
;*     modified September 2002 for SALAMBBO only, R Friedel
;******************************************************************************
PRO p_salammbo_spec, panel, plotinfo, OVERPLOT=OVERPLOT, $
                                    PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                                    SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed:
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl		       ;man/auto yscaling
COMMON zscale, zscl		       ;man/auto zscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                       ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_salammbo_spec. 
common salammbo_spec_data, input_header, input_data

COMMON papco_color_names  
color_indx=[black, red, green, blue, magenta, cyan, $
            burgundy, olive, dark_green, teal, royal_blue, violet]

papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
panelset,panel		             ;sets the panel position viewport
  
;get control variables for this module
control = *plotinfo.USR_PTR1

; restrict data to actual time range requested. 
; For plotting, always start at zero to get maximum resolution
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1  & time=data.tai-tai_1

nodata = -99.0
uytitle = input_header.units
utitle = 'salammbo '

;choose data product
CASE control.product OF
    0: BEGIN                    ;original
        zmat = data.ORIG_DATA
        utitle = utitle+'orig'
    END
    1: BEGIN                    ;model
        zmat = data.MODEL_DATA
        utitle = utitle+'model'        

    END
    2:BEGIN                     ;error
        zmat = data.error
        utitle = utitle+'error'        
   
    END
ENDCASE   

;choose int/diff etc.

energy_names = string(input_header.MEV_CH)
        
;see what channels are set
set = bytarr(input_header.nch)
IF control.chnum EQ '' THEN set(*) = 1 ELSE BEGIN
    ch_idx = fix(strsplit(control.chnum, ',', /EXTRACT))
    FOR i = 0, n_elements(ch_idx)-1 DO $
        ch_idx = ch_idx > 0 < (input_header.nch-1)
    set(ch_idx) = 1
ENDELSE 

CASE control.graph OF
    0: BEGIN                    ;line
        rot = 0

        ;set y scaling
        IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
            yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
        ENDIF ELSE BEGIN
            papco_autorange,zmat, yst, yen, $
              log=zscl(panel(0),3), exclude=0, nodata=nodata
            yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
        ENDELSE       

        extra_plotPar = {xrange:[xut1,xut2], yrange:[yst, yen], $
             ylog:yscl(panel(0),3), xtickformat:'noticks'}

        ; add keyword structure set here with the one passed in
        extra_plotPar=create_struct(extra_plotPar, extra_par)

        down=0
        papco_draw_time_axis, panel, OVERPLOT=OVERPLOT,_extra=extra_plotPar

        count=0
        FOR i = 0, input_header.nch-1 DO BEGIN 
            IF NOT set(i) THEN continue
            extra_plotPar.color=color_indx(i mod n_elements(color_indx))
            ;plot, time, zmat(i, *), _EXTRA = extra_plotPar 

            papco_gap_plot, time, zmat(i,*), nodata, xstyle = 5, ystyle = 5, $
                                             _EXTRA = extra_plotPar 

            c=count/5 & mult=count mod 5 & name = energy_names(i)
            FOR j=1,c DO  name='!C'+name
            xyouts,!P.position(0)+0.01+0.135*mult, $
                   !P.position(3)-0.015, name, /normal, $
            charsize=!p.charsize, color=extra_plotPar.color
            count=count+1  
        ENDFOR 

    END
    1: BEGIN                    ;spec
        

    END

    2: BEGIN                    ;L-spec
        rot = 1
        
        ;choose first channel set
        ch_idx = where(set EQ 1) & ch_idx = ch_idx(0)
        
        ;set y scaling - preset L range as 1 to 8
        IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
            yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
        ENDIF ELSE BEGIN
            yst = 1.0 & yen = 8.0
            yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
        ENDELSE     

        papco_make_Lvt, time, zmat(ch_idx, *), input_data.LVAL, yst, yen, $
                        Y_arr, T_cut, Zarr       
       
       ;set z-scaling
        if (zscl(panel(0),0) eq 1) then begin
            zst=zscl(panel(0),1)  &  zen=zscl(panel(0),2)
        endif else begin
            papco_autorange, Zarr, zst, zen, $
              log=zscl(panel(0),3), exclude=0, nodata = nodata
            zscl(panel(0),1)=zst  &  zscl(panel(0),2)=zen   
        ENDELSE 

        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zst,zen],$
                       ylog:0, zlog:zscl(panel(0),3), $
                       xtickformat:'noticks', $
                       ztickformat:'papco_color_bar_log_ticks', ztitle:uytitle}

        ;add keyword structure set locally and the one passed in
        extra_plotPar=create_struct(extra_plotPar, extra_par)

       ; do color plot and color bar plot. nodata specifies the "nodata" flag
       ; in your data. Here we do not plot axis and data seperate, as
       ; papco_plot_colorspec cannot be used as an overplot!
       resx=2 & dx = 4
       n_ch = n_elements(Y_arr)/2

       IF yscl(panel(0),3) THEN resy=(alog10(yen)-alog10(yst))/n_ch / 10 ELSE $
           resy=((yen)-(yst))/n_ch / 10

       papco_plot_colorspec, zarr, t_cut,y_arr, nodata=nodata, $
             resx=resx, resy=resy, $
             dx = dx, xstyle=5, ystyle=5, _extra=extra_plotPar
 
       ; store the coordinate information into common block
       plotted_x = !x  &  plotted_y = !y

       down=0
       papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar 
  
       papco_color_bar, _extra=extra_plotPar

       uytitle = 'L'

    END

ENDCASE   

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y   

; plot info at right of panel
right_side_label,panel,utitle, ROT90 = rot, _extra=extra_plotpar

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel,uytitle, /ROT90

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
IF KEYWORD_SET(OUTPUT) THEN BEGIN 

    message,'Writing plot data out to file',  /cont
    description='salammbo spec data'
    time_label='TAI'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=1e-20
    papco_write_data, time+tai1, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
ENDIF 
  
return

;jump to here if no data to plot 
noplot: 
  get_err_no = 10
  message, get_err_msg, /cont
  papco_Draw_ErrorPanel, panel, plotinfo.panelkind+': ' + get_err_msg
  plotted_x = !x  &  plotted_y = !y

END 







