;******************************************************************************
;*  NAME:        r_salammbo
;*
;*  DESCRIPTION: Routine to read salammbo files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;* Salammbo convention: All data for one run is in a file per week, file names 
;* start with date of begiining of week, eg. yyyymmdd_salammbo
;* 
;* Input satellites used in running the model are identified
;* next, seperated by underscores, eg. _1994-084_ns33.
;* 
;* The full run results for one week are given the extension .psd
;* The last state of the model is given the extension .last
;* 
;* For output files of a satellite flow through the model naming is
;* the model file plus _o_satname_psd.dat, file format is in simple
;* ascii. IDL saveset verion of this has extension .idl.
;*
;* Examples:
;* 19980927_salammbo_1994-084_ns33.psd 
;*                          - full model output for run of one week
;*                            starting 27 Sept 1998 using geo 1994-084
;*                            and gps ns33 as input
;* 19980927_salammbo_1994-084_ns33.last 
;*                          - last state of same run
;* 19980927_salammbo_1994-084_ns33_o_HEO_3_psd.dat
;*                          - model output of run along heo_3 orbit and
;*                            for heo-3 energy ranges
;* 19980927_salammbo_1994-084_ns33_o_HEO_3_psd.idl
;*                          - idl saveset verson of output
;* 
;* This routine follows the normal papco philosophy of handling the
;* request for data for a plot output. The plot output is the data as
;* a function of time of some ephemeris flown through the model. That
;* can either be a seatllite orbit, or a fictional orbit such as a
;* radial cut at the equator. The read routine looks for the output
;* file, if not found it "flies" the required orbit through the model,
;* if the model is not found it goes and runs the model, if the input
;* files for the model are not found they are constructed.
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1997
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     September 1999. Rewritten to be more general and fully
;*               adhering to papco philosophy. 
;*               May 2003: added data ingestion, GEO, GPS, HEO
;*               June 2004: Added POLAR spacecraft
;******************************************************************************
pro r_salammbo, plotinfo, VERBOSE=VERBOSE

COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

COMMON salammbo_data, input_header, input_data
COMMON salammbo
COMMON salammbo_initial,  fln_init

get_err_no=0  &  get_err_msg='salammbo read succesfully'
!QUIET = 0 

if keyword_set(OUTPUT) then output=OUTPUT else output=0
  
;make a copy of the incoming plotinfo
local_plotinfo = plotinfo
;get control variables for this module
control = *plotinfo.USR_PTR1
 
; check for env. variable pointing to data directory
if not (papco_check_data_env('SALAMMBO_DATA', path=PATH)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   

;need to make a list of files per week to be read here.
;first find date of start of week back from current papco start time.
;define start of week as SUNDAY (0).
FOR i = 0, 7 DO BEGIN
    utc_1 = {mjd:mjdt_start.mjd-i, time:mjdt_start.t*1000}
    dow = utc2dow(utc_1) & IF dow EQ 0 THEN break
ENDFOR 

;now beginning of next week forward from current papco end time.
FOR i = 0, 7 DO BEGIN
    utc_2 = {mjd:mjdt_end.mjd+i, time:mjdt_start.t*1000}
    dow = utc2dow(utc_2) & IF dow EQ 0 THEN break
ENDFOR 

n_weeks = (utc_2.mjd-utc_1.mjd) /7

;make filename
n_ch=n_elements(input_sats) & set=bytarr(n_ch) & sat_str = ''
FOR i = 0, n_ch-1 DO IF (control.input and 2^i) EQ 2^i THEN set(i)=1 $
                       ELSE set(i)=0  
idx = where(set EQ 1, c)
IF c EQ 0 THEN BEGIN
    get_err_msg = 'no input satellites chosen' & get_err_no = 1
    message, get_err_msg, /info & return
END
FOR i = 0, c-1 DO sat_str = sat_str+input_sats(idx(i))+'_'

file_id_str = '_salammbo_'+sat_str+'o_'+output_sats(control.output)+'_psd'
utc = utc_1

old_mjdt_start = mjdt_start & old_mjdt_end = mjdt_end

no_files_read=0

FOR i = 0, n_weeks-1 DO BEGIN ;loop weeks

    utc.mjd = utc_1.mjd+7*i ;step in weeks

    ;set time common for PAPCO
    mjdt_start = {mjd:utc.mjd, t:0.0}
    mjdt_end = {mjd:utc.mjd+7, t:0.0}
 
    r = anytim2utc(utc, /EXTERNAL)
    date = string(r.year, r.month, r.day, format = "(i4.4,i2.2,i2.2)")
    fln = date+file_id_str
   
    ;get the file name for the initial state to be used.
    ;this is the name of the last run + '.last'
    utc_init = utc
    utc_init.mjd = utc_init.mjd -7 ;step in weeks
    r = anytim2utc(utc_init, /EXTERNAL)
    date_init = string(r.year, r.month, r.day, format = "(i4.4,i2.2,i2.2)")  
    ;remove the output satellite from filename
    pos = strpos(file_id_str, '_o_')
    salammbo_file_id_str = strmid(file_id_str,0,pos)  
    fln_init = date_init+salammbo_file_id_str+'.last'

    ;print, mjdt_start, mjdt_end
    ;print, fln & print, fln_init

    r_salammbo_onefile, plotinfo, fln, /verbose

    ;if a file was succesfully read, concatonate!
    IF get_err_no EQ 0 THEN BEGIN  
        IF no_files_read EQ 0 THEN BEGIN
            input_data_buf=temporary(input_data)
        ENDIF ELSE BEGIN
            input_data_buf=[temporary(input_data_buf),temporary(input_data)]
        ENDELSE
        input_data=0                     ;clear array for reuse
        no_files_read=no_files_read+1   
    ENDIF

ENDFOR 

mjdt_start = old_mjdt_start & mjdt_end = old_mjdt_end

IF no_files_read NE 0 THEN BEGIN 
    input_data = temporary(input_data_buf)
ENDIF ELSE BEGIN 
    get_err_no=1  &  get_err_msg='no data found for time range'
    message, get_err_msg, /info
ENDELSE      

plotinfo = local_plotinfo

END

;******************************************************************************
;*  NAME:        r_salammbo_onefile, plotinfo
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1997
;*
;*  AUTOR:       S. BOURDARIE
;******************************************************************************
pro r_salammbo_onefile, plotinfo, fln, VERBOSE=VERBOSE
  
common salammbo_data, input_header, input_data
common get_error, get_err_no, get_err_msg
   
;make a copy of the incoming plotinfo
local_plotinfo = plotinfo

get_err_no=0  &  get_err_msg='salammbo onefile read succesfully'

IF NOT (papco_check_data_env('SALAMMBO_DATA', PATH=out_dir)) then return
IF NOT (papco_check_data_env('SALAMMBO_INPUT_DATA', PATH=in_dir)) then return
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
;get control variables for this module
control = *local_plotinfo.USR_PTR1

;if forcing a new run, delete exisiting idl file
CASE control.run OF
    0: BEGIN ;use existing, delete nothing
    END
    1: BEGIN ;new input/run/output - delete input, output and run files
        message, 'removing all input, run and output files', /info
        print, '  path : '+out_dir 
        pos = strpos(fln, '_o_')
        mask = strmid(fln, 0, pos)+'*' 
        print, '  files: ',  mask
        files = file_search(out_dir+mask)
        file_delete, files, /ALLOW_NONEXISTENT 
    END
    2: BEGIN ;new output - remove output files only
        message, 'removing model output files', /info
        print, '  path : '+out_dir
        print, '  files: ', fln+'*'
        files = file_search(out_dir+fln+'*')
        file_delete, files, /ALLOW_NONEXISTENT
    END 
    3: BEGIN ;new run and new output, keep inputs
        message, 'removing output and run files', /info
        stop
        
    END 
ENDCASE

on_ioerror, read_original
restore, out_dir+fln+'.idl', /verbose
dealloc_lun,/quiet    ;frees logical units used by save/restore
message,'read', /info
print, '  ', out_dir+fln+'.idl'
return

read_original:
; if the .idl save file could not be found, then try to read the original
r_salammbo_onefile_original, local_plotinfo, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_salammbo_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1997
;*
;*  AUTOR:       S. BOURDARIE
;*
;*  MODIFICATION HYSTORY : September 1999, by S. BOURDARIE
;******************************************************************************
pro r_salammbo_onefile_original, plotinfo, fln, $
                                 NOMAKE = NOMAKE, VERBOSE=VERBOSE
  
COMMON salammbo_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON get_error, get_err_no, get_err_msg  

get_err_no=0  &  get_err_msg='salammbo_onefile_orignal read succesfully'
  
if not (papco_check_data_env('SALAMMBO_DATA', PATH=out_dir)) then return
 
; clear data arrays for reuse
input_header=0  &  input_data=0

;get control variables for this module
control = *plotinfo.USR_PTR1

;------------------------------------------------------------------------------
; read the original data file

;set variable needed for reading
ndat = 0l & nch = 0l
on_ioerror, out
openr, u, out_dir+fln+'.dat', /get_lun

message,'Reading', /info
print, '  ', out_dir+fln+'.dat' & print, ''

;get some timestrings from papco time common block
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
r = tai2utc(tai_1, /EXTERNAL)
st_sec = UTC2TAI({year:r.year, MONTH:1, DAY:1, $
                  HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
en_sec = UTC2TAI({year:r.year+1, MONTH:1, DAY:1, $
                  HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
year_sec = en_sec-st_sec

; read the header of the original file
readf, u, ndat, nch

;make data arrays, read
units = ''
mev_ch = dblarr(nch)

readf, u, units
readf, u, mev_ch

input_header = {ndat:ndat, nch:nch, mev_ch:mev_ch, units:units}
dmy1 = {time:0.0d, $
        orig_data:dblarr(nch), model_data:dblarr(nch), error:dblarr(nch), $
        lval:0.0d, bbequ:0.0d}
dmy2 = create_struct('tai', 0.0d, dmy1) 
input_data = replicate(dmy2, ndat)

FOR i = 0, ndat-1 DO begin
    readf, u, dmy1
    dmy2 = create_struct('tai', 0.0d, dmy1) 
    ;convert time from decimal year to tai
    dmy2.tai =  st_sec + (dmy1.time - r.year) * year_sec
    input_data(i) = dmy2
ENDFOR 
    
close, u & free_lun, u

; end of user written section
;------------------------------------------------------------------------------

; save read data with correct time. Rename data to *.idl
message, 'saving to idl saveset', /info
print, '  ', out_dir+fln+'.idl' 
save, filename=out_dir+fln+'.idl', input_header, input_data, $
                                   /compress, /verbose  
dealloc_lun, /quiet              ;frees logical units used by save/restore
return
  
out:
;data does not exist, run the Salammbo Code to do the simulation run.
;time for run is always one week, time is in common mjdt

;remove the output satellite from filename
pos = strpos(fln, '_o_')
salammbo_run_fln = strmid(fln,0,pos)

in_fln = fln

IF NOT keyword_set(NOMAKE) THEN BEGIN
    m_salammbo, plotinfo, in_fln
    IF get_err_no EQ 0 THEN $ ;if succesfull make, read data
        r_salammbo_onefile_original, plotinfo, fln, /NOMAKE
ENDIF
   
END 


;*****************************************************************************
; Routine to run the salammbo code within the PAPCO ebvironment.
; 
; uses PAPCO read routines to access all data needed, writes input
; files for SALAMMBO to use (both data files and namelists), runs the
; code, then runs the codes to produce output fiels for plotting in
; PAPCO. 
;*****************************************************************************
PRO m_salammbo, plotinfo, fln, OUTSAT = OUTSAT

COMMON PLOT_COMPOSER, widgetData
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON get_error, get_err_no, get_err_msg  

COMMON yscale, yscl		      ;man/auto yscaling
COMMON zscale, zscl		      ;man/auto yscaling
COMMON extra, extra_plotinfo

COMMON salammbo
COMMON salammbo_initial, fln_init

get_err_no = 0 & get_err_msg = '' 

;buffer axis scaling into. Data is made by call to plot routines,
;which may overwrite these...
old_yscl = yscl & old_zscl = zscl

papco, /set_only

if not (papco_check_data_env('SALAMMBO_DATA', PATH=out_dir)) then return
if not (papco_check_data_env('SALAMMBO_INPUT_DATA', PATH=in_dir)) then return
if not (papco_check_data_env('SALAMMBO_CODE', PATH=code_dir)) then return
if not (papco_check_data_env('SALAMMBO_LOGS', PATH=logs_dir)) then return

yscl = fltarr(10, 4) & zscl = fltarr(10, 4)
extra_plotinfo = {color:0, thick:1, psym:0, linestyle:0}

;get control variables for this module
control = *plotinfo.USR_PTR1

;get some timestrings from papco time common block
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})   

r = tai2utc(tai_1, /EXTERNAL)
st_sec = UTC2TAI({year:r.year, MONTH:1, DAY:1, $
                  HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
en_sec = UTC2TAI({year:r.year+1, MONTH:1, DAY:1, $
                  HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
year_sec = en_sec-st_sec

;make some required filenames

;remove the output satellite from filename
pos = strpos(fln, '_o_')
salammbo_run_fln = strmid(fln,0,pos)

out_file = out_dir+salammbo_run_fln+'.psd'
output_file = out_dir+fln+'.dat'
log_file = logs_dir+salammbo_run_fln
last_file = out_dir+salammbo_run_fln+'.last'

IF keyword_set(OUTSAT) THEN BEGIN
    out_fln_str = '_out' 
ENDIF ELSE BEGIN
    out_fln_str = ''
ENDELSE

date = strmid(fln, 0, 8)

;set is the index into input_sats required for salammbo
n_sat=n_elements(input_sats) & set=bytarr(n_sat) & sat_str = ''
FOR i = 0, n_sat-1 DO IF (control.input and 2^i) EQ 2^i THEN set(i)=1 $
                                                        ELSE set(i)=0
input_files = strarr(n_sat) & sat_count = 0

;******************************************************************************
;* Section 1: read all required input data and write out in correct
;*            format for  use by salammbo code. For this we need to
;*            read both spectra and magnetic field coordinate data for
;*            the correct mapping of data into the code.
;******************************************************************************

;all input data needs to be one one minute time resolution for now.
;make a time array to use
av = 60.0d & nn = long((tai_2-tai_1) / av) 
new_tai = dblarr(nn) & salammbo_t = dblarr(nn)

FOR k = 0, nn-1 DO new_tai(k) = tai_1+k*av+av/2.0
salammbo_t = r.year + (new_tai-st_sec)/year_sec

;make the tag name for the  mag field info structure requested
IF control.mag_dyn THEN dyn_str = 'DYNA' ELSE dyn_str = 'STAT'
mag_info_name = 'MODEL_' + varprt(control.internal_model) + '_' + $
                           varprt(control.external_model)+'_'+dyn_str

;------------------------------------------------------------------------------
; Do for LANL geosynchronous electron data. Both data and ephemeris
; are already per minute.
;------------------------------------------------------------------------------

COMMON lanl_geo, geo_sat_names, geo_select_names, geo_select_frmts, $
                 lanl_geo_control
COMMON slice_lanl_geo, geo_time, geo_ymat, geo_utitle, geo_uytitle, geo_ych
COMMON geo_eph_data, geo_eph_header, geo_eph_data
COMMON lanl_geo_eph, geo_eph_sat_names, geo_eph_names,  $
                     geo_eph_yunits, geo_eph_control

;subset of geo sats
geo_set = set & geo_set(7:n_sat-1) = 0 & sat_idx = where(geo_set EQ 1, c_geo)
IF c_geo EQ 0 THEN GOTO, done_geo

FOR j = 0, c_geo-1 DO BEGIN
    get_err_no = 0
    geo_sat = input_sats(sat_idx(j))
    idx = where(geo_sat EQ geo_sat_names)

    ;see if file is aleady made!
    geo_out_fln = in_dir + date+'_lanl_geo_'+geo_sat+out_fln_str+'.dat'
    res = findfile(geo_out_fln, count = c)
    IF (control.run EQ 1) THEN c = 0
    IF c EQ 1 THEN BEGIN
        message, 'Have ', /info & print, '  ', geo_out_fln
        input_files(sat_count) = geo_out_fln & sat_count = sat_count+1
        continue
    ENDIF

    ;get geo data
    new_plotInfo = papco_getplotinfostruct()
    new_control = lanl_geo_control
    new_control.res = 1         ;time res 1 min
    new_control.cal = 1         ;calib = flux
    new_control.ek1 = 0         ;select channels to use - the first 9
    new_control.ek2 = 0
    FOR i=0, 8 DO new_control.ek1 = new_control.ek1 + 2ll^i
    new_control.sc_id = idx(0)   ;satellite index for geo
    new_plotinfo.USR_PTR1 = ptr_new(new_control)
    output = 2 & panel = [0, 1, 1] & IsRead = 0
 
    get_err_no = 0
    draw_lanl_geo, panel, new_plotInfo, orbitNo, $
                   get_data_call, slice_type, IsRead, OUTPUT=OUTPUT
    IF get_err_no NE 0 THEN CONTINUE

    idx1 = where(geo_ymat LE 0, c)
    IF c NE 0 THEN geo_ymat(idx1) = -99.0

    ;get the geo ephemeris and magnetic model data.
    new_plotInfo = papco_getplotinfostruct()
    new_control = geo_eph_control
    index = where(geo_eph_sat_names EQ geo_sat, c)

    new_control.sc_id = idx(0)
    new_control.product = 0 ;choose L, arbitrary here
    new_control.orig = 1 ;model coords
    new_control.internal_model = control.internal_model
    new_control.external_model = control.external_model
    new_control.mag_proc = control.mag_proc
    new_control.l_calc = control.l_calc
    new_control.mag_dyn = control.mag_dyn
    new_plotinfo.USR_PTR1 = ptr_new(new_control)

    r_geo_eph, new_plotInfo, AVERAGE=60, VERBOSE=VERBOSE

    ;restrict eph data in time
    index=where((geo_eph_data.tai ge tai_1) AND (geo_eph_data.tai le tai_2),c)
    IF c NE  0 THEN geo_eph_data=geo_eph_data(index) ELSE BEGIN
        message, 'No GEO EPH  data'
    END

    mev_ch = sqrt(geo_ych(*, 0)*geo_ych(*, 1)) / 1000.0
    nch = n_elements(mev_ch)

    tags = tag_names(geo_eph_data) & idx = where(tags EQ mag_info_name, c)
    IF c EQ 0 THEN BEGIN
        message, 'Mag info '+mag_info_name+' does not exist', /info
        CONTINUE
    ENDIF

    IF control.l_calc THEN lval = geo_eph_data.(idx(0)).Lstar(0) $
    ELSE lval = geo_eph_data.(idx(0)).L(0)
    b = sqrt(geo_eph_data.(idx(0)).B1^2 + $
             geo_eph_data.(idx(0)).B2^2 + $
             geo_eph_data.(idx(0)).B3^2)
    bmin = geo_eph_data.(idx(0)).Bmin
    bbmin = b/bmin
    idx = where(bbmin LT 1.0, c) & IF c NE 0 THEN bbmin(idx) = 1.0
    idx = where((lval EQ 100) OR (bmin EQ 100), c)
    IF c NE 0 THEN BEGIN
        lval(idx) = -99.0 & bbmin(idx) = -99.0
    ENDIF

    ;convert geo flux per keV to per MeV, preserve bad data flag
    idx = where(geo_ymat NE -99.0, c)
    IF c NE 0 THEN geo_ymat(idx) = geo_ymat(idx)*1000.0

    ;write out lanl geo data in required format. 
    ndat = n_elements(salammbo_t)
    openw, u, geo_out_fln, /GET_LUN, /F77_UNFORMATTED
    writeu, u, ndat, nch                               ;long I4, long I4,
    writeu, u, double(mev_ch)                          ;double R8 * nch
    FOR i = 0, ndat-1 DO BEGIN
        writeu, u, double(salammbo_t(i)), $ ;double R8
                   double(geo_ymat(i, *)), $           ;double R8 * nch
                   double(lval(i)), $                  ;double R8
                   double(bbmin(i))                    ;double R8
    ENDFOR 
    close, u &  free_lun, u
    message, 'Written LANL GEO '+geo_sat+' data for Salammbo', /info
    print, '  ', geo_out_fln
    input_files(sat_count) = geo_out_fln & sat_count = sat_count+1

ENDFOR
print, ''
done_geo:

;------------------------------------------------------------------------------
; Do for LANL GPS electron data
;------------------------------------------------------------------------------

COMMON lanl_gps
COMMON slice_lanl_gps_4, time, yray, yarr, extra_plotPar, $
                         utitle, uytitle, uztit, tai1
COMMON gps_eph, gps_eph_sat_id, gps_eph_eph_names, gps_eph_select_names, $
                  gps_eph_control
COMMON gps_eph_data, gps_eph_header, gps_eph_data

;subset of gps sats
gps_set = set & gps_set(0:6) = 0 & gps_set(11:n_sat-1) = 0
sat_idx = where(gps_set EQ 1, c_gps)

IF c_gps EQ 0 THEN GOTO, done_gps

FOR j = 0, c_gps-1 DO BEGIN
    get_err_no = 0  
    ;get gps data
    gps_sat = input_sats(sat_idx(j)) & idx = where(gps_sat EQ sat_id)

    ;see if file is aleady made!
    gps_out_fln = date+'_lanl_gps_'+gps_sat+out_fln_str+'.dat'

    IF control.run GT 0 THEN BEGIN ;delete "output" input files
        message, 'removing out model input files', /info
        print, '  path : '+ in_dir
        print, '  file = '+ gps_out_fln
        files = file_search(in_dir+gps_out_fln)
        file_delete, files, /ALLOW_NONEXISTENT
    ENDIF 
    gps_out_fln_tmp = '/tmp/'+gps_out_fln
    gps_out_fln = in_dir+gps_out_fln

    res = findfile(gps_out_fln, count = c)

    IF c EQ 1 THEN BEGIN
        message, 'Have', /info & print, '  ', gps_out_fln
        input_files(sat_count) = gps_out_fln & sat_count = sat_count+1
        continue
    ENDIF

    new_plotinfo = papco_getplotinfostruct()
    new_control = lanl_gps_control
    new_control.sc_id = idx(0)
    new_control.plot_style = 0
    new_control.dead_time = 1
    IF keyword_set(OUTSAT) THEN cal = 6 ELSE cal = 7
    new_control.cal = cal                     ;int./ diff flux (coherent)
    new_plotinfo.USR_PTR1 = ptr_new(new_control)

    output = 2 & panel = [0, 1, 1] & IsRead = 0
 
    get_err_no = 0
    draw_lanl_gps, panel, new_plotInfo, orbitNo, $
                   get_data_call, slice_type, IsRead, OUTPUT=OUTPUT
    IF get_err_no NE 0 THEN CONTINUE

    gps_ymat = transpose(yray)
    gps_time = time
    gps_ych = yarr

    ;get gps ephemeris data

    idx = where(gps_sat EQ gps_eph_sat_id)

    new_plotinfo = papco_getplotinfostruct()
    new_control = gps_eph_control
    new_control.sc_id = idx(0)
    new_control.product = 0 ;choose L, arbitrary here
    new_control.orig = 1 ;model coords
    new_control.internal_model = control.internal_model
    new_control.external_model = control.external_model
    new_control.mag_proc = control.mag_proc
    new_control.mag_dyn = control.mag_dyn
    new_control.l_calc = control.l_calc
    new_plotinfo.USR_PTR1 = ptr_new(new_control)

    r_gps_eph, new_plotinfo, VERBOSE=VERBOSE

    ;restrict eph data in time
    index=where((gps_eph_data.time GE tai_1)AND(gps_eph_data.time LE tai_2),c)
    IF c NE  0 THEN gps_eph_data=gps_eph_data(index) ELSE BEGIN
        message, 'No GPS EPH  data'
    END

    ;time format required by salammbo is decimal year.
    gps_eph_t = r.year+ (gps_eph_data.time-st_sec) / year_sec

    ;input to code - diff channel, output integral
    IF keyword_set(OUTSAT) THEN mev_ch = gps_ych(*, 0) $
                           ELSE mev_ch = sqrt(gps_ych(*, 0)*gps_ych(*, 1))

    nch = n_elements(mev_ch)

    tags = tag_names(gps_eph_data) & idx = where(tags EQ mag_info_name, c)
    IF c EQ 0 THEN BEGIN
        message, 'Mag info '+mag_info_name+' does not exist', /info
        CONTINUE
    ENDIF

    IF control.l_calc THEN lval = gps_eph_data.(idx(0)).Lstar(0) $
    ELSE lval = gps_eph_data.(idx(0)).L(0)
    b = sqrt(gps_eph_data.(idx(0)).B1^2 + $
             gps_eph_data.(idx(0)).B2^2 + $
             gps_eph_data.(idx(0)).B3^2)
    bmin = gps_eph_data.(idx(0)).Bmin
    bbmin = b/bmin
    idx = where(bbmin LT 1.0, c) & IF c NE 0 THEN bbmin(idx) = 1.0
    idx = where((lval EQ 100) OR (bmin EQ 100), c)
    IF c NE 0 THEN BEGIN
        lval(idx) = -99.0 & bbmin(idx) = -99.0
    ENDIF

    ;gps data is at 96 sec res, need one min. For a week, 6300 point array 
    ;original goes to 10080 points

    ;interpolate preserving bad data flags!
    message, 'Interpolating GPS to 1 min res, preseving bad data flags', /info

    nodata = -99.0
    new_lval = papco_interpol(lval, gps_eph_t, salammbo_t, NODATA = nodata)
    new_bbmin =papco_interpol(bbmin, gps_eph_t, salammbo_t, NODATA = nodata)
    
    new_yarr = fltarr(nn, nch)
    FOR i = 0, nch-1 DO BEGIN
        y = gps_ymat(*, i)
        new_y = papco_interpol(y, gps_eph_t, salammbo_t, NODATA = nodata)
        new_yarr(*, i) = new_y
    ENDFOR

    ;write out lanl gps data in required format.  first into /tmp/..
    ndat = n_elements(salammbo_t)
    openw, u, gps_out_fln_tmp, /GET_LUN, /F77_UNFORMATTED
    writeu, u, ndat, nch                               ;long I4, long I4,
    writeu, u, double(mev_ch)                          ;double R8 * nch
    FOR i = 0, ndat-1 DO BEGIN
        writeu, u, double(salammbo_t(i)), $ ;double R8
                   double(new_yarr(i, *)), $           ;double R8 * nch
                   double(new_lval(i)), $              ;double R8
                   double(new_bbmin(i))                ;double R8
    ENDFOR 
    close, u &  free_lun, u
    
    ; then move whole file to destination directory
    file_move, gps_out_fln_tmp, gps_out_fln, /overwrite
    message, 'Written LANL GPS '+gps_sat+' data for Salammbo', /info
    print, '  ', gps_out_fln
    input_files(sat_count) = gps_out_fln & sat_count = sat_count+1

ENDFOR
print, ''
done_gps:

;------------------------------------------------------------------------------
; Do for Aerospace HEO electron data
;------------------------------------------------------------------------------

COMMON aero_heo_data, aero_heo_header, aero_heo_data
COMMON heo_eph_data, heo_eph_header, heo_eph_data
COMMON aero_heo_slice, heo_time, heo_ymat, heo_yarr, heo_extra_plotPar, $
                       heo_utitle, heo_uytitle
COMMON aero_heo
COMMON heo_eph

;subset of heo sats
heo_set = set & heo_set(0:10) = 0 & heo_set(13) = 0
sat_idx = where(heo_set EQ 1, c_heo)
IF c_heo EQ 0 THEN GOTO, done_heo

FOR j = 0, c_heo-1 DO BEGIN
    
    ;get heo data
    heo_sat = input_sats(sat_idx(j)) & idx = where(heo_sat EQ aero_heo_sats)

    ;see if file is aleady made!
    heo_out_fln = in_dir + date+'_aero_heo_'+heo_sat+out_fln_str+'.dat'
    res = findfile(heo_out_fln, count = c)
    IF control.run EQ 1 THEN c = 0

    IF c EQ 1 THEN BEGIN
        message, 'Have', /info & print, '  ', heo_out_fln
        input_files(sat_count) = heo_out_fln & sat_count = sat_count+1
        continue
    ENDIF

    new_plotinfo = papco_getplotinfostruct()
    new_control = aero_heo_control
    new_control.sat_id = idx(0)
    FOR i = 0, 5 DO $ ;choose first 6 electron channels
        new_control.product = new_control.product+2l^i
    IF keyword_set(OUTSAT) THEN cal = 1 ELSE cal = 2
    new_control.cal = cal                 ;int./ diff flux (coherent)
    new_plotinfo.USR_PTR1 = ptr_new(new_control)

    output = 2 & panel = [0, 1, 1] & IsRead = 0
 
    get_err_no = 0
    draw_aero_heo, panel, new_plotInfo, orbitNo, $
                   get_data_call, slice_type, IsRead, OUTPUT=OUTPUT    
    IF get_err_no NE 0 THEN CONTINUE

    ;heo data now in COMMON aero_heo_slice

    idx = where(heo_ymat LE 0, c) & IF c NE 0 THEN heo_ymat(idx) = -99.0

    ;input to code - diff channel, output integral
    IF keyword_set(OUTSAT) THEN mev_ch = heo_yarr(0:5) $
                           ELSE mev_ch = sqrt(heo_yarr(i+1)*heo_yarr(i))

    ;get heo ephemeris data
    idx = where(heo_sat EQ heo_eph_sats)

    new_plotinfo = papco_getplotinfostruct()
    new_control = heo_eph_control
    new_control.sat_id = idx(0)
    new_control.product = 0
    new_control.orig = 1
    new_control.internal_model = 0 ;IGRF  
    new_control.external_model = 5 ;Olson Pfitzer quiet 77
    new_control.l_calc = control.l_calc
    new_control.mag_proc = control.mag_proc
    new_control.mag_dyn = control.mag_dyn
    new_plotinfo.USR_PTR1 = ptr_new(new_control)

    r_heo_eph, new_plotinfo, VERBOSE=VERBOSE

    ;time format required by salammbo is decimal year.
    ;for heo, both data and eph are on same timebase
    heo_eph_t = r.year+ (heo_time+tai_1-st_sec) / year_sec
    
    lval = heo_eph_data.MODEL_0_5.L
    b = heo_eph_data.MODEL_0_5.B_GSM(3)
    bequ = heo_eph_data.MODEL_0_5.B_EQU

    idx = where((lval LT 0) OR (bequ LE 0) OR (b LE 0), c)
    IF c NE 0 THEN BEGIN
        lval(idx) = -99.0 & b(idx) = -99.0 & bequ(idx) = 1
    ENDIF
    bbequ = b/bequ

    ;need to interpolate heo data onto one minute resolution

    ;make time array required. 
    av = 60.0d & ndat = long((tai_2-tai_1) / av)
    new_heo_t = dblarr(ndat)
    FOR k = 0, ndat-1 DO new_heo_t(k) = $
        r.year + (tai_1 + k*av+av/2.0 - st_sec ) / year_sec    
    
    ;heo data is at 15 sec res, need one min. For a week, 40185 point array 
    ;original goes to 10080 points

    ;interpolate preserving bad data flags!
    message, 'Interpolating HEO to 1 min res, preseving bad data flags', /info
    nodata = -99.0

    new_lval = papco_interpol(lval, heo_eph_t, new_heo_t, NODATA = nodata)
    new_bbequ =papco_interpol(bbequ, heo_eph_t, new_heo_t, NODATA = nodata)
    
    ndat = n_elements(new_heo_t)
    nch  = n_elements(mev_ch)

    new_yarr = fltarr(ndat, nch)
    FOR i = 0, nch-1 DO BEGIN
        if keyword_set(OUTSAT) then y=heo_ymat(*, i) else $
          y = heo_ymat(*, i)-heo_ymat(*,i+1)
        new_y = papco_interpol(y, heo_eph_t, new_heo_t, NODATA = nodata)
        new_yarr(*, i) = new_y
    ENDFOR

    ;write out aero heo data in required format. 
    ndat = n_elements(new_heo_t)
    openw, u, heo_out_fln, /GET_LUN, /F77_UNFORMATTED
    writeu, u, ndat, nch                               ;long I4, long I4,
    writeu, u, double(mev_ch)                          ;double R8 * nch
    FOR i = 0, ndat-1 DO BEGIN
        writeu, u, double(new_heo_t(i)), $             ;double R8
                   double(new_yarr(i, *)), $           ;double R8 * nch
                   double(new_lval(i)), $              ;double R8
                   double(new_bbequ(i))                ;double R8
    ENDFOR 
    close, u &  free_lun, u
    message, 'Written AERO HEO '+heo_sat+' data for Salammbo', /info
    print, '  ', heo_out_fln
    input_files(sat_count) = heo_out_fln & sat_count = sat_count+1
    
ENDFOR

print, ''
done_heo:

;------------------------------------------------------------------------------
; Do for POLAR CEPPAD electron data - combine IES and HISTe!
;------------------------------------------------------------------------------
COMMON polar_ceppad_slice, polar_time, polar_yarr, polar_zmat, polar_data, $
               polar_extra_plotPar, polar_utitle, polar_uytitle, polar_uztitle
COMMON polar_ceppad, polar_typeNames, polar_pitchNames, $
               polar_select_names, polar_select_range, polar_select_frmts, $
               polar_average_names, polar_average_range, polar_average_frmts, $
               polar_ceppad_control
COMMON polar_ephemeris, polar_eph_orig_names, polar_eph_control
COMMON polar_eph, polar_eph__header, polar_eph_data 

polar_set = set(13)
IF polar_set EQ 0 THEN GOTO, done_polar

;get polar ceppad data for IES and HISTe

;see if file is aleady made!
polar_out_fln = in_dir + date+'_polar_ceppad'+out_fln_str+'.dat'
res = findfile(polar_out_fln, count = c)
IF control.run EQ 1 THEN c = 0

IF c EQ 1 THEN BEGIN
    message, 'Have', /info & print, '  ', polar_out_fln
    input_files(sat_count) = polar_out_fln & sat_count = sat_count+1
    GOTO, done_polar
ENDIF

new_plotinfo = papco_getplotinfostruct()
new_control = polar_ceppad_control

new_control.pl_type = 0  ;0: Ek @ Det/Sec
new_control.cal = 1      ;Flux
new_control.reload = 0   ;always reload data
new_control.spec = 1     ;color plot
new_control.mask = 0     ;no mask
new_control.bckgnd = 1   ;take out background
new_control.satur = 1    ;ignore saturated levels
new_control.crosscal = 1 ;do spacecarft inter-adjust

;get IES data
new_control.product = 1 ;IES SURVEY
new_plotinfo.USR_PTR1 = ptr_new(new_control)
draw_polar_ceppad, [0, 1, 1], new_plotinfo, 0, '', 0, 0, OUTPUT = 2
IF get_err_no NE 0 THEN GOTO, done_polar

ies_polar_yarr = polar_yarr(1:8, *)
ies_polar_zmat = polar_zmat(*, 1:8)
ies_polar_time = polar_time

;time format required by salammbo is decimal year.
decyr_ies_polar = r.year+ (ies_polar_time+tai_1-st_sec) / year_sec

message, 'Interp. POLAR IES to 1 min res, preserving bad data flags', /info
nodata = 1.0e-20
new_ies_polar_yarr = fltarr(nn, 8)
FOR i = 0, 7 DO BEGIN
    y = ies_polar_zmat(*, i)
    new_y = papco_interpol(y, decyr_ies_polar, salammbo_t, NODATA = nodata)
    new_ies_polar_yarr(*, i) = new_y
ENDFOR

;get HISTe data 
new_control.product = 4 ;HISTe Spin_16_Z4
new_control.hist =  1   ;ABC mode
new_plotinfo.USR_PTR1 = ptr_new(new_control)
draw_polar_ceppad, [0, 1, 1], new_plotinfo, 0, '', 0, 0, OUTPUT = 2
IF get_err_no NE 0 THEN GOTO, done_polar

hist_polar_yarr = polar_yarr(2:14, *)
hist_polar_zmat = polar_zmat(*, 2:14)
hist_polar_time = polar_time

;now HIST data switches modes, so take out those times which have no
;data (wrong mode)
FOR i = 0, n_elements(polar_time)-1 DO BEGIN
    spec = hist_polar_zmat(i, *)
    idx = where(spec NE nodata, c)
    IF c EQ 0 THEN hist_polar_zmat(i, *) = 0
ENDFOR
idx = where(hist_polar_zmat(*, 0) NE 0, c)
IF c NE 0 THEN BEGIN
    hist_polar_zmat = hist_polar_zmat(idx, *)
    hist_polar_time = hist_polar_time(idx)
ENDIF

;time format required by salammbo is decimal year.
decyr_hist_polar = r.year+ (hist_polar_time+tai_1-st_sec) / year_sec

message, 'Interp. POLAR HISTe to 1 min res, preserving bad data flags', /info
nodata = 1.0e-20
new_hist_polar_yarr = fltarr(nn, 13)
FOR i = 0, 12 DO BEGIN
    y = hist_polar_zmat(*, i)
    new_y = papco_interpol(y, decyr_hist_polar, salammbo_t, NODATA = nodata)
    new_hist_polar_yarr(*, i) = new_y
ENDFOR

;make combined channel array
polar_yarr = [ies_polar_yarr, hist_polar_yarr] / 1000.0
;input to code - diff channel, output integral
IF keyword_set(OUTSAT) THEN mev_ch = polar_yarr(*, 0) $
                       ELSE mev_ch = sqrt(polar_yarr(*, 0)*polar_yarr(*, 1))
nch  = n_elements(mev_ch)

;make combined data array
polar_zmat = transpose([transpose(new_ies_polar_yarr), $
                        transpose(new_hist_polar_yarr)])
idx = where(polar_zmat EQ nodata, c)
IF c NE 0 THEN polar_zmat(idx) = -99.0
    
;convert polar flux per keV to per MeV, preserve bad data flag
idx = where(polar_zmat NE -99.0, c)
IF c NE 0 THEN polar_zmat(idx) = polar_zmat(idx)*1000.0

;get ephemeris data
new_plotinfo = papco_getplotinfostruct()
new_control = polar_eph_control
new_control.product = 0
new_control.orig = 1
new_control.internal_model = 0 ;IGRF  
new_control.external_model = 5 ;Olson Pfitzer quiet 77
new_control.l_calc = control.l_calc
new_control.mag_proc = control.mag_proc
new_control.mag_dyn = control.mag_dyn
new_plotinfo.USR_PTR1 = ptr_new(new_control)

r_polar_eph, new_plotinfo
;time format required by salammbo is decimal year.
decyr_eph_polar = r.year+ (polar_eph_data.tai-st_sec) / year_sec

tags = tag_names(polar_eph_data) & idx = where(tags EQ mag_info_name, c)
IF c EQ 0 THEN BEGIN
    message, 'Mag info '+mag_info_name+' does not exist', /info
    GOTO, done_polar
ENDIF

IF control.l_calc THEN lval = polar_eph_data.(idx(0)).LSTAR(0) $
ELSE lval = polar_eph_data.(idx(0)).L(0)
b = sqrt(polar_eph_data.(idx(0)).B1^2 + $
         polar_eph_data.(idx(0)).B2^2 + $
         polar_eph_data.(idx(0)).B3^2)
bmin = polar_eph_data.(idx(0)).Bmin
bbmin = b/bmin
idx = where(bbmin LT 1.0, c) & IF c NE 0 THEN bbmin(idx) = 1.0

idx = where((lval EQ 100) OR (bmin EQ 100), c)
IF c NE 0 THEN BEGIN
    lval(idx) = -99.0 & bbmin(idx) = -99.0
ENDIF

;interpolate preserving bad data flags!
message, 'Interp. POLAR eph to 1 min res, preserving bad data flags', /info
nodata = -99.0
new_lval = papco_interpol(lval, decyr_eph_polar, salammbo_t, NODATA = nodata)
new_bbmin = papco_interpol(bbmin, decyr_eph_polar, salammbo_t, NODATA = nodata)

;write out aero polar data in required format. 
openw, u, polar_out_fln, /GET_LUN, /F77_UNFORMATTED
writeu, u, nn, nch                                 ;long I4, long I4,
writeu, u, double(mev_ch)                          ;double R8 * nch
FOR i = 0, nn-1 DO BEGIN
    spec = transpose(polar_zmat(i, *))
    writeu, u, double(salammbo_t(i)), $            ;double R8
               double(spec), $                     ;double R8 * nch
               double(new_lval(i)), $              ;double R8
               double(new_bbmin(i))                ;double R8
ENDFOR 
close, u &  free_lun, u
message, 'Written POLAR data for Salammbo', /info
print, '  ', polar_out_fln
input_files(sat_count) = polar_out_fln & sat_count = sat_count+1

print, ''
done_polar:

;------------------------------------------------------------------------------
; If we're only making data files for a test satellite then we're done here
;------------------------------------------------------------------------------

IF keyword_set(OUTSAT) THEN BEGIN    
    fln = input_files(0) & yscl = old_yscl & zscl = old_zscl & return 
ENDIF

;------------------------------------------------------------------------------
; Do for OMNI data - parameters needed by Salammbo
;------------------------------------------------------------------------------

COMMON omni_data, omni_header, omni_data

;see if file is aleady made!
omni_out_fln = in_dir + date+'_omni.dat'
res = findfile(omni_out_fln, count = c)
IF control.run EQ 1 THEN c = 0
IF c EQ 1 THEN BEGIN
    message, 'Have', /info & print, '  ', omni_out_fln & GOTO, done_omni
ENDIF

new_plotInfo = papco_getplotinfostruct()
new_plotinfo.typevector(1) = 0 ;do not update

r_omni, new_PlotInfo, VERBOSE=VERBOSE

;restrict data in time
index=where((omni_data.tai ge tai_1) AND (omni_data.tai le tai_2),c)
IF c NE  0 THEN data=omni_data(index) ELSE BEGIN
    message, 'No OMNI data'
END 

;time format required by salammbo is decimal year.
decimal_year_time = r.year+ (data.tai-st_sec) / year_sec

;write out omni data in required format
openw, u, omni_out_fln, /GET_LUN, /F77_UNFORMATTED
ndat = n_elements(data)
writeu, u, ndat
FOR i = 0, ndat-1 DO BEGIN
    writeu, u, double(decimal_year_time(i)), $      ;double R8
               double(data(i).BZ_GSM), $            ;double R8
               double(data(i).PROTON_DENSITY), $    ;double R8
               double(data(i).PLASMA_BULK_SPEED), $ ;double R8
               long(data(i).KP), $                  ;long   I4
               long(data(i).DST)                    ;long   I4
ENDFOR               

close, u &  free_lun, u
message, 'Written OMNI data for Salammbo', /info
print, '  ',omni_out_fln 
done_omni:
print, ''

;******************************************************************************
; make required output file. It's the same format as the input
; files. These are read by Salammbo, the L, B/B0 info is used to "fly"
; through the code, and the code output is added to the file, plus an
; estimate of the error between the code and the original data.
;******************************************************************************
output_plotinfo = papco_getplotinfostruct()
output_plotinfo.USR_PTR1 = ptr_new(salammbo_control)
output_control = *output_plotinfo.USR_PTR1

;set the input_sat tag to the required output satellite, and
;recursively call m_salammbo with a keyword OUTSAT
output_sat = output_sats(control.output)
idx = where(input_sats EQ output_sat)
output_control.input = 2^idx(0)
output_control.run = control.run
*output_plotinfo.USR_PTR1 = output_control

m_salammbo, output_plotinfo, fln, /OUTSAT
input_file = fln

;******************************************************************************
;now write the salammbo namelist file needed!

;assemble list of salammbo input files.
message, 'SALAMMBO input files:', /info
FOR i = 0, sat_count-1 DO print, input_files(i)
print, ''
print, omni_out_fln
print, ''
message, 'SALAMMBO output file:', /info
print,output_file
print, ''

write_namelist:
message, 'Writing Salammbo namelist file', /info
print, ''

;full_fill - 0: fill spacecraft down to loss cone, 1: fill whole L
;nqs - # steps in code (10 sec each) for output

year = string(r.year, format = "(i4.4)")
start_decimal_year = r.year+(tai_1 - st_sec) / year_sec 
duration = tai_2-tai_1

;set initial state for run. if it does not exisit, use default.
init_file = 'E25_Init_24sep91'
IF control.init EQ 1 THEN BEGIN
    res = findfile(out_dir+fln_init, count = c)
    IF c EQ 1 THEN init_file = res(0)
ENDIF 

CD, code_dir+'resol/', current = keep_path
openw, u, 'Salammbo_EyL.nml', /get_lun

printf, u, " $CONTROL"
printf, u, "  omni_path    = '"+omni_out_fln+"'"
printf, u, "  initial_state = '"+init_file+"'"
FOR i = 0, sat_count-1 DO BEGIN
    out_str = '  sat'+varprt(i+1)+'_path ='
    printf, u, out_str+"'"+input_files(i)+"'"
ENDFOR 
FOR i = sat_count, 19 DO BEGIN
    out_str = '  sat'+varprt(i+1)+'_path ='
    printf, u, out_str+"''"
ENDFOR     
printf, u, "  Nsat = "+varprt(sat_count)
printf, u, "  salammbo_output = '"+out_file+"'"
printf, u, "  salammbo_last = '"+last_file+"'"
printf, u, "  year = "+string(start_decimal_year, format = "(f13.8)")+'D0'
printf, u, "  Tmax = "+string(duration, format = "(f13.2)")+'D0'
printf, u, "  PasT = 10.D0" 
printf, u, "  nqs  = 60" 
printf, u, "  full_fill = " + varprt(control.fill) 
printf, u, " $END" 
close, u & free_lun,u
CD, keep_path

;******************************************************************************
;now run the simulation code!

;see is a previous run exists, and only run if required
res = findfile(out_file, count = c)

IF c EQ 1 THEN BEGIN
    message, 'Have', /info & print, '  ', out_file & print, ''
    GOTO, graph
ENDIF

CD, code_dir+'resol/', current = keep_path 
message, 'Running Salammbo_EyL_lowres...', /info
spawn, 'Salammbo_EyL_lowres'
CD, keep_path

;write a log of the run's output
message, 'Writing log of Salammbo run', /info

openw, u, log_file+'_run.log', /get_lun
FOR i = 0, n_elements(run_arr)-1 DO printf, u, run_arr(i)
close, u & free_lun,u

;******************************************************************************
;now make the required output files for plotting 
;write the graph namelist file needed

graph:

;see is a previous output exists, and only run if required
res = findfile(output_file, count = c)
IF (control.run EQ 2) THEN c = 0

IF c EQ 1 THEN BEGIN
    message, 'Have', /info & print, '  ', out_file
    GOTO, graph
ENDIF

CD, code_dir+'graf/', current = keep_path

;Salammbo Variable Ieny -  1: Diff flux for E1
;                          2: Integral - >E1
;                          3: Diff E1-E2

message, 'Writing Graph namelist file', /info
print, ''
 
openw, u, 'restore_fluxATsat.nml', /get_lun
printf, u, " $CONTROL"
printf, u, "  salammbo_output = '"+out_file+"'"
printf, u, "  restore_fluxATsat_input = '"+input_file+"'"
printf, u, "  restore_fluxATsat_output = '"+output_file+"'"
printf, u, "  Ieny = 2"
printf, u, " $END" 
close, u & free_lun,u
CD, keep_path

;now run 
CD, code_dir+'graf/', current = keep_path

message, 'Running restore_fluxATsat...', /info
spawn, 'restore_fluxATsat_lowres', run_arr
CD, keep_path

;write a log of the run's output
message, 'Writing log of graph run', /info

openw, u, log_file+'_graf.log', /get_lun
FOR i = 0, n_elements(run_arr)-1 DO printf, u, run_arr(i)
close, u & free_lun,u

yscl = old_yscl & zscl = old_zscl

END
;******************************************************************************
