;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  scatha_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  scatha_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON scatha_panel_data, widgetData
  common scatha2, eenergy, penergy, eenergy_names, penergy_names, ephem_names

;--- if GROUP is not specified, the group-leader will be the desktop
  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
  panel_base=papco_WIDGET_BASE(title=aPlotInfo.panelKind, row=1, $
                               GROUP_LEADER=group, /MODAL)
  panel_left=widget_base(panel_base,column=1)
  panel_right=widget_base(panel_base,column=1)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
  widgets=papco_panel_AddHeightWidgets(panel_left, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
  type_base=widget_base(panel_left,ROW=1)
  
;--- buttons to chose species -------------------------------------------------
  species_names=['Electrons','Protons']
  mpb_spec=CW_BGROUP(type_base, species_names, COLUMN=1,/EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Species', $
                     UVALUE='mpb_spec', $
                     SET_VALUE=aPlotInfo.ioptions(0))   
  
;--- buttons to chose direction -----------------------------------------------
  direc_names=['Perpendicular','Parallel']
  mpb_direc=CW_BGROUP(type_base, direc_names, COLUMN=1,/EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Look Dir', $
                     UVALUE='mpb_direc', $
                     SET_VALUE=aPlotInfo.ioptions(2))
  
;--- buttons to chose color labels --------------------------------------------
  color_names=['Auto ','User ']
  mpb_color=CW_BGROUP(type_base, color_names, COLUMN=1,/EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Line Color', $
                     UVALUE='mpb_color', $
                     SET_VALUE=aPlotInfo.ioptions(3))
  
;--- buttons to chose ephem labels --------------------------------------------
  eph_names=['Extra x-label','Line plot']
  mpb_eph=CW_BGROUP(type_base, eph_names, COLUMN=1,/EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Ephemeris', $
                     UVALUE='mpb_eph', $
                     SET_VALUE=aPlotInfo.ioptions(5))  
  
;--- buttons to chose energy ranges -------------------------------------------
  range_names=['All','Low','High','Skip odd','Skip even','None','User']
  mpb_range=CW_BGROUP(panel_left, range_names, ROW=1,/EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Select Range', $
                     UVALUE='mpb_range', $
                     SET_VALUE=aPlotInfo.ioptions(1)) 
  
;--- buttons to chose time resolution -----------------------------------------
  res_names=['1 min','3 min','5 min','10 min','30 min','1 hour','Auto']
  mpb_res=CW_BGROUP(panel_left, res_names, ROW=1,/EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Select Time Resolution', $
                     UVALUE='mpb_res', $
                     SET_VALUE=aPlotInfo.ioptions(4))
  
;--- button to chose variables - not all of them mapped! ----------------------
  vars_outer_base=WIDGET_BASE(panel_left, COLUMN=1, /align_center)
  vars_base=WIDGET_BASE(vars_outer_base, /align_center)  
  
  ;for electrons - use typevector(0) as a logical flag to set energies
  button_names=eenergy_names
  n=n_elements(button_names)  &  button_set=bytarr(n) 
  for i=0,n-1 do $
    if (aPlotInfo.typeVector(0) and 2^i) eq 2^i then button_set(i)=1 $
  else button_set(i)=0
  
  mpb_elec=CW_BGROUP(vars_base, button_names, COLUMN=5,/NONEXCLUSIVE, /FRAME, $
    LABEL_TOP='Select electron energy channels (keV)', $
    UVALUE='mpb_elec', $
    SET_VALUE=button_set)      
    
  ;for protons - use typevector(1) as a logical flag to set energies
  button_names=penergy_names
  n=n_elements(button_names)  &  button_set=bytarr(n) 
  for i=0,n-1 do $
    if (aPlotInfo.typeVector(1) and 2^i) eq 2^i then button_set(i)=1 $
  else button_set(i)=0
  
  mpb_prot=CW_BGROUP(vars_base, button_names, COLUMN=5,/NONEXCLUSIVE, /FRAME, $
    LABEL_TOP='Select proton energy channels (keV)', $
    UVALUE='mpb_prot', $
    SET_VALUE=button_set)
  
  ;for ephemeris line plots
  mpb_ephplot=CW_BGROUP(vars_base, ephem_names, column=5,/EXCLUSIVE, /FRAME, $
                     LABEL_TOP='Select Ephemeris Coordinate', $
                     UVALUE='mpb_ephplot', $
                     SET_VALUE=aPlotInfo.ioptions(6))
  
;--- buttons to chose ephemeris labels ----------------------------------------
  ;for ephemeris labels - use typevector(2) as a logical flag 
  button_names=ephem_names
  n=n_elements(button_names)  &  button_set=bytarr(n) 
  for i=0,n-1 do $
    if (aPlotInfo.typeVector(2) and 2^i) eq 2^i then button_set(i)=1 $
  else button_set(i)=0
  
  mpb_ephlab=CW_BGROUP(panel_left, button_names, ROW=1, /NONEXCLUSIVE, /FRAME,$
    LABEL_TOP='Select Ephemeris X-labels', $
    UVALUE='mpb_ephlab', $
    SET_VALUE=button_set)  
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    scatha. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_left, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
;  widgets=papco_panel_Addcolorbar_Widgets(panel_left, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_left, aPlotInfo, 0)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_panel_AddYScale_Widgets(panel_right, aPlotInfo,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
;  widgets=papco_panel_AddZScale_Widgets(panel_right, aPlotInfo,/VERTICAL)
;  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_right, Action, aPlotInfo, $
                                       /CONFIG, /VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the scatha example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
  thisWidgetsData={helpfile:'scatha_panel.help', $
                   typevector:aPlotInfo.typevector, $
                   ioptions:aPlotInfo.ioptions, $
                   mpb_range:mpb_range, $
                   mpb_elec:mpb_elec, $
                   mpb_prot:mpb_prot, $
                   mpb_ephplot:mpb_ephplot, $
                   mpb_ephlab:mpb_ephlab}
  widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
  scatha_panel_refresh, /ALL
  papco_Panel_Realize, panel_base, widgetData
  papco_xmanager, 'scatha_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
  returnData={typevector:widgetData.typevector, $
              ioptions:widgetData.ioptions}
  additionalRData=papco_panel_ReturnData(widgetData)
    
  return, create_struct(ReturnData, additionalRData)
   
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO scatha_panel_refresh, event
;* 
;* DESCRIPTION:  
;*      This is the widget refresh routine for scatha_panel
;*
;* INPUTS:       
;*      none, just keywords
;*                  
;* KEYWORDS:
;*      
;* 
;******************************************************************************
PRO scatha_panel_refresh, ALL=ALL, SPECIES=SPECIES, RANGE=RANGE, $
                          COLOR=COLOR
  
  COMMON scatha_panel_data, widgetData
  common scatha2, eenergy, penergy, eenergy_names, penergy_names, ephem_names
  
  if keyword_set(ALL) then begin
    SPECIES=1
    RANGE=1
    COLOR=1
  endif
  
  if keyword_set(SPECIES) then begin
    check=widgetData.ioptions(0)
    if widgetData.ioptions(5) eq 1 then begin
      check=2
      widgetData.YLINLOG=0
    endif else widgetData.YLINLOG=1
    papco_panel_Refresh,widgetData,/YSCALE
    case check of
      0: begin
        WIDGET_CONTROL, widgetData.mpb_elec, MAP=1
        WIDGET_CONTROL, widgetData.mpb_prot, MAP=0
        WIDGET_CONTROL, widgetData.mpb_ephplot, MAP=0
      end
      1: begin
        WIDGET_CONTROL, widgetData.mpb_elec, MAP=0
        WIDGET_CONTROL, widgetData.mpb_prot, MAP=1
        WIDGET_CONTROL, widgetData.mpb_ephplot, MAP=0
      end
      2: begin
        WIDGET_CONTROL, widgetData.mpb_elec, MAP=0
        WIDGET_CONTROL, widgetData.mpb_prot, MAP=0
        WIDGET_CONTROL, widgetData.mpb_ephplot, MAP=1
      end
    endcase
  endif  
  
  if keyword_set(RANGE) then begin    
    if widgetData.ioptions(0) then n=n_elements(penergy) else $
                                   n=n_elements(eenergy)
    energy_set=bytarr(n) 

    case widgetData.ioptions(1) of
      0: for i=0,n-1 do energy_set(i)=1     
      1: for i=0,n/2-1 do energy_set(i)=1   
      2: for i=n/2, n-1 do energy_set(i)=1          
      3: for i=0,n-1,2 do energy_set(i)=1  
      4: for i=1,n-1,2 do energy_set(i)=1 
      5: for i=0,n-1 do energy_set(i)=0    
      6: return
    endcase
    
    if widgetData.ioptions(0) eq 0 then begin ; electrons
      WIDGET_CONTROL, widgetData.mpb_elec, SET_VALUE=energy_set 
      widgetData.typevector(0)=0
      for i=0,n-1 do if energy_set(i) then $
        widgetData.typevector(0)=widgetData.typevector(0)+2^i       
    endif else begin                          ; ions
      WIDGET_CONTROL, widgetData.mpb_prot, SET_VALUE=energy_set
      widgetData.typevector(1)=0
      for i=0,n-1 do if energy_set(i) then $
        widgetData.typevector(1)=widgetData.typevector(1)+2^i   
    endelse
    
  endif  
  
  if keyword_set(COLOR) then begin
    if widgetData.ioptions(3) eq 0 then $
      WIDGET_CONTROL, widgetData.ps_base, SENSITIVE=0 $
    else $
      WIDGET_CONTROL, widgetData.ps_base, SENSITIVE=1
  endif  
  
end

;******************************************************************************
;* PROCEDURE:     
;*      PRO scatha_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO scatha_panel_event, event
  
  COMMON scatha_panel_data, widgetData

  WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

  CASE uval OF
    
    'mpb_spec': begin
      widgetData.ioptions(0)=event.value
      scatha_panel_refresh, /SPECIES, /RANGE
    end  
    'mpb_direc': begin
      widgetData.ioptions(2)=event.value
    end  
    'mpb_color': begin
      widgetData.ioptions(3)=event.value
      scatha_panel_refresh, /COLOR    
    end  
    'mpb_eph': begin
      widgetData.ioptions(5)=event.value
      scatha_panel_refresh, /SPECIES
    end  
    'mpb_range': begin
      widgetData.ioptions(1)=event.value
      scatha_panel_refresh, /RANGE
    end  
    'mpb_res': begin
      widgetData.ioptions(4)=event.value
    end  
    'mpb_elec': begin
      WIDGET_CONTROL,widgetData.mpb_elec,GET_VALUE=value
      widgetData.typevector(0)=0
      for i=0,n_elements(value)-1 do if value(i) then $
        widgetData.typevector(0)=widgetData.typevector(0)+2^i     
      widgetData.ioptions(1)=6
      WIDGET_CONTROL,widgetData.mpb_range, SET_VALUE=widgetData.ioptions(1)
    end  
    'mpb_prot': begin
      WIDGET_CONTROL,widgetData.mpb_prot,GET_VALUE=value
      widgetData.typevector(1)=0
      for i=0,n_elements(value)-1 do if value(i) then $
        widgetData.typevector(1)=widgetData.typevector(1)+2^i        
      widgetData.ioptions(1)=6
      WIDGET_CONTROL,widgetData.mpb_range, SET_VALUE=widgetData.ioptions(1)
    end  
    'mpb_ephlab': begin
      WIDGET_CONTROL,widgetData.mpb_ephlab,GET_VALUE=value
      widgetData.typevector(2)=0
      for i=0,n_elements(value)-1 do if value(i) then $
        widgetData.typevector(2)=widgetData.typevector(2)+2^i        
    end
    'mpb_ephplot': begin
      widgetData.ioptions(6)=event.value
    end  
    else: widgetData=papco_panel_Event(event, widgetData)
    
  endcase
    
END
