function load_scatha_fluxes,interval,error=error,fillval=fillval

;+
; NAME:
;      LOAD_SCATHA_FLUXES
; PURPOSE:
;      Load electron and ion fluxes from SCATHA SP CDF files.
; CALLING SEQUENCE:
;      data=load_scatha_fluxes(interval)
; INPUTS:
;      INTERVAL=DBLARR[2]. NSSDC Epoch interval to load.
; OUTPUTS:
;      DATA=STRUCTURE{EPOCH:DBLARR[N],
;                     LON:FLTARR[N],
;                     LAT:FLTARR[N],
;                     RAD:FLTARR[N],
;                     MLT:FLTARR[N],
;                     LSHELL:FLTARR[N],
;                     EENERGY:FLTARR[NE],
;                     EFPERP:FLTARR[NE,N],
;                     EFPAR:FLTARR[NE,N]
;                     PENERGY:FLTARR[NP],
;                     PFPERP:FLTARR[NP,N],
;                     PFPAR:FLTARR[NP,N]}
; KEYWORDS:
;      ERROR. If set on return, no data could be loaded.
;      FILLVAL. Value to use as fillvalue in output for missing
;         data. Default is NaN.
; MODIFICATION HISTORY:
;      Created by Anders M Jorgensen, August 26, 1999.
;-

error=1

if not(keyword_set(fillval)) then fillval=!values.f_nan

n=long((interval[1]-interval[0])/6d4*1.1d)+100l
p=0l
np=(n/10l)+10l
epoch=dblarr(n)
lon=fltarr(n)
lat=fltarr(n)
rad=fltarr(n)
mlt=fltarr(n)
lshell=fltarr(n)
eenergy=fltarr(10)
efperp=fltarr(10,n)
efpar=fltarr(10,n)
pfperp=fltarr(10,n)
pfpar=fltarr(10,n)

path=['/nh/crockpot/tmp/ajorg/scatha_sp.zip']

cdf_epoch,interval[0],yr,mo,dy,/breakdown_epoch
cdf_epoch,start,yr,mo,dy,/compute_epoch
for i=start,interval[1]-1d,86.4d6 do begin
    cdf_epoch,i,yr,mo,dy,/breakdown_epoch
    cdf_epoch,yrepoch,yr,1,1,/compute_epoch
    yr=strtrim(yr-1900,2)
    doy=strtrim(round((i-yrepoch)/86.4d6+1d),2)
    while strlen(doy) lt 3 do doy='0'+doy
    file='ssp'+yr+doy+'v...cdf'
    locate_file,path,file,ofile
    if ofile ne '' then begin
        id=cdf_open(ofile)
        cdf_control,id,variable='Epoch',get_var_info=info
        cdf_varget,id,'Epoch',tepoch,rec_count=info.maxrec+1,count=[1,1,1,1]
        index=where((tepoch ge interval[0]) and (tepoch lt interval[1]))
        if index[0] ne -1 then begin
            srec=index[0]
            nrec=n_elements(index)
            cdf_varget,id,'SCGeoLon',tlon,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where((tlon lt 0.) or (tlon gt 360.))
            if ind[0] ne -1 then tlon[ind]=fillval
            cdf_varget,id,'SCGeoLat',tlat,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where((tlat lt -90.) or (tlat gt 90.))
            if ind[0] ne -1 then tlat[ind]=fillval
            cdf_varget,id,'SCGeoRad',trad,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where(trad lt 0.)
            if ind[0] ne -1 then trad[ind]=fillval
            cdf_varget,id,'MLT',tmlt,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where((tmlt lt 0.) or (tmlt gt 24.))
            if ind[0] ne -1 then tmlt[ind]=fillval
            cdf_varget,id,'LShell',tlshell,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where(lshell lt 0.)
            if ind[0] ne -1 then tlshell[ind]=fillval
            if i eq start then begin
                cdf_varget,id,'EDfFlxEn',eenergy,rec_count=1,count=[1,1,1,10]
                cdf_varget,id,'PDfFlxEn',penergy,rec_count=1,count=[1,1,1,10]
            endif
            cdf_varget,id,'EPpDfFlx',tefperp,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tefperp lt 0.)
            if ind[0] ne -1 then tefperp[ind]=fillval
            cdf_varget,id,'EPlDfFlx',tefpar,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tefpar lt 0.)
            if ind[0] ne -1 then tefpar[ind]=fillval
            cdf_varget,id,'PPpDfFlx',tpfperp,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tpfperp lt 0.)
            if ind[0] ne -1 then tpfperp[ind]=fillval
            cdf_varget,id,'PPlDfFlx',tpfpar,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tpfpar lt 0.)
            if ind[0] ne -1 then tpfpar[ind]=fillval
            error=0
            while (p+nrec) gt n do begin
                message,'Extending arrays from '+strtrim(n)+' to '+$
                  strtrim(n+np)+'.',/informational
                epoch=[epoch,dblarr(np)]
                lon=[lon,fltarr(np)]
                lat=[lat,fltarr(np)]
                rad=[rad,fltarr(np)]
                mlt=[mlt,fltarr(np)]
                lshell=[lshell,fltarr(np)]
                efperp=[[efperp],[fltarr(10,np)]]
                efpar=[[efpar],[fltarr(10,np)]]
                pfperp=[[pfperp],[fltarr(10,np)]]
                pfpar=[[pfpar],[fltarr(10,np)]]
                n=n+np
            endwhile
            q=p+nrec-1
            epoch[p:q]=reform(tepoch[index])
            lon[p:q]=reform(tlon)
            lat[p:q]=reform(tlat)
            rad[p:q]=reform(trad)
            mlt[p:q]=reform(tmlt)
            lshell[p:q]=reform(tlshell)
            efperp[*,p:q]=reform(tefperp)
            efpar[*,p:q]=reform(tefpar)
            pfperp[*,p:q]=reform(tpfperp)
            pfpar[*,p:q]=reform(tpfpar)
            p=q+1
        endif
        cdf_close,id
    endif
endfor

if error eq 0 then begin
    p=p-1
    return,{epoch:epoch[0:p],lon:lon[0:p],lat:lat[0:p],rad:rad[0:p],$
            mlt:mlt[0:p],lshell:lshell[0:p],$
            eenergy:reform(eenergy),efperp:efperp[*,0:p],efpar:efpar[*,0:p],$
            penergy:reform(penergy),pfperp:pfperp[*,0:p],pfpar:pfpar[*,0:p]}
endif else return,0

end



