PRO Locate_file,ipath,find,found,delay=delay,scratch=scratch,order=order,$
                open=open,alast=alast,afirst=afirst,all=all,unit=unit,$
                quiet=quiet
   
;+
; NAME:
;      LOCATE_FILE
; PURPOSE:
;      Find files and prepare them for reading by the calling
;      program. 
; PROCEDURE:
;      Given a search path and a regular expression (as accepted by
;      "grep"), find files in the path that match the expression.
;      Path may be a list of directories and .zip files. In the path
;      search in a specified order (typically directories first, then
;      zip files) for files that are uncompressed, gziped, or
;      compressed. For files that are not uncompressed or are in a zip
;      file, extract them to a temporary file. Return the name of the
;      file(s) found.
; CALLING SEQUENCE:
;      locate_file,path,file,found
; INPUTS:
;      PATH = STRARR(N) A list of paths to search for the file. 
;      FILE = STRING. A wildcard expression for the file, that will
;         be accepted by the FINDFILE function.
; OUTPUTS:
;      FOUND = STRING/STRARR(M). The name(s) of the located file(s).
; KEYWORDS:
;      DELAY = INTEGER. The time that a scratch file is kept before it 
;         is erased. Default is 15 seconds. Note, that unix is smart 
;         enough not to erase files from disk, which are open. The
;         delay should not reflect how long it takes to read the file, 
;         but how long it takes to open the file. Under normal 
;         circumstances it only takes of the order of at most tenth's 
;         of seconds, so this parameter would not normally need
;         to be changed.
;      SCRATCH = STRING. The directory to use as scratch area. Default is 
;         '/tmp/'
;      ORDER = INTARR. The order in which the search should be
;         done. Default is "uncompressed" (0), "gziped" (1), 
;         "compressed" (2), "uncompressed in zip archive" (3), 
;         "gziped in zip archive" (4), "compressed in zip archive"
;          (5). More could be defined (like extract from tar archive).
;      AFIRST. If set, return only the alphabetically first file
;         found.
;      ALAST. If set, return only the alphabetically last file
;         found.
;      ALL. If set, return all files found. (Default).
;      OPEN. If set, open each file found and return the unit number.
;      UNIT = INTARR(M). List of unit numbers
;      QUIET=QUIET. If set, don't print status messages.
; EXAMPLE:
;      Example 1:
;      path=['/n/fedtmule/home/projects/crres/data',
;            '/nh/teapot/tempest/ajorg/data/crres.zip']
;      find='eph034[1-2]\*.dat'
;      locate_file,path,find,found,/open,unit=un
;      print,found
;       /n/fedtmule/home/projects/crres/data/eph0341.dat
;       /n/fedtmule/home/projects/crres/data/eph0342.dat
;      print,un
;            110     111
;      Example 2:
;      path=['/nh/teapot/tempest/ajorg/data/crres.zip']
;      locate_file,path,find,found,/open,unit=un
;      print,found
;       /tmp/locate_file.XXa12422 /tmp/locate_file.XXa12425
;      print,un
;            112     113
; MODIFICATION HISTORY:
;      Created by Anders M Jorgensen, February 16, 1997.
;      Modified to work remotely when local hostname does not contain
;      ceppad, enterprise, or buspace. AMJ, May 8, 1997.
;      Temporarily turned off the "remote" option until I can figure
;      out what to do about it here at LANL. AMJ, July 17, 1998.
;      Added capability to handle files compressed with compress '.Z',
;      AMJ, July 17, 1998.
;      Total rewrite, AMJ, November 4, 1998.
;-

print,find
; procedure: find out where files are
; if files need to be uncompressed/extracted, do that
; list name(s) of found file(s)
; set up deletion of temporary files if needed/desired.

dirsep=':' & path=''
for j=0,n_elements(ipath)-1 do begin
    path=[path,str_sep(ipath(j),dirsep)]
endfor
path=path(1:*)
print,path
if not(keyword_set(delay)) then delay=15
if not(keyword_set(scratch)) then scratch='/tmp/'
if not(keyword_set(order)) then order=[0,1,2,3,4,5]

dir=[1,1,1,0,0,0,0,0,0] & zip=[0,0,0,1,1,1,0,0,0] & tar=[0,0,0,0,0,0,1,1,1]
uncomp=[1,0,0,1,0,0,1,0,0] & gzip=[0,1,0,0,1,0,0,1,0]
comp=[0,0,1,0,0,1,0,0,1]

uncompext='' & gzipext='.gz' & compext='.Z'

zipext='.zip'
zidxext='.index'
suffixes=[uncompext,gzipext,compext]

zipindex='unzip -l' ; command for getting index of a zip file
;zipihskip=1 ; number of header lines to discard in zipindex output
;zipifskip=2 ; number of footer lines to discard in zipindex output
zipicolumn=3 ; which space separated column (0 first) the filenames are in

zipextract='unzip -p ' ; command for extracting a file from zip to pipe
gzcat='gzcat' ; command for gunzip stdin to sdout
zcat='zcat' ; comand for uncompressing stdin to stdout
sleep='sleep' ; command for waiting a number of seconds
grep='grep' ; look for regular expression in stdin to stdout
mktemp='mktemp' ; create temporary file
scratchfile=scratch+'/locate_file.XXXXXXXX'
rm='rm -f' ; command to remove file
ls='\ls -1' ; command to list files on per line

; First find the location of matching files, searching in the 
; specified order of compression and archiving:

i=0 & found=0

while not(found) and i lt n_elements(order) do begin
    if not(keyword_set(quiet)) then $
      print,'Looking for: '+$
      'dir:'+strtrim(dir(order(i)),2)+' zip:'+strtrim(zip(order(i)),2)+$
      ' tar:'+strtrim(tar(order(i)),2)+' uncomp:'+strtrim(uncomp(order(i)),2)+$
      ' gzip:'+strtrim(gzip(order(i)),2)+' comp:'+strtrim(comp(order(i)),2)
    if uncomp(order(i)) then ext=uncompext else $
      if gzip(order(i)) then ext=gzipext else $
      if comp(order(i)) then ext=compext
    if dir(order(i)) then begin ; If searching directories
        match=''
        for j=0,n_elements(path)-1 do begin
            tmp=strmid(path(j),strlen(path(j))-strlen(zipext),strlen(zipext))
            if tmp ne zipext then begin
                command=ls+" "+path(j)+" | "+grep+" '"+find+ext+"$'"
                spawn,command,tmp
                for k=0,n_elements(tmp)-1 do if tmp(k) ne '' then $
                  tmp(k)=path(j)+'/'+tmp(k)
                match=[match,tmp]
            endif
        endfor
        index=where(match ne '')
        if index(0) ne -1 then begin
            found=1 & match=match(index)
        endif
    endif else if zip(order(i)) then begin ; If searching zip files
        zipfiles=''
        for j=0,n_elements(path)-1 do begin
            if strmid(path(j),strlen(path(j))-strlen(zipext),strlen(zipext)) $
              eq zipext then zipfiles=[zipfiles,findfile(path(j))]
        endfor
        index=where(zipfiles ne '')
        if index(0) ne -1 then begin
            zipfiles=zipfiles(index)
            match='' & zipfile='' ; now find all matching files in ZIPFILES
            for j=0,n_elements(zipfiles)-1 do begin
                idxfile=(findfile(strmid(zipfiles(j),0,strlen(zipfiles(j))-$
                                         strlen(zipext))+zidxext))(0)
                if idxfile ne '' then begin
                    spawn,grep+' '+find+ext+'$ '+idxfile,tmp
                    match=[match,tmp]
                    zipfile=[zipfile,replicate(zipfiles(j),n_elements(tmp))]
                endif else begin
                    command=zipindex+' '+zipfiles(j)+' | '+grep+' '+find+ext+$
                      '$'
                    spawn,command,tmp
                    if tmp(0) ne '' then begin
                        tmp=strtrim(strcompress(tmp),2)
                        for k=0,n_elements(tmp)-1 do begin
                            match=[match,(str_sep(tmp(k),' '))(zipicolumn)]
                            zipfile=[zipfile,zipfiles(j)]
                        endfor
                    endif
                endelse
            endfor
            index=where(match ne '')
            if index(0) ne -1 then begin
                found=1 & match=match(index) & zipfile=zipfile(index)
            endif
        endif
    endif
    i=i+1
endwhile
i=i-1

if match(0) eq '' then begin
    found='' & return
endif

; Here is where we find the files among the matches that we want to 
; extract (Alphabetically first, last, all). Need to remove the
; extensions before sorting.

if keyword_set(afirst) or keyword_set(alast) then begin
    files=strarr(n_elements(match))
    for j=0,n_elements(match)-1 do begin
        tmp=str_sep(match(j),'/')
        tmp=tmp(n_elements(tmp)-1)
        files(j)=strmid(tmp,0,strlen(tmp)-strlen(ext))
    endfor
    index=sort(files)
    if keyword_set(afirst) then index=index(0)
    if keyword_set(alast) then index=index(n_elements(index)-1)
    match=match(index)
    if zip(order(i)) then zipfile=zipfile(index)
endif

; Here is where we extract the files and create a list of actual
; temporary file names, and filenames as stored (without paths and
; extensions)

found=strarr(n_elements(match))
for j=0,n_elements(match)-1 do begin
    if zip(order(i)) then begin
        command=zipextract+' '+zipfile(j)+' '+match(j)
        if gzip(order(i)) then command=command+' | '+gzcat
        if comp(order(i)) then command=command+' | '+zcat
        spawn,mktemp+' '+scratchfile,tmpfile
        command=command+' > '+tmpfile
        spawn,command
        found(j)=tmpfile
      endif else if gzip(order(i)) then begin
         spawn,mktemp+' '+scratchfile,tmpfile
        spawn,gzcat+' '+match(j)+' > '+tmpfile
        found(j)=tmpfile
    endif else if comp(order(i)) then begin
        spawn,mktemp+' '+scratchfile,tmpfile
        spawn,zcat+' '+match(j)+' > '+tmpfile
        found(j)=tmpfile
    endif else if uncomp(order(i)) then begin
        found(j)=match(j)
    endif
endfor
if n_elements(found) eq 1 then found=found(0) ; if only one file, make it string, rather than strarr.

; If file(s) were to be opened and unit numbers returned, we do that
; here

if keyword_set(open) then begin
    unit=intarr(n_elements(found))
    for j=0,n_elements(found)-1 do begin
        openr,un,found(j),/get_lun
        unit(j)=un
    endfor
    if n_elements(unit) eq 1 then unit=unit(0)
endif

    
if not(keyword_set(quiet)) then $
  for j=0,n_elements(found)-1 do begin
    tmp=find+': '
    if zip(order(i)) then tmp=tmp+zipfile(j)+':'
    tmp=tmp+match(j)
    if not(uncomp(order(i))) or zip(order(i)) then tmp=tmp+' -> '+found(j)
    if keyword_set(open) then tmp=tmp+' ('+strtrim(unit(j),2)+')'
    print,tmp
endfor

; If temporary files are to be deleted after a certain amount of time, 
; we do that here.
if delay gt 0 then begin
    for j=0,n_elements(found)-1 do begin
        if found(j) ne match(j) then begin
            command='( '+sleep+' '+string(fix(delay))+' ; '+rm+' '+$
              found(j)+' ) &'
            spawn,command
        endif
    endfor
endif

; Return found files/units

end
