;******************************************************************************
;*  NAME:        r_scatha
;*
;*  DESCRIPTION: Routine to read scatha files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        November 1999
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_scatha, VERBOSE=VERBOSE

  common scatha_data, input_header, input_data
  common mjdt, mjdt_start, mjdt_end
  common mjdt_day, time
  common get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

  get_err_no=0  &  get_err_msg='scatha read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('SCATHA_DATA', path=PATH)) then return
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. PAPCO standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
  time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
  while mjdt_end.mjd ge time.mjd do begin
     
    ;assemble the data file from the time range requested
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt   
    datestr=string(year,month,day_om,format='(i4.4,i2.2,i2.2)')
    fln='scatha_'+datestr
     
    ;read one day of data
    r_scatha_onefile, path+fln, VERBOSE=VERBOSE
  
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin      
      c_url=papco_getenv('SCATHA_FTP_ARCHIVE')
      c_file=fln+'.idl'  &  c_download_dir=path
      papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      if get_err_no eq 0 then begin
        r_scatha_onefile, path+fln, VERBOSE=VERBOSE
      endif else begin
        get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
        return
      endelse  
    endif   
     
    ;if a file was succesfully read, concatonate!
    ;scatha time is a continuous double of type cfd_epoch.
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse        
      input_data=0                      ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    day_count=day_count+1
    time.mjd=time.mjd+1                 ; next day
  endwhile         
   
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
  endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    return
  endelse      
   
; restrict data to actual time range requested. 
; scatha time is a continuous double of type cfd_epoch.
  convert_t90_to_date, mjdt_start , year, day_oy, hour, minute, second,$
                       month, cmonth, day_om, /mjdt     
  cdf_epoch,ut1, year, month, day_om, hour, minute, second, /COMPUTE_EPOCH 
  convert_t90_to_date, mjdt_end, year, day_oy, hour, minute, second,$
                       month, cmonth, day_om, /mjdt     
  cdf_epoch,ut2, year, month, day_om, hour, minute, second, /COMPUTE_EPOCH  

  index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
  if count ne 0 then input_data=input_data(index)
    
end

;******************************************************************************
;*  NAME:        r_scatha_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_scatha_onefile, fln, VERBOSE=VERBOSE
  
  common scatha_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0  &  get_err_msg='scatha onefile read succesfully'
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
  on_ioerror, read_original
  restore,fln+'.idl', VERBOSE=VERBOSE
  dealloc_lun,/quiet   ;frees logical units used by save/restore
  message,'Saveset: '+fln+'.idl', /cont
  return

  read_original:       ;if the .idl file is not found, try to read the original
  r_scatha_onefile_original, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_scatha_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_scatha_onefile_original, fln, VERBOSE=VERBOSE
  
  common scatha_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
  common mjdt_day, time  
   
  get_err_no=0  &  get_err_msg='scatha_onefile_orignal read succesfully'
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
; construct call to Anders Jorgensen's read routine  
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt    
    
    cdf_epoch,b,year,month,day_om, /COMPUTE_EPOCH
    
    data=load_scatha_fluxes([b,b+86.4d6], error=get_err_no)
    
    if get_err_no eq 1 then goto, out
    
; define the header and data structures. Edit to suit your data!
  input_header={scatha_header1,eenergy:data.eenergy,penergy:data.penergy}
  dummy={scatha_data2, time:0.0d, lon:0., lat:0., rad:0., mlt:0., lshell:0.,$
         efperp:fltarr(10),efpar:fltarr(10),pfperp:fltarr(10),pfpar:fltarr(10)}
  max_no_rec=n_elements(data.epoch)
  input_data=replicate(dummy, max_no_rec)
  
  for i=0,max_no_rec-1 do begin
    input_data[i].time=data.epoch[i]
    input_data[i].lon=data.lon[i]
    input_data[i].lat=data.lat[i]
    input_data[i].rad=data.rad[i]
    input_data[i].mlt=data.mlt[i]
    input_data[i].lshell=data.lshell[i]
    input_data[i].efperp=data.efperp[*,i]
    input_data[i].efpar=data.efpar[*,i]
    input_data[i].pfperp=data.pfperp[*,i]
    input_data[i].pfpar=data.pfpar[*,i]
  endfor
    
; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
  message,'Saving as IDL saveset: '+fln+'.idl',/cont
  save,filename=fln+'.idl',input_header,input_data, VERBOSE=VERBOSE
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. 
  get_err_msg='!C '+'cdf file read error'
   
end

;******************************************************************************
;scatha read routine from Anders

function load_scatha_fluxes,interval,error=error,fillval=fillval

;******************************************************************************
;+
; NAME:
;      LOAD_SCATHA_FLUXES
; PURPOSE:
;      Load electron and ion fluxes from SCATHA SP CDF files.
; CALLING SEQUENCE:
;      data=load_scatha_fluxes(interval)
; INPUTS:
;      INTERVAL=DBLARR[2]. NSSDC Epoch interval to load.
; OUTPUTS:
;      DATA=STRUCTURE{EPOCH:DBLARR[N],
;                     LON:FLTARR[N],
;                     LAT:FLTARR[N],
;                     RAD:FLTARR[N],
;                     MLT:FLTARR[N],
;                     LSHELL:FLTARR[N],
;                     EENERGY:FLTARR[NE],
;                     EFPERP:FLTARR[NE,N],
;                     EFPAR:FLTARR[NE,N]
;                     PENERGY:FLTARR[NP],
;                     PFPERP:FLTARR[NP,N],
;                     PFPAR:FLTARR[NP,N]}
; KEYWORDS:
;      ERROR. If set on return, no data could be loaded.
;      FILLVAL. Value to use as fillvalue in output for missing
;         data. Default is NaN.
; MODIFICATION HISTORY:
;      Created by Anders M Jorgensen, August 26, 1999.
;-
  
  
; check env. variable pointing to data directory
  if not (papco_check_data_env('SCATHA_DATA_ARCHIVE', path=PATH)) then begin
    error=1
    return,-1
  endif
  
error=1

if not(keyword_set(fillval)) then fillval=!values.f_nan

n=long((interval[1]-interval[0])/6d4*1.1d)+100l
p=0l
np=(n/10l)+10l
epoch=dblarr(n)
lon=fltarr(n)
lat=fltarr(n)
rad=fltarr(n)
mlt=fltarr(n)
lshell=fltarr(n)
eenergy=fltarr(10)
efperp=fltarr(10,n)
efpar=fltarr(10,n)
pfperp=fltarr(10,n)
pfpar=fltarr(10,n)

path=[path+'scatha_sp.zip']

cdf_epoch,interval[0],yr,mo,dy,/breakdown_epoch
cdf_epoch,start,yr,mo,dy,/compute_epoch
for i=start,interval[1]-1d,86.4d6 do begin
    cdf_epoch,i,yr,mo,dy,/breakdown_epoch
    cdf_epoch,yrepoch,yr,1,1,/compute_epoch
    yr=strtrim(yr-1900,2)
    doy=strtrim(round((i-yrepoch)/86.4d6+1d),2)
    while strlen(doy) lt 3 do doy='0'+doy
    file='ssp'+yr+doy+'v...cdf'
    locate_file,path,file,ofile
    
    if ofile ne '' then begin
        id=cdf_open(ofile)
        cdf_control,id,variable='Epoch',get_var_info=info
        cdf_varget,id,'Epoch',tepoch,rec_count=info.maxrec+1,count=[1,1,1,1]
        index=where((tepoch ge interval[0]) and (tepoch lt interval[1]))
        if index[0] ne -1 then begin
            srec=index[0]
            nrec=n_elements(index)
            cdf_varget,id,'SCGeoLon',tlon,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where((tlon lt 0.) or (tlon gt 360.))
            if ind[0] ne -1 then tlon[ind]=fillval
            cdf_varget,id,'SCGeoLat',tlat,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where((tlat lt -90.) or (tlat gt 90.))
            if ind[0] ne -1 then tlat[ind]=fillval
            cdf_varget,id,'SCGeoRad',trad,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where(trad lt 0.)
            if ind[0] ne -1 then trad[ind]=fillval
            cdf_varget,id,'MLT',tmlt,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where((tmlt lt 0.) or (tmlt gt 24.))
            if ind[0] ne -1 then tmlt[ind]=fillval
            cdf_varget,id,'LShell',tlshell,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,1]
            ind=where(lshell lt 0.)
            if ind[0] ne -1 then tlshell[ind]=fillval
            if i eq start then begin
                cdf_varget,id,'EDfFlxEn',eenergy,rec_count=1,count=[1,1,1,10]
                cdf_varget,id,'PDfFlxEn',penergy,rec_count=1,count=[1,1,1,10]
            endif
            cdf_varget,id,'EPpDfFlx',tefperp,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tefperp lt 0.)
            if ind[0] ne -1 then tefperp[ind]=fillval
            cdf_varget,id,'EPlDfFlx',tefpar,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tefpar lt 0.)
            if ind[0] ne -1 then tefpar[ind]=fillval
            cdf_varget,id,'PPpDfFlx',tpfperp,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tpfperp lt 0.)
            if ind[0] ne -1 then tpfperp[ind]=fillval
            cdf_varget,id,'PPlDfFlx',tpfpar,rec_start=srec,rec_count=nrec,$
              count=[1,1,1,10]
            ind=where(tpfpar lt 0.)
            if ind[0] ne -1 then tpfpar[ind]=fillval
            error=0
            while (p+nrec) gt n do begin
                message,'Extending arrays from '+strtrim(n)+' to '+$
                  strtrim(n+np)+'.',/informational
                epoch=[epoch,dblarr(np)]
                lon=[lon,fltarr(np)]
                lat=[lat,fltarr(np)]
                rad=[rad,fltarr(np)]
                mlt=[mlt,fltarr(np)]
                lshell=[lshell,fltarr(np)]
                efperp=[[efperp],[fltarr(10,np)]]
                efpar=[[efpar],[fltarr(10,np)]]
                pfperp=[[pfperp],[fltarr(10,np)]]
                pfpar=[[pfpar],[fltarr(10,np)]]
                n=n+np
            endwhile
            q=p+nrec-1
            epoch[p:q]=reform(tepoch[index])
            lon[p:q]=reform(tlon)
            lat[p:q]=reform(tlat)
            rad[p:q]=reform(trad)
            mlt[p:q]=reform(tmlt)
            lshell[p:q]=reform(tlshell)
            efperp[*,p:q]=reform(tefperp)
            efpar[*,p:q]=reform(tefpar)
            pfperp[*,p:q]=reform(tpfperp)
            pfpar[*,p:q]=reform(tpfpar)
            p=q+1
        endif
        cdf_close,id
    endif
endfor

if error eq 0 then begin
    p=p-1
    return,{epoch:epoch[0:p],lon:lon[0:p],lat:lat[0:p],rad:rad[0:p],$
            mlt:mlt[0:p],lshell:lshell[0:p],$
            eenergy:reform(eenergy),efperp:efperp[*,0:p],efpar:efpar[*,0:p],$
            penergy:reform(penergy),pfperp:pfperp[*,0:p],pfpar:pfpar[*,0:p]}
endif else return,0

end



