function ct_time_from_string, datetime
;  datetime = [ date time | time date | date | time ]
;  time = [ hour:minute:second | hour:minute ] 
;  date = [ yearnnnn | month/day/year | month.day.year ]
;  hour = [ n | nn ]
;  minute = [ n | nn ] 
;  second = [ n | n.n+ ]
;  year = [ nnnn | nn ]   ; nn -> 19nn
;  month = [ nn | n ]
;  day = [ nn | n ]
;  n = [ 1|2|3|4|5|6|7|8|9|0 ]

  result= dblarr( n_elements( datetime ) )
  for i=0,n_elements(datetime)-1 do begin
      parse= str_sep( datetime(i), ' ' )
      r= where( parse ne '' )
      parse= parse(r)
      hour='' & minute='' & second='' & year='' & month='' & day=''
      for j=0,n_elements(parse)-1 do begin
          this= parse(j)
          istime= strpos( this, ':' ) ne -1
          if istime then begin
              parse1= str_sep( this,':')
              hour= parse1(0)
              minute= parse1(1)
              if n_elements(parse1) gt 2 then second= parse1(2)
          endif else begin
              if strpos( this,'/' ) ne -1 then begin
                  parse1= str_sep( this,'/' )
                  year= parse1(2)
                  month= parse1(0)
                  day= parse1(1)
              endif else if strpos( this,'.' ) ne -1 then begin
                  parse1= str_sep( this,'.' )
                  year= parse1(2)
                  month= parse1(1)
                  day= parse1(0)
              endif else if strlen( this ) eq 8 then begin
                  year= strmid(this,0,4)
                  month= strmid(this,4,2)
                  day= strmid(this,6,2)
              endif else if strlen( this ) eq 6 then begin
                  year= strmid(this,0,2)
                  month= strmid(this,2,2)
                  day= strmid(this,4,2)
              endif
          endelse
      endfor
      if year eq '' then iyear=1972 else begin
          iyear= long( year )
          if iyear lt 90 then iyear=iyear+2000 else $
            if iyear lt 100 then iyear=iyear+1900
      endelse
      if month eq '' then imonth=1 else imonth=fix(month)
      if imonth gt 12 then $
        message, 'warning: date format month gt 12.', /cont
      if day eq '' then iday=17 else iday=fix(day)
      if iday gt 31 then $
        message, 'warning: date format day gt 31.', /cont
      if hour eq '' then ihour=0 else ihour=fix(hour)
      if ihour gt 24 then $
        message, 'warning: time format hour gt 24.', /cont
      if minute eq '' then iminute=0 else iminute=fix(minute)      
      if second eq '' then isecond=0. else isecond=double(second)
      result(i)= double( julday( imonth, iday, iyear ) - 2441334 ) + $
        ihour/24. + iminute/1440. + isecond/86400.
  endfor   
  return, result
end

function ct_time, datetime, $
                  date=date_in, $
                  time=time_in, $
                  hours= units_hour, $
                  seconds=units_second, $
                  minutes=units_minute, $
                  days=units_day, $
                  dec=dec, hms=hms, $
                  julian=julian, $
                  t90=t90, $
                  mjdt=mjdt

;+
; NAME: ct_time
;
; PURPOSE: converts times to ct_time standard 
;
; CATEGORY: utility
;
; CALLING SEQUENCE: 
;   result= ct_time( [date_time], [date=<date>], [time=<time>] )
; 
; OPTIONAL INPUTS:
;   date_time (optional, use instead of date and time keywords): 
;      an array or scalar of times in a format that contains both time
;      and date information, like t90 (papco internal) or a string in
;      the form '19960320 00:00', '3/20/96 00:00:00', '20.3.1996 00:00:00.0',
;      etc.  Two-digit years are assumed to be 19xx years if gt 90,
;      otherwise they are assumed to be 20xx.
;	
; KEYWORD PARAMETERS:
;   date (optional):  an array or scalar of dates in the form yyyymmdd
;      (a long), '3/20/96', etc.
;   time (optional):  an array or scalar of times in the form
;      'hh:mm:ss', hhmmss, or a number.  Additional keywords specify
;      the format of number: /units_hour, /units_second, etc.
;   /hours, /minutes, /seconds     units of input time
;   /hms                           time is in HHMMSS (eg. 0330 -> 3.5 )
;   /dec                           time is in decimal time. ( 3.5 -> 3:30 )
;   /julian                        return julian day instead of doy96.
;   /t90                           date_time argument is t90 from PaPCo
;   /mjdt                          date_time is in papco's mjdt
;
; OUTPUTS:
;      The time in ct_time format.  ct_time format is the decimal number of
;   days since midnight 1/17/1996.  If time is not specified, then
;   midnight is assumed.  If the day is not specified, then the result 
;   will be between 0 and 1.
;
; OPTIONAL OUTPUTS: none.
; COMMON BLOCKS: none.
; SIDE EFFECTS: none.
; RESTRICTIONS: 
;   note that ct_time is defined to be negative before midnight 1/17/1972
;
; EXAMPLE: 
;    r= ct_time( '19960529 00:00' )
;    print, ct_time_to( /string, r )
;
; MODIFICATION HISTORY: 
;    written, jbf, cottagesystems, Feb 14, 2001  
;-

  is_array= $
    n_elements( size(date_in) ) gt 3 or $
    n_elements( size(time_in) ) gt 3 or $
    n_elements( size(datetime) ) gt 3
  
;  format the dates
  if n_elements( date_in ) eq 0 then begin
      ip_day= 0
  endif else begin
      date= date_in
      
      date= strtrim( date,2 )
      only_digits= max( byte( date ), min=min ) le (byte('9'))(0) and $
        min ge (byte('0'))(0)
      
      if only_digits then begin
          if strlen( date(0) ) eq 6 then date= '  '+date
          year= fix( strmid( date, 0, 4 ) )
          month= fix( strmid( date, 4, 2 ) )
          day= fix( strmid( date, 6, 2 ) )
      endif else begin
          n= n_elements( date )
          year= intarr(n)
          month= intarr(n)
          day= intarr(n)
          for i=0,n_elements(date)-1 do begin
              if strpos( date(i), '/' ) ge 0 then begin
                  s= str_sep(date(i),'/')
              endif else if strpos( date(i), '.' ) ge 0 then begin ; .=European
                  s= str_sep(date(i),'.')
                  s= s([1,0,2])
              endif else begin
                  message, 'Date format not recognized.', /cont
                  return, 0
              endelse
              year(i)= fix( s(2) )
              month(i)= fix( s(0) )
              day(i)= fix( s(1) )
          endfor
      endelse
      
      r= where( year ge 90 and year lt 100 )
      if r(0) ne -1 then year(r)=year(r)+1900
      r= where( year lt 90 )
      if r(0) ne -1 then year(r)=year(r)+2000
      
      ct_time= lonarr( n_elements( month ) )
      
      if keyword_set(julian) then begin
          for i=0,n_elements(year)-1 do $
            ct_time[i]= julday( month[i], day[i], year[i] ) 
      endif else begin
          for i=0,n_elements(year)-1 do $
            ct_time[i]= julday( month[i], day[i], year[i] ) - 2441334
      endelse
      
      ip_day= ct_time
  endelse
  
;  format the times
  if n_elements( time_in ) eq 0 then begin
      fp_day=double(0.)
  endif else begin
      time= time_in
      if keyword_set( hms ) then begin
          time= strtrim( time, 2 )
          
          n= strlen( time )
          
          r= where( n mod 2 ) 
          if r(0) ne -1 then time(r)='0'+time(r)
          
          r= where( n eq 4 ) 
          if r(0) ne -1 then time(r)= time(r)+'00'
          
          fp_day= double( strmid( time, 0, 2 ) ) / 24. + $
            double( strmid( time, 2, 2 ) ) / 1440. + $
            double( strmid( time, 4, 2 ) ) / 86400.
          
      endif else begin
          s= size( time )
          if s(n_elements(s)-2) eq 7 then begin ; it's a string assume HH:MM:SS
              n= n_elements( time )
              fp_day= dblarr( n )
              for i=0,n-1 do begin
                  s= str_sep( time[i], ':' )
                  fp_day[i]= double(s(0))/24.+double(s[1])/1440.
                  if n_elements(s) eq 3 then $
                    fp_day[i]=fp_day[i]+double(s[2])/86400.
              endfor
          endif else if keyword_set( units_hour ) then begin
              fp_day= time / 24.D
          endif else if keyword_set( units_minute ) then begin
              fp_day= time / 1400.D
          endif else if keyword_set( units_second ) then begin
              fp_day= time / 86400.D
          endif else if keyword_set( units_day ) then begin
              fp_day= time
          endif else begin
              message, 'Time units not specified', /cont
              stop
          endelse
      endelse
  endelse
  
  if n_elements( datetime ) eq 0 then begin
      ct_time= ip_day + fp_day  
  endif else begin
      s= size( datetime )
      if s(n_elements(s)-2) eq 7 then begin ; it's a string, parse it.
          ct_time= ct_time_from_string( datetime )
      endif
      if keyword_set( t90 ) then begin
          ct_time= ( datetime + 566697600 ) / double(86400.)
      endif else if keyword_set( mjdt ) then begin
          ct_time= datetime.mjd + 2400001 - 2441334 + datetime.T / 86400.D
      endif
  endelse
  
  if is_array then begin
      return, ct_time
  endif else begin
      return, ct_time[0]
  endelse
  
end

