;+
; NAME: ct_web_open
;
; PURPOSE: "open" a website as if it were a file
;
; CATEGORY: utility
;
; CALLING SEQUENCE: ct_web_open, unit, url
; 
; INPUTS:
;   unit  the unit to associate with the stream.
;   url   the name of the URL to open
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;   /get_lun   get logical unit and set unit.
;   header=    return header infomation.  Not filled if wget is used.
;
; OUTPUTS:
;   unit  the unit number used if /get_lun is set.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;   unit parameter is set if /get_lun is set.
;
; RESTRICTIONS:
;   Early versions of IDL do not have the socket command, and
;   must reply on a spawn to the popular Unix program "wget" to
;   open the socket.  Actually, for various reasons we're using
;   wget no matter what, but we hope to remove this dependance.
;
; PROCEDURE:
;   detect IDL version.  If IDL version < 5.3, then spawn wget 
;        otherwise use socket procedure.
;
; EXAMPLE:
;   ct_web_open, unit, 'http://www.yahoo.com', /get_lun
;   s=''
;   while not eof(unit) do begin   ; eof(unit) is not good practice
;      readf, unit, s
;      print, s
;   endwhile
;   free_lun, unit
;
; MODIFICATION HISTORY:
;   written, jbf, cottagesystems, Feb 12 2001.
;-
  
pro ct_web_open, unit, url, get_lun=get_lun, _extra=e, $
                 timeout=timeout, header=header

  if getenv( 'ct_web_open_verbose' ) ne '' then begin
      message, 'URL='+url, /cont
  endif

  mode= 'wget'  ; | 'socket' | 'wget_pipe'

  if float(!version.release) le 5.3 and mode eq 'socket' then begin
      message, 'socket not supported by IDL version 5.3 or lower.'
  endif
  
  case mode of
      'socket': begin
          message, 'socket version not fully developed.', /cont
          s= str_sep( url, '://' )
          if n_elements(s) eq 1 then begin
              protocol= 'http'
              host_file=s[0]
          endif else begin
              protocol= s[0]
              host_file= s[1]
          endelse
          ii= strpos( host_file, '/' )
          if ii ne -1 then begin
              host= strmid(host_file,0,ii)
              file= strmid(host_file,ii,strlen(host_file)-ii)
          endif else begin
              host= s[0]
              file= '/'
          endelse
          
          socket, unit, host, 80, _extra=e, get_lun=get_lun
          printf, unit, 'GET '+file
          
          s=''
          
          header= { content_length:long(-1), $
                    content_type:'' }
          
          if eof(unit) then begin
              message, 'eof on web open--ignoring--you try.', /cont
          endif else begin          
              readf, unit, s
              if strmid( s,0,9 ) eq '<!DOCTYPE' then begin
             ; ignore the header              
              endif else if strmid( s,0,4 ) eq 'HTTP' then begin
                  repeat begin
                      readf, unit, s
                      if strmid(s,0,15) eq 'Content-Length:' then $
                        header.content_length= long( strmid(s,15,65 )  )
                      if strmid(s,0,13) eq 'Content-Type:' then $
                        header.content_type= strtrim(strmid(s,14,65),2)
                  endrep until ( strtrim(s,2) eq '' )
              endif else begin                        ; uh-oh: simple response
                  if keyword_set(get_lun) then $      ;   no header.
                    free_lun, unit $
                  else $
                    close, unit
                  socket, unit, host, 80, _extra=e, get_lun=get_lun
                  printf, unit, 'GET '+file
              endelse
              
          endelse
      end
      'wget_pipe': begin
          if not keyword_set(get_lun) then begin
              message, 'using wget and /get_lun must be used with wget.'
          endif
          if n_elements(timeout) eq 0 then timeoutstr='15' else $
            timeoutstr=strtrim(timeout,2)
          spawn, "wget -T "+timeoutstr+" '"+url+"' -o /dev/null -O -", $
            unit=unit
      end
      'wget': begin
          f= findfile( '/tmp/ct_web_open.???.tmp~', count=count )
          temp_file= '/tmp/ct_web_open.'+string(count,format='(i3.3)')+'.tmp~'
          if n_elements(timeout) eq 0 then timeoutstr='15' else $
            timeoutstr=strtrim(timeout,2)
          spawn, "wget -T "+timeoutstr+" '"+url+"' -o /dev/null -O "+temp_file
          openr, unit, get_lun=get_lun, /delete, temp_file
      end
  endcase

end



