;  GifReadByte
;       Read a single byte out of the given file
;
FUNCTION GifReadByte, unit
  COMPILE_OPT hidden

  ch      = 0b
  READU, unit, ch
  RETURN, ch
END

;----------------------------------------------------------------------
;+
; NAME:
;       ct_read_gif
;
; PURPOSE:
;       Read GIF from an input stream and return image and colortables.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       ct_read_gif, unit, Image [, R, G, B]
;
; INPUTS:
;       unit:   logical unit opened with openr, socket, or spawn.
;
; Keyword Inputs:
;       /multiple   skip the header information and read the next
;                   image.  An error will occur if the end-of-file is
;                   encountered.  The first read must not have
;                   multiple set.
;
; OUTPUTS:
;       Image:  The 2D byte array to contain the image.
;
;
; OPTIONAL OUTPUT PARAMETERS:
;     R, G, B:  The variables to contain the Red, Green, and Blue color vectors
;               if the rasterfile containes colormaps.
;
; SIDE EFFECTS:
;       None.
;
; COMMON BLOCKS:
;
; RESTRICTIONS:
;       This routine only reads in the first image in a file (the format
;       allows many). Local colormaps are not supported.
;       Only 8 bit images are supported.
;
;       The Graphics Interchange Format(c) is the Copyright property
;       of CompuServ Incorporated.  GIF(sm) is a Service Mark property of
;       CompuServ Incorporated.
;
;       Uses read and write gif external codes that are no longer
;         distributed by RSI.
;
; EXAMPLE:
;       ct_web_open, unit, /get_lun, $
;           'http://www.google.com/images/title_homepage4.gif'
;       ct_read_gif, unit, image, r, g, b
;       close, unit
;       tvlct, r, g, b
;       s= size(image)
;       window, xsize=s[1], ysize=s[2]
;       tv, image
;
; MODIFICATION HISTORY:
;       Modified RSI's write_gif to work on open lun, jbf, Feb 23, 2001
;
;-
;

PRO ct_read_gif, unit, IMAGE, R, G, B, multiple=mult


  catch, err
  if err ne 0 then begin
      catch, /cancel
      message, !ERROR_STATE.MSG
      return
  endif

;; Main GIF Header declaration.
  header = { magic	: bytarr(6),            $
             width_lo  : 0b,                   $
             width_hi  : 0b,                   $
             height_lo : 0b, 			$
             height_hi : 0b,                   $
             screen_info : 0b, 		$
             background : 0b, 			$
             reserved  : 0b }

;; local image header declaration
  ihdr  = {   left_lo         : 0B,           $
              left_hi         : 0B,           $       
              top_lo          : 0B,           $
              top_hi          : 0B,           $
              iwidth_lo       : 0B,           $
              iwidth_hi       : 0B,           $
              iheight_lo      : 0B,           $       
              iheight_hi      : 0B,           $
              image_info      : 0b } ; its content

  if(keyword_set(mult))then $
    goto, next_image
  
  READU, unit, header           ;Read gif header

;; Check Magic in header: GIF87a or GIF89a.
  gif  = STRING(header.magic[0:2])
  vers = STRING(header.magic[3:5])

  if( gif NE 'GIF')then begin
      MESSAGE, 'Logical unit is not a GIF file.', /cont
      image=-1
      return
  endif
  

  if(vers ne '87a' and vers ne '89a')then $
    MESSAGE, /INFO, 'Unknown GIF Version: '+vers+'. Attempting to read...'

;; Get the virtual screen width and height

  scrWidth   = header.width_hi * 256 + header.width_lo
  scrHeight  = header.height_hi * 256 + header.height_lo

;; Find out how big the color map is

  bits_per_pixel  = (header.screen_info AND 'F'X) + 1
  color_map_size  = 2 ^ bits_per_pixel

;; Read in the colormap (optional)

  if((header.screen_info AND '80'X) NE 0 )then begin
      map     = BYTARR(3,color_map_size, /NOZERO)
      READU, unit, map
      map     = transpose(map)
      r       = map[*,0]
      g       = map[*,1]
      b       = map[*,2]
  endif

;; Read the image description

next_image:

  not_done=1
  while( not_done )do begin                ;; Read till we get a terminator
      cmd = GifReadByte(unit)        ;; Loop thru commands in file.

      case string(cmd) of

          '                     ;':    begin            ;; GIF trailer (0x3b)
              not_done=0
          END

          ',':    begin                  ;; Image description (0x2c)
              readu,unit,ihdr

              ;; Check for file formats we don't support
              ;; We don't support local colormaps

              if((ihdr.image_info AND '80'X) NE 0)then begin  ;;Local color map
                  message, 'local colormaps not supported', /cont
                  image= -1
                  return
              endif

              ;; Size of this image?
              iWidth   = ihdr.iwidth_hi  * 256 + ihdr.iwidth_lo
              iHeight  = ihdr.iheight_hi * 256 + ihdr.iheight_lo

              ;; Allocate an array to hold the image
              image   = BYTARR(iWidth, iHeight, /NOZERO)

              ;; Now call special GIF-LZW routine hidden within IDL
              ;; to do the ugly serial bit stream decoding

              DECODE_GIF,unit,image ; magic

              ;; This should be the 0 byte that ends the series:

              junk = GifReadByte(unit) ;Loop thru commands in file.

              if(junk ne 0)then $
                message,/info,'No trailing 0.'

              ;; Reorder rows in an interlaced image

              if((ihdr.image_info AND '40'X) NE 0 )then begin
                  l = lindgen(iHeight) ;Row indices...

                  ;;  Gif interlace ordering

                  p = [l[where(l mod 8 eq 0)], l[where(l mod 8 eq 4)], $
                       l[where(l mod 4 eq 2)], l[where(l and 1)]]

                  tmpImage = bytarr(iWidth, iHeight, /NOZERO)
                  l = iHeight-1
                  for i=0, l do $
                    tmpImage[0, l-p[i]] = image[*,l-i]
                  image = temporary(tmpImage)
              endif

              ;; Ok, is this image the same size as the screen size (main image)?

              if( iHeight ne scrHeight or iWidth ne scrWidth)then begin
                  tmpImage = replicate(header.background, scrWidth, scrHeight)
                  x0 = ihdr.left_hi * 256 + ihdr.left_lo
                  y0 = scrHeight - ((ihdr.top_hi * 256 + ihdr.top_lo) + iHeight)
                  tmpImage[x0:x0+iWidth-1, y0:y0+iHeight-1] = $
                    image[0:iWidth-1, 0:iHeight-1]
                  image = temporary(tmpImage) ;; Nuc the memory
              endif

              not_done=0
          end

          '!':    BEGIN         ;Gif Extention block (ignored) (0x21)
              label = GifReadByte(unit)       ;; toss extension block label
              repeat begin                    ;; read and ignore blkss
                  blk_size = GifReadByte(unit)  ;; block size
                  if(blk_size ne 0 )then begin
                      dummy= bytarr(blk_size)
                      readu, unit, dummy
                  endif
              endrep until(blk_size eq 0)
          end
          
          ELSE:   message,'Unknown GIF keyword in ' + $
            file + string(cmd, format='(2x,Z2)')
      ENDCASE
  endwhile  ; not_done

END
