;+
; NAME: sparc_get_registry
;
; PURPOSE: get the list of available data sets from the registry
;
; CATEGORY: sparc
;
; CALLING SEQUENCE: registry= sparc_get_registry(status=status)
;
; KEYWORD PARAMETERS:
;    status=   returns a status structure, { code:0, message:'' }
;    /verbose  print verbose messages while loading.
;    id_message=  specify id of widget where verbose messages 
;                 should be printed.
;
; OUTPUTS:
;    returns the registry in an array of structures.
;
; COMMON BLOCKS: none 
;
; RESTRICTIONS: 
;    uses ct_web_get(url), which has its own restrictions
; 
; PROCEDURE:
;
; EXAMPLE:
;    registry= sparc_get_registry(status=status) 
;    help, /str, registry
;    print, registry.name
;
; MODIFICATION HISTORY:
;    written, Feb 24, 2001.  jbf, cottagesystems.
;-

function sparc_get_registry, status=status, $
                             verbose=verbose, $
                             id_message= id_message

  status= sparc_get_status_struct()

  verbose= keyword_set( verbose )
  if n_elements( id_message ) eq 0 then id_message=-1

  msg= 'Getting data center list from registry...'
  if verbose then message, msg, /cont
  if id_message gt 0 then widget_control, id_message, set_value=msg

  dataCenters= sparc_get_datacenters(status=status)
  
  if status.code ne 0 then return, -1
  
  msg= 'Got '+strtrim(n_elements(dataCenters),2)+' data centers.'
  if verbose then message, msg, /cont
  if id_message gt 0 then widget_control, id_message, set_value=msg
  
  registry= replicate( { registry_struct }, 1000 )
  ni=1000
  i=0

  for idc=0,n_elements(dataCenters)-1 do begin

      i0= i-1

      msg= 'Getting directory from ' + dataCenters[idc] + '...'
      if verbose then message, msg, /cont
      if id_message gt 0 then widget_control, id_message, set_value=msg

      url= 'http://' + dataCenters[idc] + $
        '?name=directory&view=name&option=list'
      
      ct_web_open, unit, /get_lun, url

      aline=''
      while not eof( unit ) do begin
          
          readf, unit, aline
          
          s= str_sep( aline, '|' )
          registry[i].name= strtrim(s[0],2)
          registry[i].descriptor= strtrim(s[1],2)
          registry[i].type= strtrim(s[2],2)
          registry[i].dataCenter= dataCenters[idc]
          i=i+1

          if i gt ni then begin
              registry= [ registry, replicate( registry_element, 100 ) ]
              ni= ni+100
          endif

          if (i-i0) mod 40 eq 0 then begin ; progress feedback
              if id_message gt 0 then begin
                  sthink=(['|','/','-','\'])[((i-i0)/40 mod 4)]
                  widget_control, id_message, $
                    set_value= msg + sthink
              endif
          endif
          
      endwhile
      free_lun, unit

  endfor

  registry= registry[0:(i-1)]
  
  status.code=0
  status.message=''

  if id_message gt 0 then begin
      widget_control, id_message, set_value= ' '
  endif

  
  return, registry

end

