;+
; NAME: sparc_get_view
;
; PURPOSE: get image of view for particular name
;
; CATEGORY: sparc
;
; CALLING SEQUENCE: 
;   image= sparc_get_view( datacenter, name, view, ctstart, ctend )
;
; INPUTS:
;   parameter     example   definition
;   datacenter    'http://sparc-dc9.si.umich.edu/sparc/DataCenter'
;                           datacenter serving the data
;   name          'pixie'   name of the data set
;   view          'timelinegif'  view type for image
;   ctstart       10608.0   start time of interval in ct_time.
;   ctend         10610.0   end time of interval in ct_time.
;
; OPTIONAL INPUTS:
;
; KEYWORD PARAMETERS:
;   size        [ 380,100]  request size of the image, in pixels
;   /include_timeaxis       put the timeaxis as part of the image.
;   /title                  return title image
;   /yaxis                  return yaxis image
;   /colorbar               return colorbar image
;   /timeaxis               return timeaxis image (ignores datacenter, name)
;     title, yaxis and colorbar are ignored if /include_axis is set.
;   options=   ''           string to insert into URL for options.
;
; ENVIRONMENT PARAMETERS:
;   SPARC_TRANSPARENT       try to detect the transparent color.  The 
;                             transparent color is the most frequently
;                             occuring color.
; OUTPUTS:
;   returns                 image bytarr(4,n,m)  R,G,B,alpha (opacity).
;
; RESTRICTIONS:
;
; EXAMPLE:
;   image= sparc_get_view_image( $
;      'sparc-dc9.si.umich.edu/sparc/DataCenter', $
;      'pixie', 'timelinegif', $
;      10608.0, 10610.0 )
;
; MODIFICATION HISTORY:
;   written, feb 23, 2001, jbf, cottagesystems
;-

pro testit
  COMPILE_OPT idl2, HIDDEN
  image= sparc_get_view_image( $
                               'sparc-dc2.si.umich.edu/sparc/DataCenter', $
                               'Jicamarca.ISRadar', $
                               'timeseries', $
                               9954.0, 9957.0, $
                               size=[640,200] )
  tv, image, /true
end

function sparc_get_view_image, dataCenter, name, view, $
                               ctstart, ctend, $
                               size=size, status=status, $
                               include_timeaxis=include_timeaxis, $
                               yaxis=yaxis, title=title, $
                               timeaxis=timeaxis, $
                               colorbar=colorbar, $
                               options=options

  COMPILE_OPT idl2

  status= sparc_get_status_struct()

  if n_elements(size) eq 0 then ssize='' else $
    ssize='&size='+strtrim(size[0],2)+','+strtrim(size[1],2)

  if keyword_set(include_timeaxis) then begin
      spart= '' 
  endif else begin
      case 1 of 
          keyword_set(title): spart='t'
          keyword_set(yaxis): spart='y'
          keyword_set(colorbar): spart='b'
          keyword_set(timeaxis): spart='m'
          else: spart= 'g'
      endcase
      spart= '&part='+spart
  endelse

  soptions=''
  if n_elements(options) ne 0 then begin
      if strtrim(options,2) ne '' then begin
          soptions= '&'+options
      endif
  endif

  url= 'http://'+dataCenter+$
    '?name='+name+$
    '&view='+view+$
    ssize+$
    spart+$
    soptions+$
    '&time='+ct_time_to_sparc(ctstart)+$
    '-'+ct_time_to_sparc(ctend)

  ct_web_open, unit, url, /get_lun

  if eof(unit) then begin
      status.code= -2
      status.message= 'Unexpected eof on web read.'
      status.action= 'Possible time-out.  Check URL: '+url
      free_lun, unit
      return, -1
  endif

  ct_read_gif, unit, image, red, green, blue
  free_lun, unit

  if image[0] eq -1 then begin
      status.code=-1
      status.message='Error on gif read.'
      image3=-1
  endif else begin
      s= size( image )
      image4= bytarr( 4, s[1], s[2] )
      
      opacity= replicate(255B,s[1],s[2])
      if getenv( 'SPARC_TRANSPARENT' ) eq '1' then begin
          corner_color= image[[0,s[1]-1,s[1]-1,0],$
                              [0,0,s[2]-1,s[2]-1] ]
          h= histogram( corner_color )
          ncorner_color_max= max( h, icorner_color_max )
          if ncorner_color_max ge n_elements(corner_color)*0.75 then begin
              itransparent= corner_color[icorner_color_max]
          endif else begin
              h= histogram( image )
              x= max( h, imax )
              itransparent=  imax
          endelse
          r= where( image eq itransparent )              
          if r[0] ne -1 then opacity[r]=0
      endif

      image4[0,*,*]= red[image]
      image4[1,*,*]= green[image]
      image4[2,*,*]= blue[image]
      image4[3,*,*]= opacity
      
  endelse
  
  status.code=0
  status.message=''

  return, image4

end

