;******************************************************************************
;* file papcoadd_sparc_image
;*      papcoadd_sparc_image
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a sparc. 
;*
;* THIS IS A WORKING EXAMPLE MODULE WHICH CAN BE INCLUDED INTO papco AS IS
;* (it doesn't plot very exiting data though!)
;*
;* THIS TEMPLATE IS INTENDED AS THE STARTING POINT FOR DEVELOPING OWN MODULES.
;* COPY THE WHOLE sparc DIRECTORY TREE AND RENAME EVERY OCCURRENCE OF 
;* "sparc" WITH YOUR MODULE NAME, IN ALL THE FILES AND FILENAMES.
;*
;* If the file starting with "papcoadd" exist in the module directory, then the
;* routines contained thereinare utilized to add that plot type into PAPCO.
;* This makes papco "configurable" by adding or leaving out modules.
;*
;* ALL that is needed to include the module is to include the path to the
;* module directory tree in the IDL search path. This is done automatically 
;* by the PAPCO configure widget when you start PAPCO with
;* 
;* idl papco_setup
;*
;* If the plot type is in itself a sub-menu containing more sparcs, then
;* these must be listed at the top of this file, as it is read by papco.
;* the entries for each sub-plot-type must come after the line 
;* "file papcoadd_sparc" and before the "end" line, one entry per line.
;* Some of the routines in this file must then be duplicated for each 
;* sub-plot-type used: see some of the exisiting modules for an example.
;*
;* Apart from editing this template the user has to write a panel 
;* editor widget for his sparc. Use the file 
;*    $papco_XX/papco_modules/sparc/papco_interface/sparc_image_panel.pro
;* as the template.
;*
;* If slice functions are to be supported a slice program needs to be written
;* too. Use the template file
;*    $papco_XX/papco_modules/sparc/papco_interface/sparc_image_slice.pro.
;*
;* Finally, a data read and data plot routine are required as well. The data
;* read routine will be specific for each sparc. An example for a standard
;* line plot is given in the file 
;*    $papco_XX/papco_modules/sparc/plot_read/p_data_type_line.pro
;* and for a standard color spectrogram in 
;*    $papco_XX/papco_modules/sparc/plot_read/p_data_type_spec.pro. 
;* These templates can also be used as a basis for more complex plots.
;*
;*
;* The Way these add-in routines work:
;* ===================================
;*
;* On initializing IDL for PAPCO, the routine find_modules.pro is used to
;* parse the IDL search path, and all the init_module.pro batch files found in 
;* the module sub-directories \papco_interface are written to the 
;* papco_add_init.pro batch file and then executed: In this way the user can 
;* specify all the routines that need to be compiled for his module at startup.
;* 
;* On starting papco, all module sub-directories \papco_interface are searched
;* for a file beginning with "papcoadd_". From the names of this file the 
;* sparc name is stripped off and used to construct all the procedure 
;* names contained in the file. The first routine, "addplottype_sparc", is
;* used to add the button for this sparc to the papco main window. Since
;* "addplottype_sparc" simply returns widget ID of the button, the button
;* itself can contain further sub-menus for plot sub-types.
;*
;* To inform papco of all these sub-plot types (and to avoid having to write a 
;* separate file for each sub-plot type) all the types have to be listed in the
;* header of the file papcoadd_sparc.pro. These entries have to finish off 
;* with the line ";* end". All sparc names are stored in the string array
;* sparcs and are placed in a common block:
;*
;*    COMMON ADDINS, sparcs
;*
;* If there are no sub-plot types, then the sparc as used in the file name
;* for papcoadd_sparc.pro is used to construct the calls to the procedures
;* in that file. The convention for naming these procedures is
;* "procedure_name_sparc" , so that papco can always construct the right
;* procedure call when needed. 
;*
;* When there are sub-plot types, their names get used to construct the
;* required calls. There are 6 procedures that need to be called for each
;* sparc, so that a set of these 6 procedures has to be written for each
;* sub-plot type. In the same way, you need to write a panel editor for each
;* plot type (sometimes several plot sub-types may share the same editor,
;* e.g. multiple line plots or color spectra, see papcoadd_crres_plots).
;* 
;* In PAPCO, whenever an action for a sparc is taken, the information
;* stored in the structure widgetData is queried. In this structure there
;* is an array of PlotInfo structures, which contain all the relevant 
;* information for a chosen panel. The field plotInfo.panelkind is loaded
;* with the name of the sparc. This is set by the user in this file in
;* the routine addpanel_sparc ( "panelKind='sparc'" ). THIS MUST
;* BE SET CORRECTLY AS ALL THE ROUTINE NAMES USED IN THIS FILE ARE
;* CONSTRUCTED USING THIS INFORMATION!
;*
;* Whenever papco has to undertake some action for a sparc, the field 
;* plotInfo.panelkind is compared to the entries in the (COMMON ADDINS) 
;* sparcs array. If a match is found, the required procedure name is
;* constructed and called If not, an error occurs.
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_sparc_image,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_sparc_image, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO canget_sparc_image, canget
;*        -> Sets a variable to indicate output to file is supported.
;*
;*    PRO descriptor_sparc_image, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_sparc_image
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_sparc_image, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           papco_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_sparc_image, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     updated for OUTPUT functionality, June 1998, Reiner Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_sparc_image,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this sparc_image to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      sparc_images.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_sparc_image,plottypes_base,buttonGeometry,pb_plottype
;******************************************************************************
pro addplottype_sparc,plottypes_base,buttonGeometry,pdm_sparc
  
; This example is for a simple button with a submenu. 
; **NEW** from PAPCO 09 is the inclusion of the configure button as
; the first entry of the module submenu. Further entries into the
; submenu are easily added using
;
;        { CW_PDMENU_S, 0, ' sparc submenu ' }, $
;  
; The second entry in this structure defines the following:
;                       3   -   label for the button
;                       0   -   submenu entry
;                       2   -   last submenu entry  
;
; The third entry in this structure is the name of the submenu entry.

  MenuDesc = [ $
               { CW_PDMENU_S, 3, strcenter('sparc',10) }, $ 
               { CW_PDMENU_S, 0, 'configure sparc' }, $
               { CW_PDMENU_S, 2, 'sparc image' } $
             ]  
  
  pdm_sparc = CW_PDMENU(plottypes_base, MenuDesc, $
                              UVALUE='pdm_sparc')
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_sparc
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the sparc.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_sparc
;***************************************************************************
pro addpanel_sparc, event

  common get_error, get_err_no, get_err_msg
  
; The case statement here needs an entry for every submenu type you
; added in  the first procedure in this file (addplottype_sparc_image).
; For each submenu entry, you will need to specify the panelEditor to use.  
  
; The FIRST time you use this module you will have to configure it,
; i.e. set the data paths etc. Once you have done this PAPCO will
; write a sparc_image.config file into your papco_user_XX/papco_modules_defaults
; directory. 
; So here we check if this sparc_image.config file exists, and if not,
; we disallow the execution of any other option apart from "configure
; sparc_image". In this way the user is forced to configure the module
; before using it the first time!
  
  panelKind='sparc'

  r_module_config, panelKind, config_data, msg
  if (get_err_no ne 0) and (event.value ne 1) then begin
    message,'You need to configure the module first', /cont
    papco_module_configure,panelKind
    return
  endif  

  CASE event.value of
      1: BEGIN
          papco_module_configure, panelKind
      end
      2: BEGIN
          panelKind='sparc_image'  
          panelEditor=panelkind+'_panel'
          defaults=papco_getPlotInfoStruct() 
          defaults.ioptions[2]=1 ; list by description
          defaults.soptions[2]='default' ; default view
          defaults.soptions[6]='default Graph Type' ; default view
          papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
      end

      else: begin
          message, 'Nothing defined for this submenu entry',  /cont
          return
      end  
  endcase
      
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_sparc_image, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable with / without orbit.
;*      PAPCO provides "set time by orbit" feature for CRRES and POLAR
;*      so far. Included for compatibility with older CRRES modules
;*      which ONLY use orbit as input to their read/plot routines
;*      (plotting over orbit boundaries is not enabled for CRRES)
;* 
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "plottable_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_sparc_image.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by orbit
;*                                 1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;******************************************************************************
pro plottable_sparc_image, plottable

  plottable=1     ; 1 -> plotable without orbit info, by time 
                  ; 0 -> plotable only by orbit number not by time 
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_sparc_image, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that sparc_image support the OUTPUT
;*      functionality. 
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_sparc_image.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                               1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_sparc_image, canget
;******************************************************************************
pro canget_sparc_image, canget

  canget=  1      ; 1 -> plot routine can return just data to file
                  ; 0 -> feature disabled
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_sparc_image, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this sparc_image. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "descriptor_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_sparc_image.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_sparc_image, plotInfo, descriptStr
;******************************************************************************
pro descriptor_sparc_image, plotInfo, descriptStr

  if plotInfo.soptions[3] eq '' or $
    plotInfo.ioptions[2] eq 0 then begin
      descriptStr= 'N:'+plotInfo.soptions[1] + ' DC: '+plotInfo.soptions[0]
  endif else begin
      descriptStr= plotInfo.soptions[3]
  endelse

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_sparc_image, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      papco_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several sparc_images to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for sparc_image.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "editor_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_sparc_image.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_sparc_image, plotNumber, newPlotData
;******************************************************************************
PRO editor_sparc_image, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData
  
  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=sparc_image_panel(plotInfo, $
                                ACTION='Update '+plotInfo.panelKind, $
                                GROUP=widgetData.main_base)
END




